"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.registerContextProvider = exports.provideContextValues = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const cloudformation_deployments_1 = require("../api/cloudformation-deployments");
const logging_1 = require("../logging");
const settings_1 = require("../settings");
const ami_1 = require("./ami");
const availability_zones_1 = require("./availability-zones");
const endpoint_service_availability_zones_1 = require("./endpoint-service-availability-zones");
const hosted_zones_1 = require("./hosted-zones");
const load_balancers_1 = require("./load-balancers");
const security_groups_1 = require("./security-groups");
const ssm_parameters_1 = require("./ssm-parameters");
const vpcs_1 = require("./vpcs");
/**
 * Iterate over the list of missing context values and invoke the appropriate providers from the map to retrieve them
 */
async function provideContextValues(missingValues, context, sdk) {
    for (const missingContext of missingValues) {
        const key = missingContext.key;
        const constructor = availableContextProviders[missingContext.provider];
        if (!constructor) {
            // eslint-disable-next-line max-len
            throw new Error(`Unrecognized context provider name: ${missingContext.provider}. You might need to update the toolkit to match the version of the construct library.`);
        }
        const provider = new constructor(sdk);
        let value;
        try {
            const environment = cxapi.EnvironmentUtils.make(missingContext.props.account, missingContext.props.region);
            const resolvedEnvironment = await sdk.resolveEnvironment(environment);
            const arns = await cloudformation_deployments_1.replaceEnvPlaceholders({
                lookupRoleArn: missingContext.props.lookupRoleArn,
            }, resolvedEnvironment, sdk);
            value = await provider.getValue({ ...missingContext.props, lookupRoleArn: arns.lookupRoleArn });
        }
        catch (e) {
            // Set a specially formatted provider value which will be interpreted
            // as a lookup failure in the toolkit.
            value = { [cxapi.PROVIDER_ERROR_KEY]: e.message, [settings_1.TRANSIENT_CONTEXT_KEY]: true };
        }
        context.set(key, value);
        logging_1.debug(`Setting "${key}" context to ${JSON.stringify(value)}`);
    }
}
exports.provideContextValues = provideContextValues;
/**
 * Register a context provider
 *
 * (Only available for testing right now).
 */
function registerContextProvider(name, provider) {
    availableContextProviders[name] = provider;
}
exports.registerContextProvider = registerContextProvider;
const availableContextProviders = {
    [cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER]: availability_zones_1.AZContextProviderPlugin,
    [cxschema.ContextProvider.SSM_PARAMETER_PROVIDER]: ssm_parameters_1.SSMContextProviderPlugin,
    [cxschema.ContextProvider.HOSTED_ZONE_PROVIDER]: hosted_zones_1.HostedZoneContextProviderPlugin,
    [cxschema.ContextProvider.VPC_PROVIDER]: vpcs_1.VpcNetworkContextProviderPlugin,
    [cxschema.ContextProvider.AMI_PROVIDER]: ami_1.AmiContextProviderPlugin,
    [cxschema.ContextProvider.ENDPOINT_SERVICE_AVAILABILITY_ZONE_PROVIDER]: endpoint_service_availability_zones_1.EndpointServiceAZContextProviderPlugin,
    [cxschema.ContextProvider.SECURITY_GROUP_PROVIDER]: security_groups_1.SecurityGroupContextProviderPlugin,
    [cxschema.ContextProvider.LOAD_BALANCER_PROVIDER]: load_balancers_1.LoadBalancerContextProviderPlugin,
    [cxschema.ContextProvider.LOAD_BALANCER_LISTENER_PROVIDER]: load_balancers_1.LoadBalancerListenerContextProviderPlugin,
};
//# sourceMappingURL=data:application/json;base64,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