"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SSMContextProviderPlugin = void 0;
const cxapi = require("@aws-cdk/cx-api");
const api_1 = require("../api");
const logging_1 = require("../logging");
/**
 * Plugin to read arbitrary SSM parameter names
 */
class SSMContextProviderPlugin {
    constructor(aws) {
        this.aws = aws;
    }
    async getValue(args) {
        const region = args.region;
        const account = args.account;
        if (!('parameterName' in args)) {
            throw new Error('parameterName must be provided in props for SSMContextProviderPlugin');
        }
        const parameterName = args.parameterName;
        logging_1.debug(`Reading SSM parameter ${account}:${region}:${parameterName}`);
        const response = await this.getSsmParameterValue(account, region, parameterName, args.lookupRoleArn);
        if (!response.Parameter || response.Parameter.Value === undefined) {
            throw new Error(`SSM parameter not available in account ${account}, region ${region}: ${parameterName}`);
        }
        return response.Parameter.Value;
    }
    /**
     * Gets the value of an SSM Parameter, while not throwin if the parameter does not exist.
     * @param account       the account in which the SSM Parameter is expected to be.
     * @param region        the region in which the SSM Parameter is expected to be.
     * @param parameterName the name of the SSM Parameter
     * @param lookupRoleArn the ARN of the lookup role.
     *
     * @returns the result of the ``GetParameter`` operation.
     *
     * @throws Error if a service error (other than ``ParameterNotFound``) occurs.
     */
    async getSsmParameterValue(account, region, parameterName, lookupRoleArn) {
        const options = { assumeRoleArn: lookupRoleArn };
        const ssm = (await this.aws.forEnvironment(cxapi.EnvironmentUtils.make(account, region), api_1.Mode.ForReading, options)).ssm();
        try {
            return await ssm.getParameter({ Name: parameterName }).promise();
        }
        catch (e) {
            if (e.code === 'ParameterNotFound') {
                return {};
            }
            throw e;
        }
    }
}
exports.SSMContextProviderPlugin = SSMContextProviderPlugin;
//# sourceMappingURL=data:application/json;base64,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