"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cloud_assembly_1 = require("../../lib/api/cxapp/cloud-assembly");
const util_1 = require("../util");
// behave like v2
process.env.CXAPI_DISABLE_SELECT_BY_ID = '1';
test('do not throw when selecting stack without errors', async () => {
    // GIVEN
    const cxasm = await testCloudAssembly();
    // WHEN
    const selected = await cxasm.selectStacks({ patterns: ['withouterrorsNODEPATH'] }, {
        defaultBehavior: cloud_assembly_1.DefaultSelection.AllStacks,
    });
    selected.processMetadataMessages();
    // THEN
    expect(selected.firstStack.template.resource).toBe('noerrorresource');
});
test('do throw when selecting stack with errors', async () => {
    // GIVEN
    const cxasm = await testCloudAssembly();
    // WHEN
    const selected = await cxasm.selectStacks({ patterns: ['witherrors'] }, {
        defaultBehavior: cloud_assembly_1.DefaultSelection.AllStacks,
    });
    // THEN
    expect(() => selected.processMetadataMessages()).toThrow(/Found errors/);
});
test('select all top level stacks in the presence of nested assemblies', async () => {
    // GIVEN
    const cxasm = await testNestedCloudAssembly();
    // WHEN
    const x = await cxasm.selectStacks({ allTopLevel: true, patterns: [] }, { defaultBehavior: cloud_assembly_1.DefaultSelection.AllStacks });
    // THEN
    expect(x.stackCount).toBe(2);
    expect(x.stackIds).toContain('witherrors');
    expect(x.stackIds).toContain('withouterrors');
});
test('select stacks by glob pattern', async () => {
    // GIVEN
    const cxasm = await testCloudAssembly();
    // WHEN
    const x = await cxasm.selectStacks({ patterns: ['with*'] }, { defaultBehavior: cloud_assembly_1.DefaultSelection.AllStacks });
    // THEN
    expect(x.stackCount).toBe(2);
    expect(x.stackIds).toContain('witherrors');
    expect(x.stackIds).toContain('withouterrors');
});
test('select behavior: all', async () => {
    // GIVEN
    const cxasm = await testCloudAssembly();
    // WHEN
    const x = await cxasm.selectStacks({ patterns: [] }, { defaultBehavior: cloud_assembly_1.DefaultSelection.AllStacks });
    // THEN
    expect(x.stackCount).toBe(2);
});
test('select behavior: none', async () => {
    // GIVEN
    const cxasm = await testCloudAssembly();
    // WHEN
    const x = await cxasm.selectStacks({ patterns: [] }, { defaultBehavior: cloud_assembly_1.DefaultSelection.None });
    // THEN
    expect(x.stackCount).toBe(0);
});
test('select behavior: single', async () => {
    // GIVEN
    const cxasm = await testCloudAssembly();
    // WHEN
    await expect(cxasm.selectStacks({ patterns: [] }, { defaultBehavior: cloud_assembly_1.DefaultSelection.OnlySingle }))
        .rejects.toThrow('Since this app includes more than a single stack, specify which stacks to use (wildcards are supported) or specify `--all`');
});
test('stack list error contains node paths', async () => {
    // GIVEN
    const cxasm = await testCloudAssembly();
    // WHEN
    await expect(cxasm.selectStacks({ patterns: [] }, { defaultBehavior: cloud_assembly_1.DefaultSelection.OnlySingle }))
        .rejects.toThrow('withouterrorsNODEPATH');
});
test('select behavior: repeat', async () => {
    // GIVEN
    const cxasm = await testCloudAssembly();
    // WHEN
    const x = await cxasm.selectStacks({ patterns: ['withouterrorsNODEPATH', 'withouterrorsNODEPATH'] }, {
        defaultBehavior: cloud_assembly_1.DefaultSelection.AllStacks,
    });
    // THEN
    expect(x.stackCount).toBe(1);
});
test('select behavior with nested assemblies: all', async () => {
    // GIVEN
    const cxasm = await testNestedCloudAssembly();
    // WHEN
    const x = await cxasm.selectStacks({ patterns: [] }, { defaultBehavior: cloud_assembly_1.DefaultSelection.AllStacks });
    // THEN
    expect(x.stackCount).toBe(3);
});
test('select behavior with nested assemblies: none', async () => {
    // GIVEN
    const cxasm = await testNestedCloudAssembly();
    // WHEN
    const x = await cxasm.selectStacks({ patterns: [] }, { defaultBehavior: cloud_assembly_1.DefaultSelection.None });
    // THEN
    expect(x.stackCount).toBe(0);
});
test('select behavior with nested assemblies: single', async () => {
    // GIVEN
    const cxasm = await testNestedCloudAssembly();
    // WHEN
    await expect(cxasm.selectStacks({ patterns: [] }, { defaultBehavior: cloud_assembly_1.DefaultSelection.OnlySingle }))
        .rejects.toThrow('Since this app includes more than a single stack, specify which stacks to use (wildcards are supported) or specify `--all`');
});
test('select behavior with nested assemblies: repeat', async () => {
    // GIVEN
    const cxasm = await testNestedCloudAssembly();
    // WHEN
    const x = await cxasm.selectStacks({ patterns: ['deeply/hidden/withouterrors', 'nested'] }, {
        defaultBehavior: cloud_assembly_1.DefaultSelection.AllStacks,
    });
    // THEN
    expect(x.stackCount).toBe(2);
});
async function testCloudAssembly({ env } = {}) {
    const cloudExec = new util_1.MockCloudExecutable({
        stacks: [{
                stackName: 'withouterrors',
                displayName: 'withouterrorsNODEPATH',
                env,
                template: { resource: 'noerrorresource' },
            },
            {
                stackName: 'witherrors',
                env,
                template: { resource: 'errorresource' },
                metadata: {
                    '/resource': [
                        {
                            type: cxschema.ArtifactMetadataEntryType.ERROR,
                            data: 'this is an error',
                        },
                    ],
                },
            }],
    });
    return cloudExec.synthesize();
}
async function testNestedCloudAssembly({ env } = {}) {
    const cloudExec = new util_1.MockCloudExecutable({
        stacks: [{
                stackName: 'withouterrors',
                env,
                template: { resource: 'noerrorresource' },
                // The nesting in the path should be independent of the position in the tree
                displayName: 'deeply/hidden/withouterrors',
            },
            {
                stackName: 'witherrors',
                env,
                template: { resource: 'errorresource' },
                metadata: {
                    '/resource': [
                        {
                            type: cxschema.ArtifactMetadataEntryType.ERROR,
                            data: 'this is an error',
                        },
                    ],
                },
            }],
        nestedAssemblies: [{
                stacks: [{
                        stackName: 'nested',
                        env,
                        template: { resource: 'nestederror' },
                        metadata: {
                            '/resource': [
                                {
                                    type: cxschema.ArtifactMetadataEntryType.ERROR,
                                    data: 'this is another error',
                                },
                            ],
                        },
                    }],
            }],
    });
    return cloudExec.synthesize();
}
//# sourceMappingURL=data:application/json;base64,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