"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const cloud_assembly_1 = require("../../lib/api/cxapp/cloud-assembly");
const context_providers_1 = require("../../lib/context-providers");
const util_1 = require("../util");
describe('AWS::CDK::Metadata', () => {
    test('is generated for relocatable stacks', async () => {
        var _a;
        const cx = await testCloudExecutable({ env: `aws://${cxapi.UNKNOWN_ACCOUNT}/${cxapi.UNKNOWN_REGION}`, versionReporting: true });
        const cxasm = await cx.synthesize();
        const result = cxasm.stackById('withouterrors').firstStack;
        const metadata = result.template.Resources && result.template.Resources.CDKMetadata;
        expect(metadata).toEqual({
            Type: 'AWS::CDK::Metadata',
            Properties: {
                // eslint-disable-next-line @typescript-eslint/no-require-imports
                Modules: `${require('../../package.json').name}=${require('../../package.json').version}`,
            },
            Condition: 'CDKMetadataAvailable',
        });
        expect((_a = result.template.Conditions) === null || _a === void 0 ? void 0 : _a.CDKMetadataAvailable).toBeDefined();
    });
    test('is generated for stacks in supported regions', async () => {
        const cx = await testCloudExecutable({ env: 'aws://012345678912/us-east-1', versionReporting: true });
        const cxasm = await cx.synthesize();
        const result = cxasm.stackById('withouterrors').firstStack;
        const metadata = result.template.Resources && result.template.Resources.CDKMetadata;
        expect(metadata).toEqual({
            Type: 'AWS::CDK::Metadata',
            Properties: {
                // eslint-disable-next-line @typescript-eslint/no-require-imports
                Modules: `${require('../../package.json').name}=${require('../../package.json').version}`,
            },
        });
    });
    test('is not generated for stacks in unsupported regions', async () => {
        const cx = await testCloudExecutable({ env: 'aws://012345678912/bermuda-triangle-1337', versionReporting: true });
        const cxasm = await cx.synthesize();
        const result = cxasm.stackById('withouterrors').firstStack;
        const metadata = result.template.Resources && result.template.Resources.CDKMetadata;
        expect(metadata).toBeUndefined();
    });
});
test('stop executing if context providers are not making progress', async () => {
    context_providers_1.registerContextProvider(cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER, class {
        async getValue(_) {
            return 'foo';
        }
    });
    const cloudExecutable = new util_1.MockCloudExecutable({
        stacks: [{
                stackName: 'thestack',
                template: { resource: 'noerrorresource' },
            }],
        // Always return the same missing keys, synthesis should still finish.
        missing: [
            { key: 'abcdef', props: { account: '1324', region: 'us-east-1' }, provider: cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER },
        ],
    });
    const cxasm = await cloudExecutable.synthesize();
    // WHEN
    await cxasm.selectStacks({ patterns: ['thestack'] }, { defaultBehavior: cloud_assembly_1.DefaultSelection.AllStacks });
    // THEN: the test finishes normally});
});
test('fails if lookups are disabled and missing context is synthesized', async () => {
    // GIVEN
    const cloudExecutable = new util_1.MockCloudExecutable({
        stacks: [{
                stackName: 'thestack',
                template: { resource: 'noerrorresource' },
            }],
        // Always return the same missing keys, synthesis should still finish.
        missing: [
            { key: 'abcdef', props: { account: '1324', region: 'us-east-1' }, provider: cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER },
        ],
    });
    cloudExecutable.configuration.settings.set(['lookups'], false);
    // WHEN
    await expect(cloudExecutable.synthesize()).rejects.toThrow(/Context lookups have been disabled/);
});
async function testCloudExecutable({ env, versionReporting = true } = {}) {
    const cloudExec = new util_1.MockCloudExecutable({
        stacks: [{
                stackName: 'withouterrors',
                env,
                template: { resource: 'noerrorresource' },
            },
            {
                stackName: 'witherrors',
                env,
                template: { resource: 'errorresource' },
                metadata: {
                    '/resource': [
                        {
                            type: cxschema.ArtifactMetadataEntryType.ERROR,
                            data: 'this is an error',
                        },
                    ],
                },
            }],
    });
    cloudExec.configuration.settings.set(['versionReporting'], versionReporting);
    return cloudExec;
}
//# sourceMappingURL=data:application/json;base64,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