"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const api_1 = require("../../lib/api");
const util_1 = require("../util");
const mock_sdk_1 = require("../util/mock-sdk");
const FAKE_STACK = util_1.testStack({
    stackName: 'withouterrors',
});
const FAKE_STACK_WITH_PARAMETERS = util_1.testStack({
    stackName: 'withparameters',
    template: {
        Parameters: {
            HasValue: { Type: 'String' },
            HasDefault: { Type: 'String', Default: 'TheDefault' },
            OtherParameter: { Type: 'String' },
        },
    },
});
const FAKE_STACK_TERMINATION_PROTECTION = util_1.testStack({
    stackName: 'termination-protection',
    template: util_1.DEFAULT_FAKE_TEMPLATE,
    terminationProtection: true,
});
let sdk;
let sdkProvider;
let cfnMocks;
beforeEach(() => {
    sdkProvider = new mock_sdk_1.MockSdkProvider();
    sdk = new mock_sdk_1.MockSdk();
    cfnMocks = {
        describeStackEvents: jest.fn().mockReturnValue({}),
        describeStacks: jest.fn()
            // First call, no stacks exist
            .mockImplementationOnce(() => ({ Stacks: [] }))
            // Second call, stack has been created
            .mockImplementationOnce(() => ({
            Stacks: [
                {
                    StackStatus: 'CREATE_COMPLETE',
                    StackStatusReason: 'It is magic',
                    EnableTerminationProtection: false,
                },
            ],
        })),
        createChangeSet: jest.fn((_o) => ({})),
        deleteChangeSet: jest.fn((_o) => ({})),
        describeChangeSet: jest.fn((_o) => ({
            Status: 'CREATE_COMPLETE',
            Changes: [],
        })),
        executeChangeSet: jest.fn((_o) => ({})),
        deleteStack: jest.fn((_o) => ({})),
        getTemplate: jest.fn((_o) => ({ TemplateBody: JSON.stringify(util_1.DEFAULT_FAKE_TEMPLATE) })),
        updateTerminationProtection: jest.fn((_o) => ({ StackId: 'stack-id' })),
    };
    sdk.stubCloudFormation(cfnMocks);
});
function standardDeployStackArguments() {
    return {
        stack: FAKE_STACK,
        sdk,
        sdkProvider,
        resolvedEnvironment: mock_sdk_1.mockResolvedEnvironment(),
        toolkitInfo: api_1.ToolkitInfo.bootstraplessDeploymentsOnly(sdk),
    };
}
test('do deploy executable change set with 0 changes', async () => {
    // WHEN
    const ret = await api_1.deployStack({
        ...standardDeployStackArguments(),
    });
    // THEN
    expect(ret.noOp).toBeFalsy();
    expect(cfnMocks.executeChangeSet).toHaveBeenCalled();
});
test('correctly passes CFN parameters, ignoring ones with empty values', async () => {
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        parameters: {
            A: 'A-value',
            B: 'B=value',
            C: undefined,
            D: '',
        },
    });
    // THEN
    expect(cfnMocks.createChangeSet).toHaveBeenCalledWith(expect.objectContaining({
        Parameters: [
            { ParameterKey: 'A', ParameterValue: 'A-value' },
            { ParameterKey: 'B', ParameterValue: 'B=value' },
        ],
    }));
});
test('reuse previous parameters if requested', async () => {
    // GIVEN
    givenStackExists({
        Parameters: [
            { ParameterKey: 'HasValue', ParameterValue: 'TheValue' },
            { ParameterKey: 'HasDefault', ParameterValue: 'TheOldValue' },
        ],
    });
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        stack: FAKE_STACK_WITH_PARAMETERS,
        parameters: {
            OtherParameter: 'SomeValue',
        },
        usePreviousParameters: true,
    });
    // THEN
    expect(cfnMocks.createChangeSet).toHaveBeenCalledWith(expect.objectContaining({
        Parameters: [
            { ParameterKey: 'HasValue', UsePreviousValue: true },
            { ParameterKey: 'HasDefault', UsePreviousValue: true },
            { ParameterKey: 'OtherParameter', ParameterValue: 'SomeValue' },
        ],
    }));
});
test('do not reuse previous parameters if not requested', async () => {
    // GIVEN
    givenStackExists({
        Parameters: [
            { ParameterKey: 'HasValue', ParameterValue: 'TheValue' },
            { ParameterKey: 'HasDefault', ParameterValue: 'TheOldValue' },
        ],
    });
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        stack: FAKE_STACK_WITH_PARAMETERS,
        parameters: {
            HasValue: 'SomeValue',
            OtherParameter: 'SomeValue',
        },
    });
    // THEN
    expect(cfnMocks.createChangeSet).toHaveBeenCalledWith(expect.objectContaining({
        Parameters: [
            { ParameterKey: 'HasValue', ParameterValue: 'SomeValue' },
            { ParameterKey: 'OtherParameter', ParameterValue: 'SomeValue' },
        ],
    }));
});
test('throw exception if not enough parameters supplied', async () => {
    // GIVEN
    givenStackExists({
        Parameters: [
            { ParameterKey: 'HasValue', ParameterValue: 'TheValue' },
            { ParameterKey: 'HasDefault', ParameterValue: 'TheOldValue' },
        ],
    });
    // WHEN
    await expect(api_1.deployStack({
        ...standardDeployStackArguments(),
        stack: FAKE_STACK_WITH_PARAMETERS,
        parameters: {
            OtherParameter: 'SomeValue',
        },
    })).rejects.toThrow(/CloudFormation Parameters are missing a value/);
});
test('deploy is skipped if template did not change', async () => {
    // GIVEN
    givenStackExists();
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
    });
    // THEN
    expect(cfnMocks.executeChangeSet).not.toBeCalled();
});
test('deploy is skipped if parameters are the same', async () => {
    // GIVEN
    givenTemplateIs(FAKE_STACK_WITH_PARAMETERS.template);
    givenStackExists({
        Parameters: [
            { ParameterKey: 'HasValue', ParameterValue: 'HasValue' },
            { ParameterKey: 'HasDefault', ParameterValue: 'HasDefault' },
            { ParameterKey: 'OtherParameter', ParameterValue: 'OtherParameter' },
        ],
    });
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        stack: FAKE_STACK_WITH_PARAMETERS,
        parameters: {},
        usePreviousParameters: true,
    });
    // THEN
    expect(cfnMocks.createChangeSet).not.toHaveBeenCalled();
});
test('deploy is not skipped if parameters are different', async () => {
    // GIVEN
    givenTemplateIs(FAKE_STACK_WITH_PARAMETERS.template);
    givenStackExists({
        Parameters: [
            { ParameterKey: 'HasValue', ParameterValue: 'HasValue' },
            { ParameterKey: 'HasDefault', ParameterValue: 'HasDefault' },
            { ParameterKey: 'OtherParameter', ParameterValue: 'OtherParameter' },
        ],
    });
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        stack: FAKE_STACK_WITH_PARAMETERS,
        parameters: {
            HasValue: 'NewValue',
        },
        usePreviousParameters: true,
    });
    // THEN
    expect(cfnMocks.createChangeSet).toHaveBeenCalledWith(expect.objectContaining({
        Parameters: [
            { ParameterKey: 'HasValue', ParameterValue: 'NewValue' },
            { ParameterKey: 'HasDefault', UsePreviousValue: true },
            { ParameterKey: 'OtherParameter', UsePreviousValue: true },
        ],
    }));
});
test('if existing stack failed to create, it is deleted and recreated', async () => {
    // GIVEN
    givenStackExists({ StackStatus: 'ROLLBACK_COMPLETE' }, // This is for the initial check
    { StackStatus: 'DELETE_COMPLETE' }, // Poll the successful deletion
    { StackStatus: 'CREATE_COMPLETE' });
    givenTemplateIs({
        DifferentThan: 'TheDefault',
    });
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
    });
    // THEN
    expect(cfnMocks.deleteStack).toHaveBeenCalled();
    expect(cfnMocks.createChangeSet).toHaveBeenCalledWith(expect.objectContaining({
        ChangeSetType: 'CREATE',
    }));
});
test('if existing stack failed to create, it is deleted and recreated even if the template did not change', async () => {
    // GIVEN
    givenStackExists({ StackStatus: 'ROLLBACK_COMPLETE' }, // This is for the initial check
    { StackStatus: 'DELETE_COMPLETE' }, // Poll the successful deletion
    { StackStatus: 'CREATE_COMPLETE' });
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
    });
    // THEN
    expect(cfnMocks.deleteStack).toHaveBeenCalled();
    expect(cfnMocks.createChangeSet).toHaveBeenCalledWith(expect.objectContaining({
        ChangeSetType: 'CREATE',
    }));
});
test('deploy not skipped if template did not change and --force is applied', async () => {
    // GIVEN
    givenStackExists();
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        force: true,
    });
    // THEN
    expect(cfnMocks.executeChangeSet).toHaveBeenCalled();
});
test('deploy is skipped if template and tags did not change', async () => {
    // GIVEN
    givenStackExists({
        Tags: [
            { Key: 'Key1', Value: 'Value1' },
            { Key: 'Key2', Value: 'Value2' },
        ],
    });
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        tags: [
            { Key: 'Key1', Value: 'Value1' },
            { Key: 'Key2', Value: 'Value2' },
        ],
    });
    // THEN
    expect(cfnMocks.createChangeSet).not.toBeCalled();
    expect(cfnMocks.executeChangeSet).not.toBeCalled();
    expect(cfnMocks.describeStacks).toHaveBeenCalledWith({ StackName: 'withouterrors' });
    expect(cfnMocks.getTemplate).toHaveBeenCalledWith({ StackName: 'withouterrors', TemplateStage: 'Original' });
});
test('deploy not skipped if template did not change but tags changed', async () => {
    // GIVEN
    givenStackExists({
        Tags: [
            { Key: 'Key', Value: 'Value' },
        ],
    });
    // WHEN
    await api_1.deployStack({
        stack: FAKE_STACK,
        sdk,
        sdkProvider,
        resolvedEnvironment: mock_sdk_1.mockResolvedEnvironment(),
        tags: [
            {
                Key: 'Key',
                Value: 'NewValue',
            },
        ],
        toolkitInfo: api_1.ToolkitInfo.bootstraplessDeploymentsOnly(sdk),
    });
    // THEN
    expect(cfnMocks.createChangeSet).toHaveBeenCalled();
    expect(cfnMocks.executeChangeSet).toHaveBeenCalled();
    expect(cfnMocks.describeChangeSet).toHaveBeenCalled();
    expect(cfnMocks.describeStacks).toHaveBeenCalledWith({ StackName: 'withouterrors' });
    expect(cfnMocks.getTemplate).toHaveBeenCalledWith({ StackName: 'withouterrors', TemplateStage: 'Original' });
});
test('deployStack reports no change if describeChangeSet returns specific error', async () => {
    var _a;
    (_a = cfnMocks.describeChangeSet) === null || _a === void 0 ? void 0 : _a.mockImplementation(() => ({
        Status: 'FAILED',
        StatusReason: 'No updates are to be performed.',
    }));
    // WHEN
    const deployResult = await api_1.deployStack({
        ...standardDeployStackArguments(),
    });
    // THEN
    expect(deployResult.noOp).toEqual(true);
});
test('deploy not skipped if template did not change but one tag removed', async () => {
    // GIVEN
    givenStackExists({
        Tags: [
            { Key: 'Key1', Value: 'Value1' },
            { Key: 'Key2', Value: 'Value2' },
        ],
    });
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        tags: [
            { Key: 'Key1', Value: 'Value1' },
        ],
    });
    // THEN
    expect(cfnMocks.createChangeSet).toHaveBeenCalled();
    expect(cfnMocks.executeChangeSet).toHaveBeenCalled();
    expect(cfnMocks.describeChangeSet).toHaveBeenCalled();
    expect(cfnMocks.describeStacks).toHaveBeenCalledWith({ StackName: 'withouterrors' });
    expect(cfnMocks.getTemplate).toHaveBeenCalledWith({ StackName: 'withouterrors', TemplateStage: 'Original' });
});
test('deploy is not skipped if stack is in a _FAILED state', async () => {
    // GIVEN
    givenStackExists({
        StackStatus: 'DELETE_FAILED',
    });
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        usePreviousParameters: true,
    }).catch(() => { });
    // THEN
    expect(cfnMocks.createChangeSet).toHaveBeenCalled();
});
test('existing stack in UPDATE_ROLLBACK_COMPLETE state can be updated', async () => {
    // GIVEN
    givenStackExists({ StackStatus: 'UPDATE_ROLLBACK_COMPLETE' }, // This is for the initial check
    { StackStatus: 'UPDATE_COMPLETE' });
    givenTemplateIs({ changed: 123 });
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
    });
    // THEN
    expect(cfnMocks.deleteStack).not.toHaveBeenCalled();
    expect(cfnMocks.createChangeSet).toHaveBeenCalledWith(expect.objectContaining({
        ChangeSetType: 'UPDATE',
    }));
});
test('deploy not skipped if template changed', async () => {
    // GIVEN
    givenStackExists();
    givenTemplateIs({ changed: 123 });
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
    });
    // THEN
    expect(cfnMocks.executeChangeSet).toHaveBeenCalled();
});
test('not executed and no error if --no-execute is given', async () => {
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        execute: false,
    });
    // THEN
    expect(cfnMocks.executeChangeSet).not.toHaveBeenCalled();
});
test('empty change set is deleted if --execute is given', async () => {
    var _a;
    (_a = cfnMocks.describeChangeSet) === null || _a === void 0 ? void 0 : _a.mockImplementation(() => ({
        Status: 'FAILED',
        StatusReason: 'No updates are to be performed.',
    }));
    // GIVEN
    givenStackExists();
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        execute: true,
        force: true,
    });
    // THEN
    expect(cfnMocks.createChangeSet).toHaveBeenCalled();
    expect(cfnMocks.executeChangeSet).not.toHaveBeenCalled();
    //the first deletion is for any existing cdk change sets, the second is for the deleting the new empty change set
    expect(cfnMocks.deleteChangeSet).toHaveBeenCalledTimes(2);
});
test('empty change set is not deleted if --no-execute is given', async () => {
    var _a;
    (_a = cfnMocks.describeChangeSet) === null || _a === void 0 ? void 0 : _a.mockImplementation(() => ({
        Status: 'FAILED',
        StatusReason: 'No updates are to be performed.',
    }));
    // GIVEN
    givenStackExists();
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        execute: false,
    });
    // THEN
    expect(cfnMocks.createChangeSet).toHaveBeenCalled();
    expect(cfnMocks.executeChangeSet).not.toHaveBeenCalled();
    //the first deletion is for any existing cdk change sets
    expect(cfnMocks.deleteChangeSet).toHaveBeenCalledTimes(1);
});
test('use S3 url for stack deployment if present in Stack Artifact', async () => {
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        stack: util_1.testStack({
            stackName: 'withouterrors',
            properties: {
                stackTemplateAssetObjectUrl: 'https://use-me-use-me/',
            },
        }),
    });
    // THEN
    expect(cfnMocks.createChangeSet).toHaveBeenCalledWith(expect.objectContaining({
        TemplateURL: 'https://use-me-use-me/',
    }));
    expect(cfnMocks.executeChangeSet).toHaveBeenCalled();
});
test('use REST API S3 url with substituted placeholders if manifest url starts with s3://', async () => {
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        stack: util_1.testStack({
            stackName: 'withouterrors',
            properties: {
                stackTemplateAssetObjectUrl: 's3://use-me-use-me-${AWS::AccountId}/object',
            },
        }),
    });
    // THEN
    expect(cfnMocks.createChangeSet).toHaveBeenCalledWith(expect.objectContaining({
        TemplateURL: 'https://s3.bermuda-triangle-1337.amazonaws.com/use-me-use-me-123456789/object',
    }));
    expect(cfnMocks.executeChangeSet).toHaveBeenCalled();
});
test('changeset is created when stack exists in REVIEW_IN_PROGRESS status', async () => {
    // GIVEN
    givenStackExists({
        StackStatus: 'REVIEW_IN_PROGRESS',
        Tags: [
            { Key: 'Key1', Value: 'Value1' },
            { Key: 'Key2', Value: 'Value2' },
        ],
    });
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        execute: false,
    });
    // THEN
    expect(cfnMocks.createChangeSet).toHaveBeenCalledWith(expect.objectContaining({
        ChangeSetType: 'CREATE',
        StackName: 'withouterrors',
    }));
    expect(cfnMocks.executeChangeSet).not.toHaveBeenCalled();
});
test('changeset is updated when stack exists in CREATE_COMPLETE status', async () => {
    // GIVEN
    givenStackExists({
        Tags: [
            { Key: 'Key1', Value: 'Value1' },
            { Key: 'Key2', Value: 'Value2' },
        ],
    });
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        execute: false,
    });
    // THEN
    expect(cfnMocks.createChangeSet).toHaveBeenCalledWith(expect.objectContaining({
        ChangeSetType: 'UPDATE',
        StackName: 'withouterrors',
    }));
    expect(cfnMocks.executeChangeSet).not.toHaveBeenCalled();
});
test('deploy with termination protection enabled', async () => {
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
        stack: FAKE_STACK_TERMINATION_PROTECTION,
    });
    // THEN
    expect(cfnMocks.updateTerminationProtection).toHaveBeenCalledWith(expect.objectContaining({
        EnableTerminationProtection: true,
    }));
});
test('updateTerminationProtection not called when termination protection is undefined', async () => {
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
    });
    // THEN
    expect(cfnMocks.updateTerminationProtection).not.toHaveBeenCalled();
});
test('updateTerminationProtection called when termination protection is undefined and stack has termination protection', async () => {
    // GIVEN
    givenStackExists({
        EnableTerminationProtection: true,
    });
    // WHEN
    await api_1.deployStack({
        ...standardDeployStackArguments(),
    });
    // THEN
    expect(cfnMocks.updateTerminationProtection).toHaveBeenCalledWith(expect.objectContaining({
        EnableTerminationProtection: false,
    }));
});
/**
 * Set up the mocks so that it looks like the stack exists to start with
 *
 * The last element of this array will be continuously repeated.
 */
function givenStackExists(...overrides) {
    cfnMocks.describeStacks.mockReset();
    if (overrides.length === 0) {
        overrides = [{}];
    }
    const baseResponse = {
        StackName: 'mock-stack-name',
        StackId: 'mock-stack-id',
        CreationTime: new Date(),
        StackStatus: 'CREATE_COMPLETE',
        EnableTerminationProtection: false,
    };
    for (const override of overrides.slice(0, overrides.length - 1)) {
        cfnMocks.describeStacks.mockImplementationOnce(() => ({
            Stacks: [{ ...baseResponse, ...override }],
        }));
    }
    cfnMocks.describeStacks.mockImplementation(() => ({
        Stacks: [{ ...baseResponse, ...overrides[overrides.length - 1] }],
    }));
}
function givenTemplateIs(template) {
    cfnMocks.getTemplate.mockReset();
    cfnMocks.getTemplate.mockReturnValue({
        TemplateBody: JSON.stringify(template),
    });
}
//# sourceMappingURL=data:application/json;base64,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