"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const settings_1 = require("../lib/settings");
const state = {};
beforeAll(async (done) => {
    state.previousWorkingDir = process.cwd();
    state.tempDir = await fs.mkdtemp(path.join(os.tmpdir(), 'aws-cdk-test'));
    // eslint-disable-next-line no-console
    console.log('Temporary working directory:', state.tempDir);
    process.chdir(state.tempDir);
    done();
});
afterAll(async (done) => {
    // eslint-disable-next-line no-console
    console.log('Switching back to', state.previousWorkingDir, 'cleaning up', state.tempDir);
    process.chdir(state.previousWorkingDir);
    await fs.remove(state.tempDir);
    done();
});
test('load context from both files if available', async () => {
    // GIVEN
    await fs.writeJSON('cdk.context.json', { foo: 'bar' });
    await fs.writeJSON('cdk.json', { context: { boo: 'far' } });
    // WHEN
    const config = await new settings_1.Configuration({ readUserContext: false }).load();
    // THEN
    expect(config.context.get('foo')).toBe('bar');
    expect(config.context.get('boo')).toBe('far');
});
test('deleted context disappears from new file', async () => {
    // GIVEN
    await fs.writeJSON('cdk.context.json', { foo: 'bar' });
    await fs.writeJSON('cdk.json', { context: { foo: 'bar' } });
    const config = await new settings_1.Configuration({ readUserContext: false }).load();
    // WHEN
    config.context.unset('foo');
    await config.saveContext();
    // THEN
    expect(await fs.readJSON('cdk.context.json')).toEqual({});
    expect(await fs.readJSON('cdk.json')).toEqual({ context: { foo: 'bar' } });
});
test('clear deletes from new file', async () => {
    // GIVEN
    await fs.writeJSON('cdk.context.json', { foo: 'bar' });
    await fs.writeJSON('cdk.json', { context: { boo: 'far' } });
    const config = await new settings_1.Configuration({ readUserContext: false }).load();
    // WHEN
    config.context.clear();
    await config.saveContext();
    // THEN
    expect(await fs.readJSON('cdk.context.json')).toEqual({});
    expect(await fs.readJSON('cdk.json')).toEqual({ context: { boo: 'far' } });
});
test('context is preserved in the location from which it is read', async () => {
    // GIVEN
    await fs.writeJSON('cdk.json', { context: { 'boo:boo': 'far' } });
    const config = await new settings_1.Configuration({ readUserContext: false }).load();
    // WHEN
    expect(config.context.all).toEqual({ 'boo:boo': 'far' });
    await config.saveContext();
    // THEN
    expect(await fs.readJSON('cdk.context.json')).toEqual({});
    expect(await fs.readJSON('cdk.json')).toEqual({ context: { 'boo:boo': 'far' } });
});
test('surive no context in old file', async () => {
    // GIVEN
    await fs.writeJSON('cdk.json', {});
    await fs.writeJSON('cdk.context.json', { boo: 'far' });
    const config = await new settings_1.Configuration({ readUserContext: false }).load();
    // WHEN
    expect(config.context.all).toEqual({ boo: 'far' });
    await config.saveContext();
    // THEN
    expect(await fs.readJSON('cdk.context.json')).toEqual({ boo: 'far' });
});
test('command line context is merged with stored context', async () => {
    // GIVEN
    await fs.writeJSON('cdk.context.json', { boo: 'far' });
    const config = await new settings_1.Configuration({
        readUserContext: false,
        commandLineArguments: {
            context: ['foo=bar'],
            _: ['command'],
        },
    }).load();
    // WHEN
    expect(config.context.all).toEqual({ foo: 'bar', boo: 'far' });
});
test('can save and load', async () => {
    // GIVEN
    const config1 = await new settings_1.Configuration({ readUserContext: false }).load();
    config1.context.set('some_key', 'some_value');
    await config1.saveContext();
    expect(config1.context.get('some_key')).toEqual('some_value');
    // WHEN
    const config2 = await new settings_1.Configuration({ readUserContext: false }).load();
    // THEN
    expect(config2.context.get('some_key')).toEqual('some_value');
});
test('transient values arent saved to disk', async () => {
    // GIVEN
    const config1 = await new settings_1.Configuration({ readUserContext: false }).load();
    config1.context.set('some_key', { [settings_1.TRANSIENT_CONTEXT_KEY]: true, value: 'some_value' });
    await config1.saveContext();
    expect(config1.context.get('some_key').value).toEqual('some_value');
    // WHEN
    const config2 = await new settings_1.Configuration({ readUserContext: false }).load();
    // THEN
    expect(config2.context.get('some_key')).toEqual(undefined);
});
//# sourceMappingURL=data:application/json;base64,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