"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = require("fs");
const os = require("os");
const path = require("path");
const aws_1 = require("../helpers/aws");
const cdk_1 = require("../helpers/cdk");
const test_helpers_1 = require("../helpers/test-helpers");
jest.setTimeout(600 * 1000);
test_helpers_1.integTest('VPC Lookup', cdk_1.withDefaultFixture(async (fixture) => {
    fixture.log('Making sure we are clean before starting.');
    await fixture.cdkDestroy('define-vpc', { modEnv: { ENABLE_VPC_TESTING: 'DEFINE' } });
    fixture.log('Setting up: creating a VPC with known tags');
    await fixture.cdkDeploy('define-vpc', { modEnv: { ENABLE_VPC_TESTING: 'DEFINE' } });
    fixture.log('Setup complete!');
    fixture.log('Verifying we can now import that VPC');
    await fixture.cdkDeploy('import-vpc', { modEnv: { ENABLE_VPC_TESTING: 'IMPORT' } });
}));
test_helpers_1.integTest('Two ways of shoing the version', cdk_1.withDefaultFixture(async (fixture) => {
    const version1 = await fixture.cdk(['version'], { verbose: false });
    const version2 = await fixture.cdk(['--version'], { verbose: false });
    expect(version1).toEqual(version2);
}));
test_helpers_1.integTest('Termination protection', cdk_1.withDefaultFixture(async (fixture) => {
    const stackName = 'termination-protection';
    await fixture.cdkDeploy(stackName);
    // Try a destroy that should fail
    await expect(fixture.cdkDestroy(stackName)).rejects.toThrow('exited with error');
    // Can update termination protection even though the change set doesn't contain changes
    await fixture.cdkDeploy(stackName, { modEnv: { TERMINATION_PROTECTION: 'FALSE' } });
    await fixture.cdkDestroy(stackName);
}));
test_helpers_1.integTest('cdk synth', cdk_1.withDefaultFixture(async (fixture) => {
    await expect(fixture.cdk(['synth', fixture.fullStackName('test-1')], { verbose: false })).resolves.toEqual(`Resources:
  topic69831491:
    Type: AWS::SNS::Topic
    Metadata:
      aws:cdk:path: ${fixture.stackNamePrefix}-test-1/topic/Resource`);
    await expect(fixture.cdk(['synth', fixture.fullStackName('test-2')], { verbose: false })).resolves.toEqual(`Resources:
  topic152D84A37:
    Type: AWS::SNS::Topic
    Metadata:
      aws:cdk:path: ${fixture.stackNamePrefix}-test-2/topic1/Resource
  topic2A4FB547F:
    Type: AWS::SNS::Topic
    Metadata:
      aws:cdk:path: ${fixture.stackNamePrefix}-test-2/topic2/Resource`);
}));
test_helpers_1.integTest('ssm parameter provider error', cdk_1.withDefaultFixture(async (fixture) => {
    await expect(fixture.cdk(['synth',
        fixture.fullStackName('missing-ssm-parameter'),
        '-c', 'test:ssm-parameter-name=/does/not/exist'], {
        allowErrExit: true,
    })).resolves.toContain('SSM parameter not available in account');
}));
test_helpers_1.integTest('automatic ordering', cdk_1.withDefaultFixture(async (fixture) => {
    // Deploy the consuming stack which will include the producing stack
    await fixture.cdkDeploy('order-consuming');
    // Destroy the providing stack which will include the consuming stack
    await fixture.cdkDestroy('order-providing');
}));
test_helpers_1.integTest('context setting', cdk_1.withDefaultFixture(async (fixture) => {
    await fs_1.promises.writeFile(path.join(fixture.integTestDir, 'cdk.context.json'), JSON.stringify({
        contextkey: 'this is the context value',
    }));
    try {
        await expect(fixture.cdk(['context'])).resolves.toContain('this is the context value');
        // Test that deleting the contextkey works
        await fixture.cdk(['context', '--reset', 'contextkey']);
        await expect(fixture.cdk(['context'])).resolves.not.toContain('this is the context value');
        // Test that forced delete of the context key does not throw
        await fixture.cdk(['context', '-f', '--reset', 'contextkey']);
    }
    finally {
        await fs_1.promises.unlink(path.join(fixture.integTestDir, 'cdk.context.json'));
    }
}));
test_helpers_1.integTest('context in stage propagates to top', cdk_1.withDefaultFixture(async (fixture) => {
    await expect(fixture.cdkSynth({
        // This will make it error to prove that the context bubbles up, and also that we can fail on command
        options: ['--no-lookups'],
        modEnv: {
            INTEG_STACK_SET: 'stage-using-context',
        },
        allowErrExit: true,
    })).resolves.toContain('Context lookups have been disabled');
}));
test_helpers_1.integTest('deploy', cdk_1.withDefaultFixture(async (fixture) => {
    var _a;
    const stackArn = await fixture.cdkDeploy('test-2', { captureStderr: false });
    // verify the number of resources in the stack
    const response = await fixture.aws.cloudFormation('describeStackResources', {
        StackName: stackArn,
    });
    expect((_a = response.StackResources) === null || _a === void 0 ? void 0 : _a.length).toEqual(2);
}));
test_helpers_1.integTest('deploy all', cdk_1.withDefaultFixture(async (fixture) => {
    const arns = await fixture.cdkDeploy('test-*', { captureStderr: false });
    // verify that we only deployed a single stack (there's a single ARN in the output)
    expect(arns.split('\n').length).toEqual(2);
}));
test_helpers_1.integTest('nested stack with parameters', cdk_1.withDefaultFixture(async (fixture) => {
    var _a;
    // STACK_NAME_PREFIX is used in MyTopicParam to allow multiple instances
    // of this test to run in parallel, othewise they will attempt to create the same SNS topic.
    const stackArn = await fixture.cdkDeploy('with-nested-stack-using-parameters', {
        options: ['--parameters', `MyTopicParam=${fixture.stackNamePrefix}ThereIsNoSpoon`],
        captureStderr: false,
    });
    // verify that we only deployed a single stack (there's a single ARN in the output)
    expect(stackArn.split('\n').length).toEqual(1);
    // verify the number of resources in the stack
    const response = await fixture.aws.cloudFormation('describeStackResources', {
        StackName: stackArn,
    });
    expect((_a = response.StackResources) === null || _a === void 0 ? void 0 : _a.length).toEqual(1);
}));
test_helpers_1.integTest('deploy without execute a named change set', cdk_1.withDefaultFixture(async (fixture) => {
    var _a;
    const changeSetName = 'custom-change-set-name';
    const stackArn = await fixture.cdkDeploy('test-2', {
        options: ['--no-execute', '--change-set-name', changeSetName],
        captureStderr: false,
    });
    // verify that we only deployed a single stack (there's a single ARN in the output)
    expect(stackArn.split('\n').length).toEqual(1);
    const response = await fixture.aws.cloudFormation('describeStacks', {
        StackName: stackArn,
    });
    expect((_a = response.Stacks) === null || _a === void 0 ? void 0 : _a[0].StackStatus).toEqual('REVIEW_IN_PROGRESS');
    //verify a change set was created with the provided name
    const changeSetResponse = await fixture.aws.cloudFormation('listChangeSets', {
        StackName: stackArn,
    });
    const changeSets = changeSetResponse.Summaries || [];
    expect(changeSets.length).toEqual(1);
    expect(changeSets[0].ChangeSetName).toEqual(changeSetName);
    expect(changeSets[0].Status).toEqual('CREATE_COMPLETE');
}));
test_helpers_1.integTest('security related changes without a CLI are expected to fail', cdk_1.withDefaultFixture(async (fixture) => {
    // redirect /dev/null to stdin, which means there will not be tty attached
    // since this stack includes security-related changes, the deployment should
    // immediately fail because we can't confirm the changes
    const stackName = 'iam-test';
    await expect(fixture.cdkDeploy(stackName, {
        options: ['<', '/dev/null'],
        neverRequireApproval: false,
    })).rejects.toThrow('exited with error');
    // Ensure stack was not deployed
    await expect(fixture.aws.cloudFormation('describeStacks', {
        StackName: fixture.fullStackName(stackName),
    })).rejects.toThrow('does not exist');
}));
test_helpers_1.integTest('deploy wildcard with outputs', cdk_1.withDefaultFixture(async (fixture) => {
    const outputsFile = path.join(fixture.integTestDir, 'outputs', 'outputs.json');
    await fs_1.promises.mkdir(path.dirname(outputsFile), { recursive: true });
    await fixture.cdkDeploy(['outputs-test-*'], {
        options: ['--outputs-file', outputsFile],
    });
    const outputs = JSON.parse((await fs_1.promises.readFile(outputsFile, { encoding: 'utf-8' })).toString());
    expect(outputs).toEqual({
        [`${fixture.stackNamePrefix}-outputs-test-1`]: {
            TopicName: `${fixture.stackNamePrefix}-outputs-test-1MyTopic`,
        },
        [`${fixture.stackNamePrefix}-outputs-test-2`]: {
            TopicName: `${fixture.stackNamePrefix}-outputs-test-2MyOtherTopic`,
        },
    });
}));
test_helpers_1.integTest('deploy with parameters', cdk_1.withDefaultFixture(async (fixture) => {
    var _a;
    const stackArn = await fixture.cdkDeploy('param-test-1', {
        options: [
            '--parameters', `TopicNameParam=${fixture.stackNamePrefix}bazinga`,
        ],
        captureStderr: false,
    });
    const response = await fixture.aws.cloudFormation('describeStacks', {
        StackName: stackArn,
    });
    expect((_a = response.Stacks) === null || _a === void 0 ? void 0 : _a[0].Parameters).toEqual([
        {
            ParameterKey: 'TopicNameParam',
            ParameterValue: `${fixture.stackNamePrefix}bazinga`,
        },
    ]);
}));
test_helpers_1.integTest('update to stack in ROLLBACK_COMPLETE state will delete stack and create a new one', cdk_1.withDefaultFixture(async (fixture) => {
    var _a, _b, _c, _d;
    // GIVEN
    await expect(fixture.cdkDeploy('param-test-1', {
        options: [
            '--parameters', `TopicNameParam=${fixture.stackNamePrefix}@aww`,
        ],
        captureStderr: false,
    })).rejects.toThrow('exited with error');
    const response = await fixture.aws.cloudFormation('describeStacks', {
        StackName: fixture.fullStackName('param-test-1'),
    });
    const stackArn = (_a = response.Stacks) === null || _a === void 0 ? void 0 : _a[0].StackId;
    expect((_b = response.Stacks) === null || _b === void 0 ? void 0 : _b[0].StackStatus).toEqual('ROLLBACK_COMPLETE');
    // WHEN
    const newStackArn = await fixture.cdkDeploy('param-test-1', {
        options: [
            '--parameters', `TopicNameParam=${fixture.stackNamePrefix}allgood`,
        ],
        captureStderr: false,
    });
    const newStackResponse = await fixture.aws.cloudFormation('describeStacks', {
        StackName: newStackArn,
    });
    // THEN
    expect(stackArn).not.toEqual(newStackArn); // new stack was created
    expect((_c = newStackResponse.Stacks) === null || _c === void 0 ? void 0 : _c[0].StackStatus).toEqual('CREATE_COMPLETE');
    expect((_d = newStackResponse.Stacks) === null || _d === void 0 ? void 0 : _d[0].Parameters).toEqual([
        {
            ParameterKey: 'TopicNameParam',
            ParameterValue: `${fixture.stackNamePrefix}allgood`,
        },
    ]);
}));
test_helpers_1.integTest('stack in UPDATE_ROLLBACK_COMPLETE state can be updated', cdk_1.withDefaultFixture(async (fixture) => {
    var _a, _b, _c, _d;
    // GIVEN
    const stackArn = await fixture.cdkDeploy('param-test-1', {
        options: [
            '--parameters', `TopicNameParam=${fixture.stackNamePrefix}nice`,
        ],
        captureStderr: false,
    });
    let response = await fixture.aws.cloudFormation('describeStacks', {
        StackName: stackArn,
    });
    expect((_a = response.Stacks) === null || _a === void 0 ? void 0 : _a[0].StackStatus).toEqual('CREATE_COMPLETE');
    // bad parameter name with @ will put stack into UPDATE_ROLLBACK_COMPLETE
    await expect(fixture.cdkDeploy('param-test-1', {
        options: [
            '--parameters', `TopicNameParam=${fixture.stackNamePrefix}@aww`,
        ],
        captureStderr: false,
    })).rejects.toThrow('exited with error');
    ;
    response = await fixture.aws.cloudFormation('describeStacks', {
        StackName: stackArn,
    });
    expect((_b = response.Stacks) === null || _b === void 0 ? void 0 : _b[0].StackStatus).toEqual('UPDATE_ROLLBACK_COMPLETE');
    // WHEN
    await fixture.cdkDeploy('param-test-1', {
        options: [
            '--parameters', `TopicNameParam=${fixture.stackNamePrefix}allgood`,
        ],
        captureStderr: false,
    });
    response = await fixture.aws.cloudFormation('describeStacks', {
        StackName: stackArn,
    });
    // THEN
    expect((_c = response.Stacks) === null || _c === void 0 ? void 0 : _c[0].StackStatus).toEqual('UPDATE_COMPLETE');
    expect((_d = response.Stacks) === null || _d === void 0 ? void 0 : _d[0].Parameters).toEqual([
        {
            ParameterKey: 'TopicNameParam',
            ParameterValue: `${fixture.stackNamePrefix}allgood`,
        },
    ]);
}));
test_helpers_1.integTest('deploy with wildcard and parameters', cdk_1.withDefaultFixture(async (fixture) => {
    await fixture.cdkDeploy('param-test-*', {
        options: [
            '--parameters', `${fixture.stackNamePrefix}-param-test-1:TopicNameParam=${fixture.stackNamePrefix}bazinga`,
            '--parameters', `${fixture.stackNamePrefix}-param-test-2:OtherTopicNameParam=${fixture.stackNamePrefix}ThatsMySpot`,
            '--parameters', `${fixture.stackNamePrefix}-param-test-3:DisplayNameParam=${fixture.stackNamePrefix}HeyThere`,
            '--parameters', `${fixture.stackNamePrefix}-param-test-3:OtherDisplayNameParam=${fixture.stackNamePrefix}AnotherOne`,
        ],
    });
}));
test_helpers_1.integTest('deploy with parameters multi', cdk_1.withDefaultFixture(async (fixture) => {
    var _a;
    const paramVal1 = `${fixture.stackNamePrefix}bazinga`;
    const paramVal2 = `${fixture.stackNamePrefix}=jagshemash`;
    const stackArn = await fixture.cdkDeploy('param-test-3', {
        options: [
            '--parameters', `DisplayNameParam=${paramVal1}`,
            '--parameters', `OtherDisplayNameParam=${paramVal2}`,
        ],
        captureStderr: false,
    });
    const response = await fixture.aws.cloudFormation('describeStacks', {
        StackName: stackArn,
    });
    expect((_a = response.Stacks) === null || _a === void 0 ? void 0 : _a[0].Parameters).toEqual([
        {
            ParameterKey: 'DisplayNameParam',
            ParameterValue: paramVal1,
        },
        {
            ParameterKey: 'OtherDisplayNameParam',
            ParameterValue: paramVal2,
        },
    ]);
}));
test_helpers_1.integTest('deploy with notification ARN', cdk_1.withDefaultFixture(async (fixture) => {
    var _a;
    const topicName = `${fixture.stackNamePrefix}-test-topic`;
    const response = await fixture.aws.sns('createTopic', { Name: topicName });
    const topicArn = response.TopicArn;
    try {
        await fixture.cdkDeploy('test-2', {
            options: ['--notification-arns', topicArn],
        });
        // verify that the stack we deployed has our notification ARN
        const describeResponse = await fixture.aws.cloudFormation('describeStacks', {
            StackName: fixture.fullStackName('test-2'),
        });
        expect((_a = describeResponse.Stacks) === null || _a === void 0 ? void 0 : _a[0].NotificationARNs).toEqual([topicArn]);
    }
    finally {
        await fixture.aws.sns('deleteTopic', {
            TopicArn: topicArn,
        });
    }
}));
test_helpers_1.integTest('deploy with role', cdk_1.withDefaultFixture(async (fixture) => {
    const roleName = `${fixture.stackNamePrefix}-test-role`;
    await deleteRole();
    const createResponse = await fixture.aws.iam('createRole', {
        RoleName: roleName,
        AssumeRolePolicyDocument: JSON.stringify({
            Version: '2012-10-17',
            Statement: [{
                    Action: 'sts:AssumeRole',
                    Principal: { Service: 'cloudformation.amazonaws.com' },
                    Effect: 'Allow',
                }, {
                    Action: 'sts:AssumeRole',
                    Principal: { AWS: (await fixture.aws.sts('getCallerIdentity', {})).Arn },
                    Effect: 'Allow',
                }],
        }),
    });
    const roleArn = createResponse.Role.Arn;
    try {
        await fixture.aws.iam('putRolePolicy', {
            RoleName: roleName,
            PolicyName: 'DefaultPolicy',
            PolicyDocument: JSON.stringify({
                Version: '2012-10-17',
                Statement: [{
                        Action: '*',
                        Resource: '*',
                        Effect: 'Allow',
                    }],
            }),
        });
        await aws_1.retry(fixture.output, 'Trying to assume fresh role', aws_1.retry.forSeconds(300), async () => {
            await fixture.aws.sts('assumeRole', {
                RoleArn: roleArn,
                RoleSessionName: 'testing',
            });
        });
        // In principle, the role has replicated from 'us-east-1' to wherever we're testing.
        // Give it a little more sleep to make sure CloudFormation is not hitting a box
        // that doesn't have it yet.
        await aws_1.sleep(5000);
        await fixture.cdkDeploy('test-2', {
            options: ['--role-arn', roleArn],
        });
        // Immediately delete the stack again before we delete the role.
        //
        // Since roles are sticky, if we delete the role before the stack, subsequent DeleteStack
        // operations will fail when CloudFormation tries to assume the role that's already gone.
        await fixture.cdkDestroy('test-2');
    }
    finally {
        await deleteRole();
    }
    async function deleteRole() {
        try {
            for (const policyName of (await fixture.aws.iam('listRolePolicies', { RoleName: roleName })).PolicyNames) {
                await fixture.aws.iam('deleteRolePolicy', {
                    RoleName: roleName,
                    PolicyName: policyName,
                });
            }
            await fixture.aws.iam('deleteRole', { RoleName: roleName });
        }
        catch (e) {
            if (e.message.indexOf('cannot be found') > -1) {
                return;
            }
            throw e;
        }
    }
}));
test_helpers_1.integTest('cdk diff', cdk_1.withDefaultFixture(async (fixture) => {
    const diff1 = await fixture.cdk(['diff', fixture.fullStackName('test-1')]);
    expect(diff1).toContain('AWS::SNS::Topic');
    const diff2 = await fixture.cdk(['diff', fixture.fullStackName('test-2')]);
    expect(diff2).toContain('AWS::SNS::Topic');
    // We can make it fail by passing --fail
    await expect(fixture.cdk(['diff', '--fail', fixture.fullStackName('test-1')]))
        .rejects.toThrow('exited with error');
}));
test_helpers_1.integTest('cdk diff --fail on multiple stacks exits with error if any of the stacks contains a diff', cdk_1.withDefaultFixture(async (fixture) => {
    // GIVEN
    const diff1 = await fixture.cdk(['diff', fixture.fullStackName('test-1')]);
    expect(diff1).toContain('AWS::SNS::Topic');
    await fixture.cdkDeploy('test-2');
    const diff2 = await fixture.cdk(['diff', fixture.fullStackName('test-2')]);
    expect(diff2).toContain('There were no differences');
    // WHEN / THEN
    await expect(fixture.cdk(['diff', '--fail', fixture.fullStackName('test-1'), fixture.fullStackName('test-2')])).rejects.toThrow('exited with error');
}));
test_helpers_1.integTest('cdk diff --fail with multiple stack exits with if any of the stacks contains a diff', cdk_1.withDefaultFixture(async (fixture) => {
    // GIVEN
    await fixture.cdkDeploy('test-1');
    const diff1 = await fixture.cdk(['diff', fixture.fullStackName('test-1')]);
    expect(diff1).toContain('There were no differences');
    const diff2 = await fixture.cdk(['diff', fixture.fullStackName('test-2')]);
    expect(diff2).toContain('AWS::SNS::Topic');
    // WHEN / THEN
    await expect(fixture.cdk(['diff', '--fail', fixture.fullStackName('test-1'), fixture.fullStackName('test-2')])).rejects.toThrow('exited with error');
}));
test_helpers_1.integTest('cdk diff --security-only --fail exits when security changes are present', cdk_1.withDefaultFixture(async (fixture) => {
    const stackName = 'iam-test';
    await expect(fixture.cdk(['diff', '--security-only', '--fail', fixture.fullStackName(stackName)])).rejects.toThrow('exited with error');
}));
test_helpers_1.integTest('deploy stack with docker asset', cdk_1.withDefaultFixture(async (fixture) => {
    await fixture.cdkDeploy('docker');
}));
test_helpers_1.integTest('deploy and test stack with lambda asset', cdk_1.withDefaultFixture(async (fixture) => {
    var _a, _b;
    const stackArn = await fixture.cdkDeploy('lambda', { captureStderr: false });
    const response = await fixture.aws.cloudFormation('describeStacks', {
        StackName: stackArn,
    });
    const lambdaArn = (_b = (_a = response.Stacks) === null || _a === void 0 ? void 0 : _a[0].Outputs) === null || _b === void 0 ? void 0 : _b[0].OutputValue;
    if (lambdaArn === undefined) {
        throw new Error('Stack did not have expected Lambda ARN output');
    }
    const output = await fixture.aws.lambda('invoke', {
        FunctionName: lambdaArn,
    });
    expect(JSON.stringify(output.Payload)).toContain('dear asset');
}));
test_helpers_1.integTest('cdk ls', cdk_1.withDefaultFixture(async (fixture) => {
    const listing = await fixture.cdk(['ls'], { captureStderr: false });
    const expectedStacks = [
        'conditional-resource',
        'docker',
        'docker-with-custom-file',
        'failed',
        'iam-test',
        'lambda',
        'missing-ssm-parameter',
        'order-providing',
        'outputs-test-1',
        'outputs-test-2',
        'param-test-1',
        'param-test-2',
        'param-test-3',
        'termination-protection',
        'test-1',
        'test-2',
        'with-nested-stack',
        'with-nested-stack-using-parameters',
        'order-consuming',
    ];
    for (const stack of expectedStacks) {
        expect(listing).toContain(fixture.fullStackName(stack));
    }
}));
test_helpers_1.integTest('synthing a stage with errors leads to failure', cdk_1.withDefaultFixture(async (fixture) => {
    const output = await fixture.cdk(['synth'], {
        allowErrExit: true,
        modEnv: {
            INTEG_STACK_SET: 'stage-with-errors',
        },
    });
    expect(output).toContain('This is an error');
}));
test_helpers_1.integTest('synthing a stage with errors can be suppressed', cdk_1.withDefaultFixture(async (fixture) => {
    await fixture.cdk(['synth', '--no-validation'], {
        modEnv: {
            INTEG_STACK_SET: 'stage-with-errors',
        },
    });
}));
test_helpers_1.integTest('deploy stack without resource', cdk_1.withDefaultFixture(async (fixture) => {
    // Deploy the stack without resources
    await fixture.cdkDeploy('conditional-resource', { modEnv: { NO_RESOURCE: 'TRUE' } });
    // This should have succeeded but not deployed the stack.
    await expect(fixture.aws.cloudFormation('describeStacks', { StackName: fixture.fullStackName('conditional-resource') }))
        .rejects.toThrow('conditional-resource does not exist');
    // Deploy the stack with resources
    await fixture.cdkDeploy('conditional-resource');
    // Then again WITHOUT resources (this should destroy the stack)
    await fixture.cdkDeploy('conditional-resource', { modEnv: { NO_RESOURCE: 'TRUE' } });
    await expect(fixture.aws.cloudFormation('describeStacks', { StackName: fixture.fullStackName('conditional-resource') }))
        .rejects.toThrow('conditional-resource does not exist');
}));
test_helpers_1.integTest('IAM diff', cdk_1.withDefaultFixture(async (fixture) => {
    const output = await fixture.cdk(['diff', fixture.fullStackName('iam-test')]);
    // Roughly check for a table like this:
    //
    // ┌───┬─────────────────┬────────┬────────────────┬────────────────────────────-──┬───────────┐
    // │   │ Resource        │ Effect │ Action         │ Principal                     │ Condition │
    // ├───┼─────────────────┼────────┼────────────────┼───────────────────────────────┼───────────┤
    // │ + │ ${SomeRole.Arn} │ Allow  │ sts:AssumeRole │ Service:ec2.amazonaws.com     │           │
    // └───┴─────────────────┴────────┴────────────────┴───────────────────────────────┴───────────┘
    expect(output).toContain('${SomeRole.Arn}');
    expect(output).toContain('sts:AssumeRole');
    expect(output).toContain('ec2.amazonaws.com');
}));
test_helpers_1.integTest('fast deploy', cdk_1.withDefaultFixture(async (fixture) => {
    // we are using a stack with a nested stack because CFN will always attempt to
    // update a nested stack, which will allow us to verify that updates are actually
    // skipped unless --force is specified.
    const stackArn = await fixture.cdkDeploy('with-nested-stack', { captureStderr: false });
    const changeSet1 = await getLatestChangeSet();
    // Deploy the same stack again, there should be no new change set created
    await fixture.cdkDeploy('with-nested-stack');
    const changeSet2 = await getLatestChangeSet();
    expect(changeSet2.ChangeSetId).toEqual(changeSet1.ChangeSetId);
    // Deploy the stack again with --force, now we should create a changeset
    await fixture.cdkDeploy('with-nested-stack', { options: ['--force'] });
    const changeSet3 = await getLatestChangeSet();
    expect(changeSet3.ChangeSetId).not.toEqual(changeSet2.ChangeSetId);
    // Deploy the stack again with tags, expected to create a new changeset
    // even though the resources didn't change.
    await fixture.cdkDeploy('with-nested-stack', { options: ['--tags', 'key=value'] });
    const changeSet4 = await getLatestChangeSet();
    expect(changeSet4.ChangeSetId).not.toEqual(changeSet3.ChangeSetId);
    async function getLatestChangeSet() {
        var _a, _b, _c;
        const response = await fixture.aws.cloudFormation('describeStacks', { StackName: stackArn });
        if (!((_a = response.Stacks) === null || _a === void 0 ? void 0 : _a[0])) {
            throw new Error('Did not get a ChangeSet at all');
        }
        fixture.log(`Found Change Set ${(_b = response.Stacks) === null || _b === void 0 ? void 0 : _b[0].ChangeSetId}`);
        return (_c = response.Stacks) === null || _c === void 0 ? void 0 : _c[0];
    }
}));
test_helpers_1.integTest('failed deploy does not hang', cdk_1.withDefaultFixture(async (fixture) => {
    // this will hang if we introduce https://github.com/aws/aws-cdk/issues/6403 again.
    await expect(fixture.cdkDeploy('failed')).rejects.toThrow('exited with error');
}));
test_helpers_1.integTest('can still load old assemblies', cdk_1.withDefaultFixture(async (fixture) => {
    const cxAsmDir = path.join(os.tmpdir(), 'cdk-integ-cx');
    const testAssembliesDirectory = path.join(__dirname, 'cloud-assemblies');
    for (const asmdir of await listChildDirs(testAssembliesDirectory)) {
        fixture.log(`ASSEMBLY ${asmdir}`);
        await cdk_1.cloneDirectory(asmdir, cxAsmDir);
        // Some files in the asm directory that have a .js extension are
        // actually treated as templates. Evaluate them using NodeJS.
        const templates = await listChildren(cxAsmDir, fullPath => Promise.resolve(fullPath.endsWith('.js')));
        for (const template of templates) {
            const targetName = template.replace(/.js$/, '');
            await cdk_1.shell([process.execPath, template, '>', targetName], {
                cwd: cxAsmDir,
                output: fixture.output,
                modEnv: {
                    TEST_ACCOUNT: await fixture.aws.account(),
                    TEST_REGION: fixture.aws.region,
                },
            });
        }
        // Use this directory as a Cloud Assembly
        const output = await fixture.cdk([
            '--app', cxAsmDir,
            '-v',
            'synth',
        ]);
        // Assert that there was no providerError in CDK's stderr
        // Because we rely on the app/framework to actually error in case the
        // provider fails, we inspect the logs here.
        expect(output).not.toContain('$providerError');
    }
}));
test_helpers_1.integTest('generating and loading assembly', cdk_1.withDefaultFixture(async (fixture) => {
    const asmOutputDir = `${fixture.integTestDir}-cdk-integ-asm`;
    await fixture.shell(['rm', '-rf', asmOutputDir]);
    // Synthesize a Cloud Assembly tothe default directory (cdk.out) and a specific directory.
    await fixture.cdk(['synth']);
    await fixture.cdk(['synth', '--output', asmOutputDir]);
    // cdk.out in the current directory and the indicated --output should be the same
    await fixture.shell(['diff', 'cdk.out', asmOutputDir]);
    // Check that we can 'ls' the synthesized asm.
    // Change to some random directory to make sure we're not accidentally loading cdk.json
    const list = await fixture.cdk(['--app', asmOutputDir, 'ls'], { cwd: os.tmpdir() });
    // Same stacks we know are in the app
    expect(list).toContain(`${fixture.stackNamePrefix}-lambda`);
    expect(list).toContain(`${fixture.stackNamePrefix}-test-1`);
    expect(list).toContain(`${fixture.stackNamePrefix}-test-2`);
    // Check that we can use '.' and just synth ,the generated asm
    const stackTemplate = await fixture.cdk(['--app', '.', 'synth', fixture.fullStackName('test-2')], {
        cwd: asmOutputDir,
    });
    expect(stackTemplate).toContain('topic152D84A37');
    // Deploy a Lambda from the copied asm
    await fixture.cdkDeploy('lambda', { options: ['-a', '.'], cwd: asmOutputDir });
    // Remove (rename) the original custom docker file that was used during synth.
    // this verifies that the assemly has a copy of it and that the manifest uses
    // relative paths to reference to it.
    const customDockerFile = path.join(fixture.integTestDir, 'docker', 'Dockerfile.Custom');
    await fs_1.promises.rename(customDockerFile, `${customDockerFile}~`);
    try {
        // deploy a docker image with custom file without synth (uses assets)
        await fixture.cdkDeploy('docker-with-custom-file', { options: ['-a', '.'], cwd: asmOutputDir });
    }
    finally {
        // Rename back to restore fixture to original state
        await fs_1.promises.rename(`${customDockerFile}~`, customDockerFile);
    }
}));
test_helpers_1.integTest('templates on disk contain metadata resource, also in nested assemblies', cdk_1.withDefaultFixture(async (fixture) => {
    // Synth first, and switch on version reporting because cdk.json is disabling it
    await fixture.cdk(['synth', '--version-reporting=true']);
    // Load template from disk from root assembly
    const templateContents = await fixture.shell(['cat', 'cdk.out/*-lambda.template.json']);
    expect(JSON.parse(templateContents).Resources.CDKMetadata).toBeTruthy();
    // Load template from nested assembly
    const nestedTemplateContents = await fixture.shell(['cat', 'cdk.out/assembly-*-stage/*-stage-StackInStage.template.json']);
    expect(JSON.parse(nestedTemplateContents).Resources.CDKMetadata).toBeTruthy();
}));
async function listChildren(parent, pred) {
    const ret = new Array();
    for (const child of await fs_1.promises.readdir(parent, { encoding: 'utf-8' })) {
        const fullPath = path.join(parent, child.toString());
        if (await pred(fullPath)) {
            ret.push(fullPath);
        }
    }
    return ret;
}
async function listChildDirs(parent) {
    return listChildren(parent, async (fullPath) => (await fs_1.promises.stat(fullPath)).isDirectory());
}
//# sourceMappingURL=data:application/json;base64,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