"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.outputFromStack = exports.sleep = exports.retry = exports.isBucketMissingError = exports.isStackMissingError = exports.AwsClients = void 0;
const AWS = require("aws-sdk");
class AwsClients {
    constructor(region, output) {
        this.region = region;
        this.output = output;
        this.config = {
            credentials: chainableCredentials(this.region),
            region: this.region,
            maxRetries: 8,
            retryDelayOptions: { base: 500 },
            stsRegionalEndpoints: 'regional',
        };
        this.cloudFormation = makeAwsCaller(AWS.CloudFormation, this.config);
        this.s3 = makeAwsCaller(AWS.S3, this.config);
        this.ecr = makeAwsCaller(AWS.ECR, this.config);
        this.sns = makeAwsCaller(AWS.SNS, this.config);
        this.iam = makeAwsCaller(AWS.IAM, this.config);
        this.lambda = makeAwsCaller(AWS.Lambda, this.config);
        this.sts = makeAwsCaller(AWS.STS, this.config);
    }
    static async default(output) {
        var _a, _b;
        const region = (_b = (_a = process.env.AWS_REGION) !== null && _a !== void 0 ? _a : process.env.AWS_DEFAULT_REGION) !== null && _b !== void 0 ? _b : 'us-east-1';
        return AwsClients.forRegion(region, output);
    }
    static async forRegion(region, output) {
        return new AwsClients(region, output);
    }
    async account() {
        // Reduce # of retries, we use this as a circuit breaker for detecting no-config
        return (await new AWS.STS({ ...this.config, maxRetries: 1 }).getCallerIdentity().promise()).Account;
    }
    async deleteStacks(...stackNames) {
        if (stackNames.length === 0) {
            return;
        }
        // We purposely do all stacks serially, because they've been ordered
        // to do the bootstrap stack last.
        for (const stackName of stackNames) {
            await this.cloudFormation('updateTerminationProtection', {
                EnableTerminationProtection: false,
                StackName: stackName,
            });
            await this.cloudFormation('deleteStack', {
                StackName: stackName,
            });
            await retry(this.output, `Deleting ${stackName}`, retry.forSeconds(600), async () => {
                const status = await this.stackStatus(stackName);
                if (status !== undefined && status.endsWith('_FAILED')) {
                    throw retry.abort(new Error(`'${stackName}' is in state '${status}'`));
                }
                if (status !== undefined) {
                    throw new Error(`Delete of '${stackName}' not complete yet`);
                }
            });
        }
    }
    async stackStatus(stackName) {
        var _a;
        try {
            return (_a = (await this.cloudFormation('describeStacks', { StackName: stackName })).Stacks) === null || _a === void 0 ? void 0 : _a[0].StackStatus;
        }
        catch (e) {
            if (isStackMissingError(e)) {
                return undefined;
            }
            throw e;
        }
    }
    async emptyBucket(bucketName) {
        const objects = await this.s3('listObjectVersions', { Bucket: bucketName });
        const deletes = [...objects.Versions || [], ...objects.DeleteMarkers || []]
            .reduce((acc, obj) => {
            if (typeof obj.VersionId !== 'undefined' && typeof obj.Key !== 'undefined') {
                acc.push({ Key: obj.Key, VersionId: obj.VersionId });
            }
            else if (typeof obj.Key !== 'undefined') {
                acc.push({ Key: obj.Key });
            }
            return acc;
        }, []);
        if (deletes.length === 0) {
            return Promise.resolve();
        }
        return this.s3('deleteObjects', {
            Bucket: bucketName,
            Delete: {
                Objects: deletes,
                Quiet: false,
            },
        });
    }
    async deleteImageRepository(repositoryName) {
        await this.ecr('deleteRepository', { repositoryName, force: true });
    }
    async deleteBucket(bucketName) {
        try {
            await this.emptyBucket(bucketName);
            await this.s3('deleteBucket', {
                Bucket: bucketName,
            });
        }
        catch (e) {
            if (isBucketMissingError(e)) {
                return;
            }
            throw e;
        }
    }
}
exports.AwsClients = AwsClients;
/**
 * Perform an AWS call from nothing
 *
 * Create the correct client, do the call and resole the promise().
 */
async function awsCall(ctor, config, call, request) {
    const cfn = new ctor(config);
    const response = cfn[call](request);
    try {
        return response.promise();
    }
    catch (e) {
        const newErr = new Error(`${call}(${JSON.stringify(request)}): ${e.message}`);
        newErr.code = e.code;
        throw newErr;
    }
}
/**
 * Factory function to invoke 'awsCall' for specific services.
 *
 * Not strictly necessary but calling this replaces a whole bunch of annoying generics you otherwise have to type:
 *
 * ```ts
 * export function cloudFormation<
 *   C extends keyof ServiceCalls<AWS.CloudFormation>,
 * >(call: C, request: First<ServiceCalls<AWS.CloudFormation>[C]>): Promise<Second<ServiceCalls<AWS.CloudFormation>[C]>> {
 *   return awsCall(AWS.CloudFormation, call, request);
 * }
 * ```
 */
function makeAwsCaller(ctor, config) {
    return (call, request) => {
        return awsCall(ctor, config, call, request);
    };
}
function isStackMissingError(e) {
    return e.message.indexOf('does not exist') > -1;
}
exports.isStackMissingError = isStackMissingError;
function isBucketMissingError(e) {
    return e.message.indexOf('does not exist') > -1;
}
exports.isBucketMissingError = isBucketMissingError;
/**
 * Retry an async operation until a deadline is hit.
 *
 * Use `retry.forSeconds()` to construct a deadline relative to right now.
 *
 * Exceptions will cause the operation to retry. Use `retry.abort` to annotate an exception
 * to stop the retry and end in a failure.
 */
async function retry(output, operation, deadline, block) {
    let i = 0;
    output.write(`💈 ${operation}\n`);
    while (true) {
        try {
            i++;
            const ret = await block();
            output.write(`💈 ${operation}: succeeded after ${i} attempts\n`);
            return ret;
        }
        catch (e) {
            if (e.abort || Date.now() > deadline.getTime()) {
                throw new Error(`${operation}: did not succeed after ${i} attempts: ${e}`);
            }
            output.write(`⏳ ${operation} (${e.message})\n`);
            await sleep(5000);
        }
    }
}
exports.retry = retry;
/**
 * Make a deadline for the `retry` function relative to the current time.
 */
retry.forSeconds = (seconds) => {
    return new Date(Date.now() + seconds * 1000);
};
/**
 * Annotate an error to stop the retrying
 */
retry.abort = (e) => {
    e.abort = true;
    return e;
};
async function sleep(ms) {
    return new Promise(ok => setTimeout(ok, ms));
}
exports.sleep = sleep;
function outputFromStack(key, stack) {
    var _a, _b;
    return (_b = ((_a = stack.Outputs) !== null && _a !== void 0 ? _a : []).find(o => o.OutputKey === key)) === null || _b === void 0 ? void 0 : _b.OutputValue;
}
exports.outputFromStack = outputFromStack;
function chainableCredentials(region) {
    const profileName = process.env.AWS_PROFILE;
    if (process.env.CODEBUILD_BUILD_ARN && profileName) {
        // in codebuild we must assume the role that the cdk uses
        // otherwise credentials will just be picked up by the normal sdk
        // heuristics and expire after an hour.
        // can't use '~' since the SDK doesn't seem to expand it...?
        const configPath = `${process.env.HOME}/.aws/config`;
        const ini = new AWS.IniLoader().loadFrom({
            filename: configPath,
            isConfig: true,
        });
        const profile = ini[profileName];
        if (!profile) {
            throw new Error(`Profile '${profileName}' does not exist in config file (${configPath})`);
        }
        const arn = profile.role_arn;
        const externalId = profile.external_id;
        if (!arn) {
            throw new Error(`role_arn does not exist in profile ${profileName}`);
        }
        if (!externalId) {
            throw new Error(`external_id does not exist in profile ${externalId}`);
        }
        return new AWS.ChainableTemporaryCredentials({
            params: {
                RoleArn: arn,
                ExternalId: externalId,
                RoleSessionName: 'integ-tests',
            },
            stsConfig: {
                region,
            },
            masterCredentials: new AWS.ECSCredentials(),
        });
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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