import re

from .. import io
from .. import cfg


def _beginsMatch(string, query):
  q = query.lower().split()
  for s in string.lower().split():
    if q and s.startswith(q[0]):
      q.pop(0)
  return len(q) == 0


def _filtAbbr(a):
  return re.sub(r'\([^)]*\)', '', a)

def _filtName(n):
  return re.sub(r'\([^)]*\)', '', n)


def find(*name):
  matches = []
  for k, v in journals()[-1].items():
    k, v = _filtAbbr(k), _filtName(v)
    if (_beginsMatch(k, ' '.join(name))
        or _beginsMatch(v, ' '.join(name))):
      if [k, v] not in matches:
        matches.append([k, v])
  matches = sorted(matches, key=lambda m: m[0])
  for k, v in journals()[-1].items():
    k, v = _filtAbbr(k), _filtName(v)
    if (all([n.lower() in k.lower() for n in name])
        or all([n.lower() in v.lower() for n in name])):
      if [k, v] not in matches:
        matches.append([k, v])
  return matches


def _compareStr(s):
  return ' '.join([re.sub(r'[{}\\"]', '', _s.strip('.:-_'))
                    for _s in s.split(':')[0].lower().split()])


def byAbbr(abbr):
  abbr = _compareStr(abbr)
  keys, labbrs, _, j = journals()
  if abbr in labbrs:
    k = keys[labbrs.index(abbr.lower())]
    return k, j[k]
  return None, None


def byName(name):
  name = _compareStr(name)
  keys, _, lnames, j = journals()
  if name in lnames:
    k = keys[lnames.index(name.lower())]
    return k, j[k]
  return None, None


def getOrAsk(jour=None, descr=None):
  # check if journal exists in db
  k, v = byName(jour)
  if k and v:
    return k, v

  k, v = byAbbr(jour)
  if k and v:
    return k, v

  # else show full bibinfo
  if descr:
    io.info('in bibtex entry:', descr)

  # and check if similar journals exist in db
  matches = find(jour)
  if matches:
    options = [m[1] for m in matches[:10]] + ['<manually enter journal>']
    i, j = io.select(f'journal {repr(jour)} not found in database,\n'
                     f'if this journal is among one of the following\n'
                     f'similar journals select it, else select to\n'
                     f'manually enter journal details', options)
    if i < len(options)-1:
      return matches[i]

  else:
    io.info(f'journal {repr(jour)} not found in database')

  while True:
    io.info('please enter full name of the journal:')
    while True:
      full = input().strip()
      if full:
        break

    addToDb = True
    k, v = byName(full)
    if k and v and io.conf('found this journal in database, use database entry?',
                           default=True):
      addToDb = False
      abbr = k
      full = v
    else:
      io.info('please enter ISO4 abbreviated name of the journal:')
      while True:
        abbr = input().strip()
        if abbr:
          break

    if io.conf(f'the correct full name for the journal',
               repr(jour),
               f'is',
               repr(full),
               f'the correct ISO4 abbreviated name is',
               repr(abbr),
               f'is this correct?'):
      break
  if addToDb:
    cfgJournals = cfg.get('z_bib_journals')
    cfgJournals[abbr] = full
    cfg.set('z_bib_journals', cfgJournals)
    journals(forceRegen=True)
  return byName(full)


def printSuspicious():
  for k, v in journals()[-1].items():
    if any([not w.isupper()
              and not w.isdigit()
              and not w.endswith('.')
              and not len(w) > 3
              and not w.lower() in ('new low ion in of'.split())
                          for w in k.split() if w]):
      io.info(f'suspicious abbreviation: "{k}", "{v}"')
    elif any([w.endswith('.') for w in v.split() if w]):
      io.info(f'suspicious full name: "{k}", "{v}"')
    elif any([c in s for c in '()' for s in (k, v)]):
      io.info(f'suspicious characters in: "{k}", "{v}"')


_JOURNALS = None
_KEYS = None
_LKEYS = None
_LNAMES = None

def journals(forceRegen=False):
  global _JOURNALS, _KEYS, _LKEYS, _LNAMES
  if _JOURNALS is None or forceRegen:
    _JOURNALS = {}
    _JOURNALS.update(JOURNALS)
    _JOURNALS.update(CAS_JOURNALS)
    _JOURNALS.update(cfg.get('z_bib_journals'))
    _JOURNALS = {_filtAbbr(k): _filtName(v) for k, v in _JOURNALS.items()}
    _KEYS = list(_JOURNALS.keys())
    _LKEYS = [_compareStr(k) for k in _KEYS]
    _LNAMES = [_compareStr(_JOURNALS[k]) for k in _KEYS]
  return _KEYS, _LKEYS, _LNAMES, _JOURNALS


# manually managed list
JOURNALS = {
  'Physica A': 'Physica A: Statistical Mechanics and its Applications',
  'Physica B Condens. Matter': 'Physica B: Condensed Matter',
  'Physica C Supercond.': 'Physica C: Superconductivity and its Applications',
  'Physica D': 'Physica D: Nonlinear Phenomena',
  'Physica E Low Dimens. Syst. Nanostruct.': 'Physica E: Low-dimensional Systems and Nanostructures',
  'AIP Conf. Proc.': 'AIP Conference Proceedings',
  'Adv. Phys.': 'Advances in Physics',
  'Am. Sci.': 'American Scientist',
  'Ann. Phys.': 'Annalen der Physik',
  'Br. J. Philos. Sci.': 'British Journal for the Philosophy of Science',
  'Commun. Math. Phys.': 'Communications in Mathematical Physics',
  'Contemp. Phys.': ' Contemporary Physics',
  'Czechoslov. J. Phys.': ' Czechoslovak Journal of Physics',
  'Z. Phys. A': '{Zeitschrift} {f\\"ur} {Physik} A Hadrons and Nuclei',
  'Z. Phys. B': '{Zeitschrift} {f\\"ur} {Physik} B Condensed Matter',
  'Z. Phys. C': '{Zeitschrift} {f\\"ur} {Physik} C Particles and Fields',
  'Z. Phys. D': '{Zeitschrift} {f\\"ur} {Physik} D Atoms, Molecules and Clusters',
  'Found. Phys.': ' Foundations of Physics',
  'IBM J. Res. Dev.': ' IBM Journal of Research and Development',
  'J. Math. Phys.': ' Journal of Mathematical Physics',
  'J. Opt. Soc. Am. A': 'Journal of the Optical Society of America A',
  'J. Opt. Soc. Am. B': ' Journal of the Optical Society of America B',
  'J. Phys. A': 'Journal of Physics A: Mathematical and General',
  'J. Phys. B': 'Journal of Physics B: Atomic, Molecular and Optical Physics',
  'J. Phys. C': 'Journal of Physics C: Solid State Physics',
  'J. Phys. D': 'Journal of Physics D: Applied Physics',
  'J. Phys. E': 'Journal of Physics E: Scientific Instruments',
  'J. Phys. F': 'Journal of Physics F: Metal Physics',
  'J. Phys. G': 'Journal of Physics G: Nuclear and Particle Physics',
  'J. Phys. Soc. Jpn.': ' Journal of the Physical Society of Japan',
  'J. Stat. Phys.': ' Journal of Statistical Physics',
  'J. Supercond. Nov. Magn.': ' Journal of Superconductivity and Novel Magnetism',
  'Nat. Rev. Mater.': ' Nature Reviews Materials',
  'Nature': 'Nature',
  'New J. Phys.': ' New Journal of Physics',
  'Opt. Act. Int. J. Opt.': 'Optica Acta: International Journal of Optics',
  'Phys. Rep.': 'Physics Reports',
  'Phys. Rev.': 'Physical Review Journals Archive',
  'Rep. Prog. Phys.': 'Reports on Progress in Physics',
  'Rev. Mod. Phys.': 'Reviews of Modern Physics',
  'Science': 'Science',
  'Z. Phys.': '{Zeitschrift} {f\\"ur} {Physik}',
  'arXiv': 'arXiv',

  # corrections of CAS database, where text after colon in abbreviated form exists:
  'Acta Crystallogr., Sect. A': 'Acta Crystallographica, Section A: Foundations and Advances',
  'Acta Crystallogr., Sect. B': 'Acta Crystallographica, Section B: Structural Science, Crystal Engineering and Materials',
  'Acta Crystallogr., Sect. C': 'Acta Crystallographica, Section C: Structural Chemistry',
  'Acta Crystallogr., Sect. D': 'Acta Crystallographica, Section D: Biological Crystallography',
  'Acta Crystallogr., Sect. E': 'Acta Crystallographica, Section E: Crystallographic Communications',
  'Acta Crystallogr., Sect. F': 'Acta Crystallographica, Section F: Structural Biology Communications',
  'Appl. Phys. A': 'Applied Physics A: Materials Science &amp; Processing',
  'Appl. Phys. B': 'Applied Physics B: Lasers and Optics',
  'Comp. Biochem. Physiol., Part A': 'Comparative Biochemistry and Physiology, Part A: Molecular &amp; Integrative Physiology',
  'Comp. Biochem. Physiol., Part B': 'Comparative Biochemistry and Physiology, Part B: Biochemistry &amp; Molecular Biology',
  'Comp. Biochem. Physiol., Part C': 'Comparative Biochemistry and Physiology, Part C: Toxicology &amp; Pharmacology',
  'Comp. Biochem. Physiol., Part D': 'Comparative Biochemistry and Physiology, Part D: Genomics &amp; Proteomics',
  'Phys. Rev. A': 'Physical Review A: Atomic, Molecular, and Optical Physics',
  'Phys. Rev. B': 'Physical Review B: Condensed Matter and Materials Physics',
  'Phys. Rev. C': 'Physical Review C: Nuclear Physics',
  'Phys. Rev. D': 'Physical Review D: Particles, Fields, Gravitation, and Cosmology',
  'Phys. Rev. E': 'Physical Review E: Statistical, Nonlinear, and Soft Matter Physics',

  # corrections of CAS database, where fuer was used instaed of {f\\"ur}:
  'Monatsh. Chem.': 'Monatshefte {f\\"ur} Chemie',
  'Neues Jahrb. Mineral., Abh.': 'Neues Jahrbuch {f\\"ur} Mineralogie, Abhandlungen',
  'Z. Anorg. Allg. Chem.': 'Zeitschrift {f\\"ur} Anorganische und Allgemeine Chemie',
  'Z. Lebensm.-Unters. Forsch. A': 'Zeitschrift {f\\"ur} Lebensmittel-Untersuchung und -Forschung A: European Food Research and Technology',
  'Z. Naturforsch., A': 'Zeitschrift {f\\"ur} Naturforschung, A: Physical Sciences',
  'Z. Naturforsch., B': 'Zeitschrift {f\\"ur} Naturforschung, B: A Journal of Chemical Sciences',
  'Z. Naturforsch., C': 'Zeitschrift {f\\"ur} Naturforschung, C: Journal of Biosciences',
  'Z. Phys. Chem.': 'Zeitschrift {f\\"ur} Physikalische Chemie',

}

# from: https://www.cas.org/support/documentation/references/corejournals
CAS_JOURNALS = {
  'ACS Appl. Mater. Interfaces': 'ACS Applied Materials &amp; Interfaces',
  'ACS Biomater. Sci. Eng.': 'ACS Biomaterials Science &amp; Engineering',
  'ACS Catal.': 'ACS Catalysis',
  'ACS Cent. Sci.': 'ACS Central Science',
  'ACS Chem. Biol.': 'ACS Chemical Biology',
  'ACS Chem. Neurosci.': 'ACS Chemical Neuroscience',
  'ACS Comb. Sci.': 'ACS Combinatorial Science',
  'ACS Infect. Dis.': 'ACS Infectious Diseases',
  'ACS Macro Lett.': 'ACS Macro Letters',
  'ACS Med. Chem. Lett.': 'ACS Medicinal Chemistry Letters',
  'ACS Sens.': 'ACS Sensors',
  'ACS Sustainable Chem. Eng.': 'ACS Sustainable Chemistry &amp; Engineering',
  'ACS Symp. Ser.': 'ACS Symposium Series',
  'ACS Synth. Biol.': 'ACS Synthetic Biology',
  'AIChE J.': 'AIChE Journal',
  'APL Mater.': 'APL Materials',
  'ARKIVOC (Gainesville, FL, U. S.)': 'ARKIVOC (Gainesville, FL, United States)',
  'Acc. Chem. Res.': 'Accounts of Chemical Research',
  'Acta Biochim. Pol.': 'Acta Biochimica Polonica',
  'Acta Chim. Slovaca': 'Acta Chimica Slovaca',
  'Acta Chim. Slov.': 'Acta Chimica Slovenica',
  'Acta Crystallogr., Sect. A: Found. Adv.': 'Acta Crystallographica, Section A: Foundations and Advances',
  'Acta Crystallogr., Sect. B: Struct. Sci., Cryst. Eng. Mater.': 'Acta Crystallographica, Section B: Structural Science, Crystal Engineering and Materials',
  'Acta Crystallogr., Sect. C: Struct. Chem.': 'Acta Crystallographica, Section C: Structural Chemistry',
  'Acta Crystallogr., Sect. D: Biol. Crystallogr.': 'Acta Crystallographica, Section D: Biological Crystallography',
  'Acta Crystallogr., Sect. E: Crystallogr. Commun.': 'Acta Crystallographica, Section E: Crystallographic Communications',
  'Acta Crystallogr., Sect. F: Struct. Biol. Commun.': 'Acta Crystallographica, Section F: Structural Biology Communications',
  'Acta Mater.': 'Acta Materialia',
  'Acta Pharm. (Zagreb, Croatia)': 'Acta Pharmaceutica (Zagreb, Croatia)',
  'Acta Pharmacol. Sin.': 'Acta Pharmacologica Sinica',
  'Acta Phys. Pol. A': 'Acta Physica Polonica A',
  'Acta Physiol.': 'Acta Physiologica',
  'Adsorpt. Sci. Technol.': 'Adsorption Science &amp; Technology',
  'Adv. Drug Delivery Rev.': 'Advanced Drug Delivery Reviews',
  'Adv. Electron. Mater.': 'Advanced Electronic Materials',
  'Adv. Energy Mater.': 'Advanced Energy Materials',
  'Adv. Funct. Mater.': 'Advanced Functional Materials',
  'Adv. Healthcare Mater.': 'Advanced Healthcare Materials',
  'Adv. Mater. (Weinheim, Ger.)': 'Advanced Materials (Weinheim, Germany)',
  'Adv. Opt. Mater.': 'Advanced Optical Materials',
  'Adv. Synth. Catal.': 'Advanced Synthesis &amp; Catalysis',
  'Adv. Appl. Ceram.': 'Advances in Applied Ceramics',
  'Adv. Chem. Phys.': 'Advances in Chemical Physics',
  'Adv. Chromatogr. (Boca Raton, FL, U. S.)': 'Advances in Chromatography (Boca Raton, FL, United States)',
  'Adv. Colloid Interface Sci.': 'Advances in Colloid and Interface Science',
  'Adv. Electrochem. Sci. Eng.': 'Advances in Electrochemical Science and Engineering',
  'Adv. Enzyme Res.': 'Advances in Enzyme Research',
  'Adv. Inorg. Chem.': 'Advances in Inorganic Chemistry',
  'Adv. Polym. Sci.': 'Advances in Polymer Science',
  'Adv. Polym. Technol.': 'Advances in Polymer Technology',
  'Adv. X-Ray Anal.': 'Advances in X-Ray Analysis',
  'Aerosol Sci. Technol.': 'Aerosol Science and Technology',
  'Am. Ceram. Soc. Bull.': 'American Ceramic Society Bulletin',
  'Am. Chem. Sci. J.': 'American Chemical Science Journal',
  'Am. J. Clin. Nutr.': 'American Journal of Clinical Nutrition',
  'Am. J. Hum. Genet.': 'American Journal of Human Genetics',
  'Am. J. Physiol.': 'American Journal of Physiology',
  'Am. J. Respir. Cell Mol. Biol.': 'American Journal of Respiratory Cell and Molecular Biology',
  'Analyst (Cambridge, U. K.)': 'Analyst (Cambridge, United Kingdom)',
  'Anal. Chim. Acta': 'Analytica Chimica Acta',
  'Anal. Biochem.': 'Analytical Biochemistry',
  'Anal. Chem. (Washington, DC, U. S.)': 'Analytical Chemistry (Washington, DC, United States)',
  'Anal. Chem. Lett.': 'Analytical Chemistry Letters',
  'Anal. Lett.': 'Analytical Letters',
  'Anal. Sci.': 'Analytical Sciences',
  'Anal. Bioanal. Chem.': 'Analytical and Bioanalytical Chemistry',
  'Angew. Chem., Int. Ed.': 'Angewandte Chemie, International Edition',
  'Ann. Pharm. Fr.': 'Annales Pharmaceutiques Francaises',
  'Ann. Chim. (Cachan, Fr.)': 'Annales de Chimie (Cachan, France)',
  'Ann. Nucl. Energy': 'Annals of Nuclear Energy',
  'Ann. Nutr. Metab.': 'Annals of Nutrition &amp; Metabolism',
  'Ann. Phys. (Amsterdam, Neth.)': 'Annals of Physics (Amsterdam, Netherlands)',
  'Ann. Plast. Surg.': 'Annals of Plastic Surgery',
  'Annu. Rev. Biochem.': 'Annual Review of Biochemistry',
  'Annu. Rev. Biomed. Eng.': 'Annual Review of Biomedical Engineering',
  'Annu. Rev. Biophys.': 'Annual Review of Biophysics',
  'Annu. Rev. Cell Dev. Biol.': 'Annual Review of Cell and Developmental Biology',
  'Annu. Rev. Earth Planet. Sci.': 'Annual Review of Earth and Planetary Sciences',
  'Annu. Rev. Entomol.': 'Annual Review of Entomology',
  'Annu. Rev. Genet.': 'Annual Review of Genetics',
  'Annu. Rev. Genomics Hum. Genet.': 'Annual Review of Genomics and Human Genetics',
  'Annu. Rev. Immunol.': 'Annual Review of Immunology',
  'Annu. Rev. Mater. Res.': 'Annual Review of Materials Research',
  'Annu. Rev. Med.': 'Annual Review of Medicine',
  'Annu. Rev. Microbiol.': 'Annual Review of Microbiology',
  'Annu. Rev. Neurosci.': 'Annual Review of Neuroscience',
  'Annu. Rev. Nucl. Part. Sci.': 'Annual Review of Nuclear and Particle Science',
  'Annu. Rev. Nutr.': 'Annual Review of Nutrition',
  'Annu. Rev. Pathol.: Mech. Dis.': 'Annual Review of Pathology: Mechanisms of Disease',
  'Annu. Rev. Pharmacol. Toxicol.': 'Annual Review of Pharmacology and Toxicology',
  'Annu. Rev. Phys. Chem.': 'Annual Review of Physical Chemistry',
  'Annu. Rev. Physiol.': 'Annual Review of Physiology',
  'Annu. Rev. Phytopathol.': 'Annual Review of Phytopathology',
  'Annu. Rev. Plant Biol.': 'Annual Review of Plant Biology',
  'Annu. Rev. Virol.': 'Annual Review of Virology',
  'Anti-Cancer Agents Med. Chem.': 'Anti-Cancer Agents in Medicinal Chemistry',
  'Anti-Infect. Agents': 'Anti-Infective Agents',
  'Anti-Inflammatory Anti-Allergy Agents Med. Chem.': 'Anti-Inflammatory &amp; Anti-Allergy Agents in Medicinal Chemistry',
  'Antibiotics (Basel, Switz.)': 'Antibiotics (Basel, Switzerland)',
  'Antibiot. Khimioter.': 'Antibiotiki i Khimioterapiya',
  'Antimicrob. Agents Chemother.': 'Antimicrobial Agents and Chemotherapy',
  'Antioxid. Redox Signaling': 'Antioxidants &amp; Redox Signaling',
  'Antiviral Res.': 'Antiviral Research',
  'Appl. Biochem. Biotechnol.': 'Applied Biochemistry and Biotechnology',
  'Appl. Biochem. Microbiol.': 'Applied Biochemistry and Microbiology',
  'Appl. Catal., A': 'Applied Catalysis, A: General',
  'Appl. Catal., B': 'Applied Catalysis, B: Environmental',
  'Appl. Clay Sci.': 'Applied Clay Science',
  'Appl. Geochem.': 'Applied Geochemistry',
  'Appl. Microbiol. Biotechnol.': 'Applied Microbiology and Biotechnology',
  'Appl. Nanosci.': 'Applied Nanoscience',
  'Appl. Organomet. Chem.': 'Applied Organometallic Chemistry',
  'Appl. Phys. A: Mater. Sci. Process.': 'Applied Physics A: Materials Science &amp; Processing',
  'Appl. Phys. B: Lasers Opt.': 'Applied Physics B: Lasers and Optics',
  'Appl. Phys. Express': 'Applied Physics Express',
  'Appl. Phys. Lett.': 'Applied Physics Letters',
  'Appl. Phys. Rev.': 'Applied Physics Reviews',
  'Appl. Radiat. Isot.': 'Applied Radiation and Isotopes',
  'Appl. Spectrosc.': 'Applied Spectroscopy',
  'Appl. Surf. Sci.': 'Applied Surface Science',
  'Appl. Environ. Microbiol.': 'Applied and Environmental Microbiology',
  'Aquat. Toxicol.': 'Aquatic Toxicology',
  'Arch. Pharm. (Weinheim, Ger.)': 'Archiv der Pharmazie (Weinheim, Germany)',
  'Arch. Biochem. Biophys.': 'Archives of Biochemistry and Biophysics',
  'Arch. Environ. Contam. Toxicol.': 'Archives of Environmental Contamination and Toxicology',
  'Arch. Insect Biochem. Physiol.': 'Archives of Insect Biochemistry and Physiology',
  'Arch. Microbiol.': 'Archives of Microbiology',
  'Arch. Pharmacal Res.': 'Archives of Pharmacal Research',
  'Arch. Physiol. Biochem.': 'Archives of Physiology and Biochemistry',
  'Arch. Toxicol.': 'Archives of Toxicology',
  'Arch. Virol.': 'Archives of Virology',
  'Arterioscler., Thromb., Vasc. Biol.': 'Arteriosclerosis, Thrombosis, and Vascular Biology',
  'Asia-Pac. J. Chem. Eng.': 'Asia-Pacific Journal of Chemical Engineering',
  'Asian J. Chem.': 'Asian Journal of Chemistry',
  'Asian J. Org. Chem.': 'Asian Journal of Organic Chemistry',
  'Assay Drug Dev. Technol.': 'Assay and Drug Development Technologies',
  'Atherosclerosis (Amsterdam, Neth.)': 'Atherosclerosis (Amsterdam, Netherlands)',
  'Atmos. Chem. Phys.': 'Atmospheric Chemistry and Physics',
  'Atmos. Environ.': 'Atmospheric Environment',
  'At. Energy (N. Y., NY, U. S.)': 'Atomic Energy (New York, NY, United States)',
  'At. Spectrosc.': 'Atomic Spectroscopy',
  'Aust. J. Chem.': 'Australian Journal of Chemistry',
  'Auton. Autacoid Pharmacol.': 'Autonomic &amp; Autacoid Pharmacology',
  'Azerb. Khim. Zh.': 'Azerbaidzhanskii Khimicheskii Zhurnal',
  'BMB Rep.': 'BMB Reports',
  'Basic Clin. Pharmacol. Toxicol.': 'Basic &amp; Clinical Pharmacology &amp; Toxicology',
  'Beilstein J. Nanotechnol.': 'Beilstein Journal of Nanotechnology',
  'Beilstein J. Org. Chem.': 'Beilstein Journal of Organic Chemistry',
  'BioChem.: Indian J.': 'BioChemistry: An Indian Journal',
  'BioChip J.': 'BioChip Journal',
  'BioInorg. React. Mech.': 'BioInorganic Reaction Mechanisms',
  'Bioact. Carbohydr. Diet. Fibre': 'Bioactive Carbohydrates and Dietary Fibre',
  'Biocatal. Biotransform.': 'Biocatalysis and Biotransformation',
  'Biochem. Genet.': 'Biochemical Genetics',
  'Biochem. J.': 'Biochemical Journal',
  'Biochem. Pharmacol. (Amsterdam, Neth.)': 'Biochemical Pharmacology (Amsterdam, Netherlands)',
  'Biochem. Soc. Trans.': 'Biochemical Society Transactions',
  'Biochem. Syst. Ecol.': 'Biochemical Systematics and Ecology',
  'Biochem. Biophys. Res. Commun.': 'Biochemical and Biophysical Research Communications',
  'Biochem. Cell Biol.': 'Biochemistry and Cell Biology',
  'Biochem. Mol. Biol. Educ.': 'Biochemistry and Molecular Biology Education',
  'Biochim. Biophys. Acta, Bioenerg.': 'Biochimica et Biophysica Acta, Bioenergetics',
  'Biochim. Biophys. Acta, Biomembr.': 'Biochimica et Biophysica Acta, Biomembranes',
  'Biochim. Biophys. Acta, Gene Regul. Mech.': 'Biochimica et Biophysica Acta, Gene Regulatory Mechanisms',
  'Biochim. Biophys. Acta, Gen. Subj.': 'Biochimica et Biophysica Acta, General Subjects',
  'Biochim. Biophys. Acta, Mol. Basis Dis.': 'Biochimica et Biophysica Acta, Molecular Basis of Disease',
  'Biochim. Biophys. Acta, Mol. Cell Res.': 'Biochimica et Biophysica Acta, Molecular Cell Research',
  'Biochim. Biophys. Acta, Mol. Cell Biol. Lipids': 'Biochimica et Biophysica Acta, Molecular and Cell Biology of Lipids',
  'Biochim. Biophys. Acta, Proteins Proteomics': 'Biochimica et Biophysica Acta, Proteins and Proteomics',
  'Biochim. Biophys. Acta, Rev. Cancer': 'Biochimica et Biophysica Acta, Reviews on Cancer',
  'Bioconjugate Chem.': 'Bioconjugate Chemistry',
  'Biofuels, Bioprod. Biorefin.': 'Biofuels, Bioproducts &amp; Biorefining',
  'Biog. Amines': 'Biogenic Amines',
  'Bioinorg. Chem. Appl.': 'Bioinorganic Chemistry and Applications',
  'Biol. Pharm. Bull.': 'Biological &amp; Pharmaceutical Bulletin',
  'Biol. Chem.': 'Biological Chemistry',
  'Biol. Trace Elem. Res.': 'Biological Trace Element Research',
  'Biol. Membr.': 'Biologicheskie Membrany',
  'Biol. Reprod.': 'Biology of Reproduction',
  'Biomacromol. Mass Spectrom.': 'Biomacromolecular Mass Spectrometry',
  'Biomarkers Med.': 'Biomarkers in Medicine',
  'Biomater. Sci.': 'Biomaterials Science',
  'Biomed. Chromatogr.': 'Biomedical Chromatography',
  'Biomed. Mater. (Bristol, U. K.)': 'Biomedical Materials (Bristol, United Kingdom)',
  'Biomed. Res.': 'Biomedical Research',
  'Biomed. Khim.': 'Biomeditsinskaya Khimiya',
  'Biomol. Concepts': 'Biomolecular Concepts',
  'Biomol. NMR Assignments': 'Biomolecular NMR Assignments',
  'Bioorg. Med. Chem.': 'Bioorganic &amp; Medicinal Chemistry',
  'Bioorg. Med. Chem. Lett.': 'Bioorganic &amp; Medicinal Chemistry Letters',
  'Bioorg. Chem.': 'Bioorganic Chemistry',
  'Biopharm. Drug Dispos.': 'Biopharmaceutics &amp; Drug Disposition',
  'Biophys. Chem.': 'Biophysical Chemistry',
  'Biophys. J.': 'Biophysical Journal',
  'Biophysics (Engl. Transl.)': 'Biophysics (English Translation)',
  'Bioprocess Biosyst. Eng.': 'Bioprocess and Biosystems Engineering',
  'Bioresour. Technol.': 'Bioresource Technology',
  'Biosci. Rep.': 'Bioscience Reports',
  'Biosci., Biotechnol., Biochem.': 'Bioscience, Biotechnology, and Biochemistry',
  'Biosens. Bioelectron.': 'Biosensors &amp; Bioelectronics',
  'Biotechnol. J.': 'Biotechnology Journal',
  'Biotechnol. Lett.': 'Biotechnology Letters',
  'Biotechnol. Prog.': 'Biotechnology Progress',
  'Biotechnol. Appl. Biochem.': 'Biotechnology and Applied Biochemistry',
  'Biotechnol. Bioeng.': 'Biotechnology and Bioengineering',
  'Biotechnol. Bioprocess Eng.': 'Biotechnology and Bioprocess Engineering',
  'Birth Defects Res., Part A': 'Birth Defects Research, Part A: Clinical and Molecular Teratology',
  'Birth Defects Res., Part B': 'Birth Defects Research, Part B: Developmental and Reproductive Toxicology',
  'Birth Defects Res., Part C': 'Birth Defects Research, Part C: Embryo Today--Reviews',
  'Bone Res.': 'Bone Research',
  'BrJAC--Braz. J. Anal. Chem.': 'BrJAC--Brazilian Journal of Analytical Chemistry',
  'Brain Res.': 'Brain Research',
  'Breast Cancer Manage.': 'Breast Cancer Management',
  'Br. J. Nutr.': 'British Journal of Nutrition',
  'Br. J. Pharmacol.': 'British Journal of Pharmacology',
  'Bull. Environ. Contam. Toxicol.': 'Bulletin of Environmental Contamination and Toxicology',
  'Bull. Mater. Sci.': 'Bulletin of Materials Science',
  'Bull. Chem. Soc. Jpn.': 'Bulletin of the Chemical Society of Japan',
  'Bull. Korean Chem. Soc.': 'Bulletin of the Korean Chemical Society',
  'Bull. Russ. Acad. Sci.: Phys.': 'Bulletin of the Russian Academy of Sciences: Physics',
  'CNS Neurol. Disord.: Drug Targets': 'CNS &amp; Neurological Disorders: Drug Targets',
  'CNS Oncol.': 'CNS Oncology',
  'CPT: Pharmacometrics Syst. Pharmacol.': 'CPT: Pharmacometrics &amp; Systems Pharmacology',
  'Calcif. Tissue Int.': 'Calcified Tissue International',
  'Can. J. Chem. Eng.': 'Canadian Journal of Chemical Engineering',
  'Can. J. Chem.': 'Canadian Journal of Chemistry',
  'Can. J. Microbiol.': 'Canadian Journal of Microbiology',
  'Can. J. Phys.': 'Canadian Journal of Physics',
  'Can. J. Physiol. Pharmacol.': 'Canadian Journal of Physiology and Pharmacology',
  'Can. Metall. Q.': 'Canadian Metallurgical Quarterly',
  'Can. Mineral.': 'Canadian Mineralogist',
  'Cancer Genomics Proteomics': 'Cancer Genomics &amp; Proteomics',
  'Cancer Lett. (N. Y., NY, U. S.)': 'Cancer Letters (New York, NY, United States)',
  'Cancer Res.': 'Cancer Research',
  'Cancer Sci.': 'Cancer Science',
  'Carbohydr. Polym.': 'Carbohydrate Polymers',
  'Carbohydr. Res.': 'Carbohydrate Research',
  'Carbon Manage.': 'Carbon Management',
  'Cardiovasc. Hematol. Agents Med. Chem.': 'Cardiovascular &amp; Hematological Agents in Medicinal Chemistry',
  'Cardiovasc. Hematol. Disord.: Drug Targets': 'Cardiovascular &amp; Hematological Disorders: Drug Targets',
  'Cardiovasc. Res.': 'Cardiovascular Research',
  'Catal. Commun.': 'Catalysis Communications',
  'Catal. Lett.': 'Catalysis Letters',
  'Catal. Rev.: Sci. Eng.': 'Catalysis Reviews: Science and Engineering',
  'Catal. Sci. Technol.': 'Catalysis Science &amp; Technology',
  'Catal. Today': 'Catalysis Today',
  'Cell (Cambridge, MA, U. S.)': 'Cell (Cambridge, MA, United States)',
  'Cell Biochem. Biophys.': 'Cell Biochemistry and Biophysics',
  'Cell Biochem. Funct.': 'Cell Biochemistry and Function',
  'Cell Host Microbe': 'Cell Host &amp; Microbe',
  'Cell Metab.': 'Cell Metabolism',
  'Cell Rep.': 'Cell Reports',
  'Cell. Immunol.': 'Cellular Immunology',
  'Cell. Microbiol.': 'Cellular Microbiology',
  'Cell. Physiol. Biochem.': 'Cellular Physiology and Biochemistry',
  'Cell. Polym.': 'Cellular Polymers',
  'Cell. Reprogram.': 'Cellular Reprogramming',
  'Cell. Mol. Life Sci.': 'Cellular and Molecular Life Sciences',
  'Cellul. Chem. Technol.': 'Cellulose Chemistry and Technology',
  'Cem. Concr. Compos.': 'Cement &amp; Concrete Composites',
  'Cem. Concr. Res.': 'Cement and Concrete Research',
  'Cent. Nerv. Syst. Agents Med. Chem.': 'Central Nervous System Agents in Medicinal Chemistry',
  'Ceram. Int.': 'Ceramics International',
  'Ceram.-Silik.': 'Ceramics-Silikaty',
  'Cereal Chem.': 'Cereal Chemistry',
  'Ceska Slov. Farm.': 'Ceska a Slovenska Farmacie',
  'Chem. Sin.': 'Chemica Sinica',
  'Chem. Pharm. Bull.': 'Chemical &amp; Pharmaceutical Bulletin',
  'Chem. Biol. Drug Des.': 'Chemical Biology &amp; Drug Design',
  'Chem. Commun. (Cambridge, U. K.)': 'Chemical Communications (Cambridge, United Kingdom)',
  'Chem. Eng. Technol.': 'Chemical Engineering &amp; Technology',
  'Chem. Eng. Commun.': 'Chemical Engineering Communications',
  'Chem. Eng. J. (Amsterdam, Neth.)': 'Chemical Engineering Journal (Amsterdam, Netherlands)',
  'Chem. Eng. Res. Des.': 'Chemical Engineering Research and Design',
  'Chem. Eng. Sci.': 'Chemical Engineering Research and Design',
  'Chem. Eng. Process.': 'Chemical Engineering and Processing',
  'Chem. Geol.': 'Chemical Geology',
  'Chem. Pap.': 'Chemical Papers',
  'Chem. Phys.': 'Chemical Physics',
  'Chem. Phys. Lett.': 'Chemical Physics Letters',
  'Chem. Rec.': 'Chemical Record',
  'Chem. Res. Chin. Univ.': 'Chemical Research in Chinese Universities',
  'Chem. Res. Toxicol.': 'Chemical Research in Toxicology',
  'Chem. Rev. (Washington, DC, U. S.)': 'Chemical Reviews (Washington, DC, United States)',
  'Chem. Sci.': 'Chemical Science',
  'Chem. Sci. Trans.': 'Chemical Science Transactions',
  'Chem. Speciation Bioavailability': 'Chemical Speciation and Bioavailability',
  'Chem. Biochem. Eng. Q.': 'Chemical and Biochemical Engineering Quarterly',
  'Chem. Listy': 'Chemicke Listy',
  'Chem.-Biol. Interact.': 'Chemico-Biological Interactions',
  'Chem. Ing. Tech.': 'Chemie Ingenieur Technik',
  'Chem. Erde': 'Chemie der Erde',
  'Chem. Technol. (Kaunas, Lith.)': 'Chemine Technologija (Kaunas, Lithuania)',
  'Chem. Biodiversity': 'Chemistry &amp; Biodiversity',
  'Chem. Biol. (Oxford, U. K.)': 'Chemistry &amp; Biology (Oxford, United Kingdom)',
  'Chem. Biol. Interface': 'Chemistry &amp; Biology Interface',
  'Chem. Ind. (Chichester, U. K.)': 'Chemistry &amp; Industry (Chichester, United Kingdom)',
  'Chem. J. Mold.': 'Chemistry Journal of Moldova',
  'Chem. Lett.': 'Chemistry Letters',
  'Chem. Phys. Lipids': 'Chemistry and Physics of Lipids',
  'Chem. Technol. Fuels Oils': 'Chemistry and Technology of Fuels and Oils',
  'Chem. Heterocycl. Compd. (N. Y., NY, U. S.)': 'Chemistry of Heterocyclic Compounds (New York, NY, United States)',
  'Chem. Mater.': 'Chemistry of Materials',
  'Chem. Nat. Compd.': 'Chemistry of Natural Compounds',
  'Chemom. Intell. Lab. Syst.': 'Chemometrics and Intelligent Laboratory Systems',
  'Chemotherapy (Basel, Switz.)': 'Chemotherapy (Basel, Switzerland)',
  'Chim. Ind. (Milan, Italy)': "Chimica e l'Industria (Milan, Italy)",
  'Chin. Chem. Lett.': 'Chinese Chemical Letters',
  'Chin. J. Chem. Eng.': 'Chinese Journal of Chemical Engineering',
  'Chin. J. Chem. Phys.': 'Chinese Journal of Chemical Physics',
  'Chin. J. Chem.': 'Chinese Journal of Chemistry',
  'Chin. J. Geochem.': 'Chinese Journal of Geochemistry',
  'Chin. J. Polym. Sci.': 'Chinese Journal of Polymer Science',
  'Chin. J. Struct. Chem.': 'Chinese Journal of Structural Chemistry',
  'Chin. Phys. C': 'Chinese Physics C',
  'Chromatogr. Res. Int.': 'Chromatography Research International',
  'Chromosome Res.': 'Chromosome Research',
  'Circ. Res.': 'Circulation Research',
  'Clay Miner.': 'Clay Minerals',
  'Clays Clay Miner.': 'Clays and Clay Minerals',
  'Clin. Chim. Acta': 'Clinica Chimica Acta',
  'Clin. Chem. (Washington, DC, U. S.)': 'Clinical Chemistry (Washington, DC, United States)',
  'Clin. Chem. Lab. Med.': 'Clinical Chemistry and Laboratory Medicine',
  'Clin. Immunol. (Amsterdam, Neth.)': 'Clinical Immunology (Amsterdam, Netherlands)',
  'Clin. Proteomics': 'Clinical Proteomics',
  'Clin. Exp. Pharmacol. Physiol.': 'Clinical and Experimental Pharmacology and Physiology',
  'Clin. Vaccine Immunol.': 'Clinical and Vaccine Immunology',
  'Colloid J.': 'Colloid Journal',
  'Colloid Polym. Sci.': 'Colloid and Polymer Science',
  'Colloids Surf., A': 'Colloids and Surfaces, A: Physicochemical and Engineering Aspects',
  'Colloids Surf., B': 'Colloids and Surfaces, B: Biointerfaces',
  'Color. Technol.': 'Coloration Technology',
  'Comb. Chem. High Throughput Screening': 'Combinatorial Chemistry &amp; High Throughput Screening',
  'Combust. Sci. Technol.': 'Combustion Science and Technology',
  'Combust. Flame': 'Combustion and Flame',
  'Comments Inorg. Chem.': 'Comments on Inorganic Chemistry',
  'Commun. Soil Sci. Plant Anal.': 'Communications in Soil Science and Plant Analysis',
  'Comp. Biochem. Physiol., Part A: Mol. Integr. Physiol.': 'Comparative Biochemistry and Physiology, Part A: Molecular &amp; Integrative Physiology',
  'Comp. Biochem. Physiol., Part B: Biochem. Mol. Biol.': 'Comparative Biochemistry and Physiology, Part B: Biochemistry &amp; Molecular Biology',
  'Comp. Biochem. Physiol., Part C: Toxicol. Pharmacol.': 'Comparative Biochemistry and Physiology, Part C: Toxicology &amp; Pharmacology',
  'Comp. Biochem. Physiol., Part D: Genomics Proteomics': 'Comparative Biochemistry and Physiology, Part D: Genomics &amp; Proteomics',
  'Comp. Med.': 'Comparative Medicine',
  'Complex Met.': 'Complex Metals',
  'C. R. Chim.': 'Comptes Rendus Chimie',
  'C. R. Phys.': 'Comptes Rendus Physique',
  'Comput. Theor. Chem.': 'Computational &amp; Theoretical Chemistry',
  'Comput. Biol. Chem.': 'Computational Biology and Chemistry',
  'Comput. Phys. Commun.': 'Computer Physics Communications',
  'Comput. Chem. Eng.': 'Computers &amp; Chemical Engineering',
  'Contrast Media Mol. Imaging': 'Contrast Media &amp; Molecular Imaging',
  'Environ. Technol.': 'Environmental Technology',
  'Environ. Toxicol.': 'Environmental Toxicology',
  'Environ. Toxicol. Chem.': 'Environmental Toxicology and Chemistry',
  'Expert Rev. Vaccines': 'Expert Review of Vaccines',
  'FASEB J.': 'FASEB Journal',
  'FEBS J.': 'FEBS Journal',
  'FEBS Lett.': 'FEBS Letters',
  'Facial Plast. Surg.': 'Facial Plastic Surgery',
  'Faraday Discuss.': 'Faraday Discussions',
  'Ferroelectr., Lett. Sect.': 'Ferroelectrics, Letters Section',
  'Fibre Chem.': 'Fibre Chemistry',
  'Fire Mater.': 'Fire and Materials',
  'Fish Physiol. Biochem.': 'Fish Physiology and Biochemistry',
  'Fusion Eng. Des.': 'Fusion Engineering and Design',
  'Gene Ther.': 'Gene Therapy',
  'Gen. Comp. Endocrinol.': 'General and Comparative Endocrinology',
  'Genes Dev.': 'Genes &amp; Development',
  'Genes Cells': 'Genes to Cells',
  'Genes, Chromosomes Cancer': 'Genes, Chromosomes &amp; Cancer',
  'Hum. Gene Ther.': 'Human Gene Therapy',
  'Mol. Med. (Manhasset, NY, U. S.)': 'Molecular Medicine (Manhasset, NY, United States)',
  'Mol. Med. Rep.': 'Molecular Medicine Reports',
  'Mol. Membr. Biol.': 'Molecular Membrane Biology',
  'Mol. Metab.': 'Molecular Metabolism',
  'Mol. Microbiol.': 'Molecular Microbiology',
  'Mol. Nutr. Food Res.': 'Molecular Nutrition &amp; Food Research',
  'Mol. Pharmaceutics': 'Molecular Pharmaceutics',
  'Mol. Pharmacol.': 'Molecular Pharmacology',
  'Mol. Phys.': 'Molecular Physics',
  'Mol. Plant': 'Molecular Plant',
  'Mol. Plant-Microbe Interact.': 'Molecular Plant-Microbe Interactions',
  'Mol. Reprod. Dev.': 'Molecular Reproduction &amp; Development',
  'Mol. Simul.': 'Molecular Simulation',
  'Mol. Ther.': 'Molecular Therapy',
  'Mol. Ther.--Methods Clin. Dev.': 'Molecular Therapy--Methods &amp; Clinical Development',
  'Mol. Ther.--Nucleic Acids': 'Molecular Therapy--Nucleic Acids',
  'Mol. Cell. Biochem.': 'Molecular and Cellular Biochemistry',
  'Mol. Cell. Biol.': 'Molecular and Cellular Biology',
  'Mol. Cell. Endocrinol.': 'Molecular and Cellular Endocrinology',
  'Mol. Cell. Neurosci.': 'Molecular and Cellular Neuroscience',
  'Mol. Cell. Probes': 'Molecular and Cellular Probes',
  'Mol. Cells': 'Molecules and Cells',
  'Monatsh. Chem.': 'Monatshefte fuer Chemie',
  'Mucosal Immunol.': 'Mucosal Immunology',
  'Mutat. Res., Fundam. Mol. Mech. Mutagen.': 'Mutation Research, Fundamental and Molecular Mechanisms of Mutagenesis',
  'Mutat. Res., Genet. Toxicol. Environ. Mutagen.': 'Mutation Research, Genetic Toxicology and Environmental Mutagenesis',
  'Mutat. Res., Rev. Mutat. Res.': 'Mutation Research, Reviews in Mutation Research',
  'Nano Lett.': 'Nano Letters',
  'Nano Sci. Nano Technol.: Indian J.': 'Nano Science and Nano Technology: An Indian Journal',
  'Nanomedicine (London, U. K.)': 'Nanomedicine (London, United Kingdom)',
  'Nanomedicine (N. Y., NY, U. S.)': 'Nanomedicine (New York, NY, United States)',
  'Nat. Prod. Rep.': 'Natural Product Reports',
  'Nat. Prod. Res.': 'Natural Product Research',
  'Nat. Prod. Sci.': 'Natural Product Sciences',
  'Nat. Prod. J.': 'Natural Products Journal',
  'Nature (London, U. K.)': 'Nature (London, United Kingdom)',
  'Nat. Biotechnol.': 'Nature Biotechnology',
  'Nat. Cell Biol.': 'Nature Cell Biology',
  'Nat. Chem. Biol.': 'Nature Chemical Biology',
  'Nat. Chem.': 'Nature Chemistry',
  'Nat. Commun.': 'Nature Communications',
  'Nat. Genet.': 'Nature Genetics',
  'Nat. Geosci.': 'Nature Geoscience',
  'Nat. Immunol.': 'Nature Immunology',
  'Nat. Mater.': 'Nature Materials',
  'Nat. Med. (N. Y., NY, U. S.)': 'Nature Medicine (New York, NY, United States)',
  'Nat. Methods': 'Nature Methods',
  'Nat. Nanotechnol.': 'Nature Nanotechnology',
  'Nat. Neurosci.': 'Nature Neuroscience',
  'Nat. Photonics': 'Nature Photonics',
  'Nat. Phys.': 'Nature Physics',
  'Nat. Protoc.': 'Nature Protocols',
  'Nat. Rev. Cancer': 'Nature Reviews Cancer',
  'Nat. Rev. Drug Discovery': 'Nature Reviews Drug Discovery',
  'Nat. Rev. Genet.': 'Nature Reviews Genetics',
  'Nat. Rev. Immunol.': 'Nature Reviews Immunology',
  'Nat. Rev. Microbiol.': 'Nature Reviews Microbiology',
  'Nat. Rev. Mol. Cell Biol.': 'Nature Reviews Molecular Cell Biology',
  'Nat. Rev. Neurosci.': 'Nature Reviews Neuroscience',
  'Nat. Struct. Mol. Biol.': 'Nature Structural &amp; Molecular Biology',
  "Naunyn-Schmiedeberg's Arch. Pharmacol.": "Naunyn-Schmiedeberg's Archives of Pharmacology",
  'Neues Jahrb. Mineral., Abh.': 'Neues Jahrbuch fuer Mineralogie, Abhandlungen',
  'NeuroMol. Med.': 'NeuroMolecular Medicine',
  'Neurochem. Res.': 'Neurochemical Research',
  'Neurochem. Int.': 'Neurochemistry International',
  'Neuropeptides (Oxford, U. K.)': 'Neuropeptides (Oxford, United Kingdom)',
  'Neuroscience (Amsterdam, Neth.)': 'Neuroscience (Amsterdam, Netherlands)',
  'Neurosci. Lett.': 'Neuroscience Letters',
  'Neurotoxicol. Teratol.': 'Neurotoxicology and Teratology',
  'New Biotechnol.': 'New Biotechnology',
  'N. Engl. J. Med.': 'New England Journal of Medicine',
  'New J. Chem.': 'New Journal of Chemistry',
  'Nord. Pulp Pap. Res. J.': 'Nordic Pulp &amp; Paper Research Journal',
  'Nucl. Eng. Des.': 'Nuclear Engineering and Design',
  'Nucl. Fusion': 'Nuclear Fusion',
  'Nucl. Instrum. Methods Phys. Res., Sect. A': 'Nuclear Instruments &amp; Methods in Physics Research, Section A: Accelerators, Spectrometers, Detectors, and Associated Equipment',
  'Nucl. Instrum. Methods Phys. Res., Sect. B': 'Nuclear Instruments &amp; Methods in Physics Research, Section B: Beam Interactions with Materials and Atoms',
  'Nucl. Med. Biol.': 'Nuclear Medicine and Biology',
  'Nucl. Phys. A': 'Nuclear Physics A',
  'Nucl. Phys. B': 'Nuclear Physics B',
  'Nucleic Acid Ther.': 'Nucleic Acid Therapeutics',
  'Nucleic Acids Res.': 'Nucleic Acids Research',
  'Nucleosides, Nucleotides Nucleic Acids': 'Nucleosides, Nucleotides &amp; Nucleic Acids',
  'Nutr. Res. (N. Y., NY, U. S.)': 'Nutrition Research (New York, NY, United States)',
  'Open J. Inorg. Non-Met. Mater.': 'Open Journal of Inorganic Non-Metallic Materials',
  'Open J. Med. Chem.': 'Open Journal of Medicinal Chemistry',
  'Open J. Org. Polym. Mater.': 'Open Journal of Organic Polymer Materials',
  'Open J. Polym. Chem.': 'Open Journal of Polymer Chemistry',
  'Open J. Synth. Theory Appl.': 'Open Journal of Synthesis Theory and Applications',
  'Opt. Commun.': 'Optics Communications',
  'Opt. Spectrosc.': 'Optics and Spectroscopy',
  'Org. Biomol. Chem.': 'Organic &amp; Biomolecular Chemistry',
  'Org. Chem. Front.': 'Organic Chemistry Frontiers',
  'Org. Chem. Int.': 'Organic Chemistry International',
  'Org. Electron.': 'Organic Electronics',
  'Org. Geochem.': 'Organic Geochemistry',
  'Org. Lett.': 'Organic Letters',
  'Org. Prep. Proced. Int.': 'Organic Preparations and Procedures International',
  'Org. Process Res. Dev.': 'Organic Process Research &amp; Development',
  'Org. Synth.': 'Organic Syntheses',
  'Org. Med. Chem. Lett.': 'Organic and Medicinal Chemistry Letters',
  'Orient. J. Chem.': 'Oriental Journal of Chemistry',
  'Oxid. Commun.': 'Oxidation Communications',
  'Oxid. Met.': 'Oxidation of Metals',
  'Ozone: Sci. Eng.': 'Ozone: Science &amp; Engineering',
  'PLoS Biol.': 'PLoS Biology',
  'PLoS Comput. Biol.': 'PLoS Computational Biology',
  'PLoS Genet.': 'PLoS Genetics',
  'PLoS Pathog.': 'PLoS Pathogens',
  'PMSE Prepr.': 'PMSE Preprints',
  'Part. Sci. Technol.': 'Particulate Science and Technology',
  'Peptides (N. Y., NY, U. S.)': 'Peptides (New York, NY, United States)',
  'Pers. Med.': 'Personalized Medicine',
  'Pest Manage. Sci.': 'Pest Management Science',
  'Pestic. Biochem. Physiol.': 'Pesticide Biochemistry and Physiology',
  'Pet. Chem.': 'Petroleum Chemistry',
  'Pet. Sci. Technol.': 'Petroleum Science and Technology',
  'Pfluegers Arch.': 'Pfluegers Archiv',
  'Pharma Chem.': 'Pharma Chemica',
  'Pharm. Chem. J.': 'Pharmaceutical Chemistry Journal',
  'Pharm. Dev. Technol.': 'Pharmaceutical Development and Technology',
  'Pharm. Nanotechnol.': 'Pharmaceutical Nanotechnology',
  'Pharm. Pat. Anal.': 'Pharmaceutical Patent Analyst',
  'Pharm. Res.': 'Pharmaceutical Research',
  'Pharm. Sin.': 'Pharmacia Sinica',
  'Pharmacogenet. Genomics': 'Pharmacogenetics and Genomics',
  'Pharmacogenomics J.': 'Pharmacogenomics Journal',
  'Pharmacol. Ther.': 'Pharmacology &amp; Therapeutics',
  'Pharmacol., Biochem. Behav.': 'Pharmacology, Biochemistry and Behavior',
  'Philos. Mag.': 'Philosophical Magazine',
  'Philos. Mag. Lett.': 'Philosophical Magazine Letters',
  'Phosphorus, Sulfur Silicon Relat. Elem.': 'Phosphorus, Sulfur and Silicon and the Related Elements',
  'Photochem. Photobiol. Sci.': 'Photochemical &amp; Photobiological Sciences',
  'Photochem. Photobiol.': 'Photochemistry and Photobiology',
  'Photosynth. Res.': 'Photosynthesis Research',
  'Phys. B (Amsterdam, Neth.)': 'Physica B: Condensed Matter (Amsterdam, Netherlands)',
  'Phys. C (Amsterdam, Neth.)': 'Physica C: Superconductivity and Its Applications (Amsterdam, Netherlands)',
  'Phys. Scr.': 'Physica Scripta',
  'Phys. Scr., T': 'Physica Scripta, T',
  'Phys. Status Solidi A': 'Physica Status Solidi A: Applications and Materials Science',
  'Phys. Status Solidi B': 'Physica Status Solidi B: Basic Solid State Physics',
  'Phys. Status Solidi C': 'Physica Status Solidi C: Current Topics in Solid State Physics',
  'Phys. Status Solidi RRL': 'Physica Status Solidi RRL: Rapid Research Letters',
  'Phys. Chem. Chem. Phys.': 'Physical Chemistry Chemical Physics',
  'Phys. Rev. A: At., Mol., Opt. Phys.': 'Physical Review A: Atomic, Molecular, and Optical Physics',
  'Phys. Rev. Appl.': 'Physical Review Applied',
  'Phys. Rev. B: Condens. Matter Mater. Phys.': 'Physical Review B: Condensed Matter and Materials Physics',
  'Phys. Rev. C: Nucl. Phys.': 'Physical Review C: Nuclear Physics',
  'Phys. Rev. D: Part., Fields, Gravitation, Cosmol.': 'Physical Review D: Particles, Fields, Gravitation, and Cosmology',
  'Phys. Rev. E: Stat., Nonlinear, Soft Matter Phys.': 'Physical Review E: Statistical, Nonlinear, and Soft Matter Physics',
  'Phys. Rev. Lett.': 'Physical Review Letters',
  'Phys. Rev. X': 'Physical Review X',
  'Phys. Lett. A': 'Physics Letters A',
  'Phys. Lett. B': 'Physics Letters B',
  'Phys. Chem. Liq.': 'Physics and Chemistry of Liquids',
  'Phys. Chem. Miner.': 'Physics and Chemistry of Minerals',
  'Phys. At. Nucl.': 'Physics of Atomic Nuclei',
  'Phys. Fluids': 'Physics of Fluids',
  'Phys. Plasmas': 'Physics of Plasmas',
  'Phys. Solid State': 'Physics of the Solid State',
  'Physiol. Plant.': 'Physiologia Plantarum',
  'Physiol. Behav.': 'Physiology &amp; Behavior',
  'Phytochem. Anal.': 'Phytochemical Analysis',
  'Phytochem. Rev.': 'Phytochemistry Reviews',
  'Plant Cell Rep.': 'Plant Cell Reports',
  'Plant Growth Regul.': 'Plant Growth Regulation',
  'Plant J.': 'Plant Journal',
  'Plant Mol. Biol.': 'Plant Molecular Biology',
  'Plant Physiol.': 'Plant Physiology',
  'Plant Physiol. Biochem. (Issy-les-Moulineaux, Fr.)': 'Plant Physiology and Biochemistry (Issy-les-Moulineaux, France)',
  'Plant Sci. (Shannon, Irel.)': 'Plant Science (Shannon, Ireland)',
  'Plant Cell Physiol.': 'Plant and Cell Physiology',
  'Plant Soil': 'Plant and Soil',
  'Planta Med.': 'Planta Medica',
  'Plasma Chem. Plasma Process.': 'Plasma Chemistry and Plasma Processing',
  'Plasma Processes Polym.': 'Plasma Processes and Polymers',
  'Plast. Reconstr. Surg.': 'Plastic and Reconstructive Surgery',
  'Plast. Massy': 'Plasticheskie Massy',
  'Plast., Rubber Compos.': 'Plastics, Rubber and Composites',
  'Pol. J. Chem. Technol.': 'Polish Journal of Chemical Technology',
  'Polycyclic Aromat. Compd.': 'Polycyclic Aromatic Compounds',
  'Polym. Bull. (Heidelberg, Ger.)': 'Polymer Bulletin (Heidelberg, Germany)',
  'Polym. Chem.': 'Polymer Chemistry',
  'Polym. Compos.': 'Polymer Composites',
  'Polym. Degrad. Stab.': 'Polymer Degradation and Stability',
  'Polym. Eng. Sci.': 'Polymer Engineering &amp; Science',
  'Polym. Int.': 'Polymer International',
  'Polym. J. (Tokyo, Jpn.)': 'Polymer Journal (Tokyo, Japan)',
  'Polym. Prepr. (Am. Chem. Soc., Div. Polym. Chem.)': 'Polymer Preprints (American Chemical Society, Division of Polymer Chemistry)',
  'Polym. Rev. (Philadelphia, PA, U. S.)': 'Polymer Reviews (Philadelphia, PA, United States)',
  'Polym. Sci., Ser. A': 'Polymer Science, Series A',
  'Polym. Sci., Ser. B': 'Polymer Science, Series B',
  'Polym. Sci., Ser. C': 'Polymer Science, Series C',
  'Polym. Sci., Ser. D': 'Polymer Science, Series D',
  'Polym. Test.': 'Polymer Testing',
  'Polym.-Plast. Technol. Eng.': 'Polymer-Plastics Technology and Engineering',
  'Polym. Polym. Compos.': 'Polymers &amp; Polymer Composites',
  'Polymers (Basel, Switz.)': 'Polymers (Basel, Switzerland)',
  'Polym. Adv. Technol.': 'Polymers for Advanced Technologies',
  'Powder Diffr.': 'Powder Diffraction',
  'Powder Metall.': 'Powder Metallurgy',
  'Powder Metall. Met. Ceram.': 'Powder Metallurgy and Metal Ceramics',
  'Powder Technol.': 'Powder Technology',
  'Pract. Spectrosc.': 'Practical Spectroscopy',
  'Precambrian Res.': 'Precambrian Research',
  'Prep. Biochem. Biotechnol.': 'Preparative Biochemistry &amp; Biotechnology',
  'Procedia Mater. Sci.': 'Procedia Materials Science',
  'Proc. Natl. Acad. Sci. U. S. A.': 'Proceedings of the National Academy of Sciences of the United States of America',
  'Process Biochem. (Oxford, U. K.)': 'Process Biochemistry (Oxford, United Kingdom)',
  'Process Saf. Environ. Prot.': 'Process Safety and Environmental Protection',
  'Prog. Colloid Polym. Sci.': 'Progress in Colloid and Polymer Science',
  'Prog. Cryst. Growth Charact. Mater.': 'Progress in Crystal Growth and Characterization of Materials',
  'Prog. Inorg. Chem.': 'Progress in Inorganic Chemistry',
  'Prog. Nucl. Magn. Reson. Spectrosc.': 'Progress in Nuclear Magnetic Resonance Spectroscopy',
  'Prog. Org. Coat.': 'Progress in Organic Coatings',
  'Prog. Polym. Sci.': 'Progress in Polymer Science',
  'Prog. React. Kinet. Mech.': 'Progress in Reaction Kinetics and Mechanism',
  'Prog. Solid State Chem.': 'Progress in Solid State Chemistry',
  'Prog. Surf. Sci.': 'Progress in Surface Science',
  'Propellants, Explos., Pyrotech.': 'Propellants, Explosives, Pyrotechnics',
  'Prostaglandins Other Lipid Mediators': 'Prostaglandins and Other Lipid Mediators',
  'Prostaglandins, Leukotrienes Essent. Fatty Acids': 'Prostaglandins, Leukotrienes and Essential Fatty Acids',
  'Prot. Met. Phys. Chem. Surf.': 'Protection of Metals and Physical Chemistry of Surfaces',
  'Protein Pept. Lett.': 'Protein &amp; Peptide Letters',
  'Protein Eng., Des. Sel.': 'Protein Engineering, Design &amp; Selection',
  'Protein Expression Purif.': 'Protein Expression and Purification',
  'Protein J.': 'Protein Journal',
  'Protein Sci.': 'Protein Science',
  'Proteins: Struct., Funct., Bioinf.': 'Proteins: Structure, Function, and Bioinformatics',
  'Proteomics: Clin. Appl.': 'Proteomics: Clinical Applications',
  'Przem. Chem.': 'Przemysl Chemiczny',
  'Psychopharmacology (Heidelberg, Ger.)': 'Psychopharmacology (Heidelberg, Germany)',
  'Pulm. Pharmacol. Ther.': 'Pulmonary Pharmacology &amp; Therapeutics',
  'Pure Appl. Chem.': 'Pure and Applied Chemistry',
  'Quantum Electron.': 'Quantum Electronics',
  'Quim. Nova': 'Quimica Nova',
  'RSC Adv.': 'RSC Advances',
  'Radiat. Eff. Defects Solids': 'Radiation Effects and Defects in Solids',
  'Radiat. Meas.': 'Radiation Measurements',
  'Radiat. Phys. Chem.': 'Radiation Physics and Chemistry',
  'Radiat. Res.': 'Radiation Research',
  'Radiochemistry (Moscow, Russ. Fed.)': 'Radiochemistry (Moscow, Russian Federation)',
  'Radiochim. Acta': 'Radiochimica Acta',
  'Rapid Commun. Mass Spectrom.': 'Rapid Communications in Mass Spectrometry',
  'Rare Met. (Beijing, China)': 'Rare Metals (Beijing, China)',
  'React. Kinet., Mech. Catal.': 'Reaction Kinetics, Mechanisms and Catalysis',
  'React. Funct. Polym.': 'Reactive &amp; Functional Polymers',
  'Recent Adv. Cardiovasc. Drug Discovery': 'Recent Advances in Cardiovascular Drug Discovery',
  'Recent Adv. DNA Gene Sequences': 'Recent Advances in DNA &amp; Gene Sequences',
  'Recent Pat. Anti-Cancer Drug Discovery': 'Recent Patents on Anti-Cancer Drug Discovery',
  'Recent Pat. Anti-Infect. Drug Discovery': 'Recent Patents on Anti-Infective Drug Discovery',
  'Recent Pat. CNS Drug Discovery': 'Recent Patents on CNS Drug Discovery',
  'Recent Pat. Drug Delivery Formulation': 'Recent Patents on Drug Delivery &amp; Formulation',
  'Recent Pat. Endocr., Metab. Immune Drug Discovery': 'Recent Patents on Endocrine, Metabolic &amp; Immune Drug Discovery',
  'Recent Pat. Inflammation Allergy Drug Discovery': 'Recent Patents on Inflammation &amp; Allergy Drug Discovery',
  'Recent Pat. Nanotechnol.': 'Recent Patents on Nanotechnology',
  'Redox Biol.': 'Redox Biology',
  'Regener. Med.': 'Regenerative Medicine',
  'Reproduction (Bristol, U. K.)': 'Reproduction (Bristol, United Kingdom)',
  'Reprod. Toxicol.': 'Reproductive Toxicology',
  'Res. Chem. Intermed.': 'Research on Chemical Intermediates',
  'Rev. Sci. Instrum.': 'Review of Scientific Instruments',
  'Rev. Anal. Chem.': 'Reviews in Analytical Chemistry',
  'Rev. Inorg. Chem.': 'Reviews in Inorganic Chemistry',
  'Rev. Mex. Ing. Quim.': 'Revista Mexicana de Ingenieria Quimica',
  'Rev. Peru. Quim. Ing. Quim.': 'Revista Peruana de Quimica e Ingenieria Quimica',
  'Rev. Rom. Mater.': 'Revista Romana de Materiale',
  'Rev. Chim. (Bucharest, Rom.)': 'Revista de Chimie (Bucharest, Romania)',
  'Rev. Soc. Quim. Peru': 'Revista de la Sociedad Quimica del Peru',
  'Rheol. Acta': 'Rheologica Acta',
  'Rubber Chem. Technol.': 'Rubber Chemistry and Technology',
  'Russ. Chem. Bull.': 'Russian Chemical Bulletin',
  'Russ. Chem. Rev.': 'Russian Chemical Reviews',
  'Russ. J. Appl. Chem.': 'Russian Journal of Applied Chemistry',
  'Russ. J. Bioorg. Chem.': 'Russian Journal of Bioorganic Chemistry',
  'Russ. J. Coord. Chem.': 'Russian Journal of Coordination Chemistry',
  'Russ. J. Electrochem.': 'Russian Journal of Electrochemistry',
  'Russ. J. Gen. Chem.': 'Russian Journal of General Chemistry',
  'Russ. J. Genet.': 'Russian Journal of Genetics',
  'Russ. J. Inorg. Chem.': 'Russian Journal of Inorganic Chemistry',
  'Russ. J. Org. Chem.': 'Russian Journal of Organic Chemistry',
  'Russ. J. Phys. Chem. A': 'Russian Journal of Physical Chemistry A',
  'Russ. J. Phys. Chem. B': 'Russian Journal of Physical Chemistry B',
  'Russ. J. Plant Physiol.': 'Russian Journal of Plant Physiology',
  'SAR QSAR Environ. Res.': 'SAR and QSAR in Environmental Research',
  'SPE Prod. Oper.': 'SPE Production &amp; Operations',
  'SPE Reservoir Eval. Eng.': 'SPE Reservoir Evaluation &amp; Engineering',
  'Science (Washington, DC, U. S.)': 'Science (Washington, DC, United States)',
  'Sci. China: Chem.': 'Science China: Chemistry',
  'Sci. Lett.': 'Science Letters',
  'Sci. Synth., Knowl. Updates': 'Science of Synthesis, Knowledge Updates',
  'Sci. Total Environ.': 'Science of the Total Environment',
  'Sci. Pharm.': 'Scientia Pharmaceutica',
  'Sci. Rep.': 'Scientific Reports',
  'Scr. Mater.': 'Scripta Materialia',
  'Semicond. Sci. Technol.': 'Semiconductor Science and Technology',
  'Sens. Actuators, A': 'Sensors and Actuators, A: Physical',
  'Sens. Actuators, B': 'Sensors and Actuators, B: Chemical',
  'Sep. Sci. Technol. (Philadelphia, PA, U. S.)': 'Separation Science and Technology (Philadelphia, PA, United States)',
  'Sep. Purif. Technol.': 'Separation and Purification Technology',
  'Soil Biol. Biochem.': 'Soil Biology &amp; Biochemistry',
  'Soil Sci.': 'Soil Science',
  'Soil Sci. Soc. Am. J.': 'Soil Science Society of America Journal',
  'Sol. Energy Mater. Sol. Cells': 'Solar Energy Materials &amp; Solar Cells',
  'Solid Fuel Chem.': 'Solid Fuel Chemistry',
  'Solid State Commun.': 'Solid State Communications',
  'Solid State Nucl. Magn. Reson.': 'Solid State Nuclear Magnetic Resonance',
  'Solid State Sci.': 'Solid State Sciences',
  'Solid-State Electron.': 'Solid-State Electronics',
  'Solvent Extr. Res. Dev., Jpn.': 'Solvent Extraction Research and Development, Japan',
  'Solvent Extr. Ion Exch.': 'Solvent Extraction and Ion Exchange',
  'S. Afr. J. Chem.': 'South African Journal of Chemistry',
  'Spectrochim. Acta, Part A': 'Spectrochimica Acta, Part A: Molecular and Biomolecular Spectroscopy',
  'Spectrochim. Acta, Part B': 'Spectrochimica Acta, Part B: Atomic Spectroscopy',
  'Spectrosc. Lett.': 'Spectroscopy Letters',
  'Steel Res. Int.': 'Steel Research International',
  'Struct. Chem.': 'Structural Chemistry',
  'Struct. Dyn.': 'Structural Dynamics',
  'Structure (Oxford, U. K.)': 'Structure (Oxford, United Kingdom)',
  'Supercond. Sci. Technol.': 'Superconductor Science and Technology',
  'Superlattices Microstruct.': 'Superlattices and Microstructures',
  'Supramol. Chem.': 'Supramolecular Chemistry',
  'Surf. Eng.': 'Surface Engineering',
  'Surf. Rev. Lett.': 'Surface Review and Letters',
  'Surf. Sci.': 'Surface Science',
  'Surf. Sci. Rep.': 'Surface Science Reports',
  'Surf. Coat. Technol.': 'Surface and Coatings Technology',
  'Surf. Interface Anal.': 'Surface and Interface Analysis',
  'Sustainable Chem. Processes': 'Sustainable Chemical Processes',
  'Synapse (Hoboken, NJ, U. S.)': 'Synapse (Hoboken, NJ, United States)',
  'Synth. React. Inorg., Met.-Org., Nano-Met. Chem.': 'Synthesis and Reactivity in Inorganic, Metal-Organic, and Nano-Metal Chemistry',
  'Synth. Commun.': 'Synthetic Communications',
  'Synth. Met.': 'Synthetic Metals',
  'Tappi J.': 'Tappi Journal',
  'Tech. Phys.': 'Technical Physics',
  'Tech. Phys. Lett.': 'Technical Physics Letters',
  'Tenside, Surfactants, Deterg.': 'Tenside, Surfactants, Detergents',
  'Tetrahedron Lett.': 'Tetrahedron Letters',
  'Text. Res. J.': 'Textile Research Journal',
  'Theor. Chem. Acc.': 'Theoretical Chemistry Accounts',
  'Theor. Found. Chem. Eng.': 'Theoretical Foundations of Chemical Engineering',
  'Theor. Exp. Chem.': 'Theoretical and Experimental Chemistry',
  'Ther. Delivery': 'Therapeutic Delivery',
  'Thermochim. Acta': 'Thermochimica Acta',
  'Thromb. Res.': 'Thrombosis Research',
  'Tonkie Khim. Tekhnol.': 'Tonkie Khimicheskie Tekhnologii',
  'Top. Catal.': 'Topics in Catalysis',
  'Toxicol. Environ. Chem.': 'Toxicological &amp; Environmental Chemistry',
  'Toxicol. Sci.': 'Toxicological Sciences',
  'Toxicol. In Vitro': 'Toxicology In Vitro',
  'Toxicol. Lett.': 'Toxicology Letters',
  'Toxicol. Mech. Methods': 'Toxicology Mechanisms and Methods',
  'Toxicol. Res. (Cambridge, U. K.)': 'Toxicology Research (Cambridge, United Kingdom)',
  'Toxicol. Appl. Pharmacol.': 'Toxicology and Applied Pharmacology',
  'TrAC, Trends Anal. Chem.': 'TrAC, Trends in Analytical Chemistry',
  'Traffic (Oxford, U. K.)': 'Traffic (Oxford, United Kingdom)',
  'Trans. Indian Ceram. Soc.': 'Transactions of the Indian Ceramic Society',
  'Trans. Inst. Met. Finish.': 'Transactions of the Institute of Metal Finishing',
  'Transgenic Res.': 'Transgenic Research',
  'Transition Met. Chem. (Dordrecht, Neth.)': 'Transition Metal Chemistry (Dordrecht, Netherlands)',
  'Trends Biochem. Sci.': 'Trends in Biochemical Sciences',
  'Trends Biotechnol.': 'Trends in Biotechnology',
  'Trends Carbohydr. Res.': 'Trends in Carbohydrate Research',
  'Trends Endocrinol. Metab.': 'Trends in Endocrinology and Metabolism',
  'Trends Environ. Anal. Chem.': 'Trends in Environmental Analytical Chemistry',
  'Trends Food Sci. Technol.': 'Trends in Food Science &amp; Technology',
  'Trends Genet.': 'Trends in Genetics',
  'Trends Immunol.': 'Trends in Immunology',
  'Trends Microbiol.': 'Trends in Microbiology',
  'Trends Neurosci.': 'Trends in Neurosciences',
  'Trends Pharmacol. Sci.': 'Trends in Pharmacological Sciences',
  'Trends Plant Sci.': 'Trends in Plant Science',
  'Tribol. Trans.': 'Tribology Transactions',
  'Turk. J. Chem.': 'Turkish Journal of Chemistry',
  'Ukr. Biochem. J.': 'Ukrainian Biochemical Journal',
  'Ukr. Bioorg. Acta': 'Ukrainica Bioorganica Acta',
  'Ukr. Khim. Zh. (Russ. Ed.)': 'Ukrainskii Khimicheskii Zhurnal (Russian Edition)',
  'Ultrason. Sonochem.': 'Ultrasonics Sonochemistry',
  'Vasc. Pharmacol.': 'Vascular Pharmacology',
  'Vestsi Nats. Akad. Navuk Belarusi, Ser. Khim. Navuk': "Vestsi Natsyyanal'nai Akademii Navuk Belarusi, Seryya Khimichnykh Navuk",
  'Vib. Spectrosc.': 'Vibrational Spectroscopy',
  'Viral Immunol.': 'Viral Immunology',
  'Virus Res.': 'Virus Research',
  'Water Environ. Res.': 'Water Environment Research',
  'Water Qual. Res. J. Can.': 'Water Quality Research Journal of Canada',
  'Water Res.': 'Water Research',
  'Water Resour.': 'Water Resources',
  'Water, Air, Soil Pollut.': 'Water, Air, &amp; Soil Pollution',
  'Wood Sci. Technol.': 'Wood Science and Technology',
  'Wood Fiber Sci.': 'Wood and Fiber Science',
  'World J. Condens. Matter Phys.': 'World Journal of Condensed Matter Physics',
  'X-Ray Spectrom.': 'X-Ray Spectrometry',
  'X-Ray Struct. Anal. Online': 'X-Ray Structure Analysis Online',
  'Z. Anorg. Allg. Chem.': 'Zeitschrift fuer Anorganische und Allgemeine Chemie',
  'Z. Lebensm.-Unters. Forsch. A': 'Zeitschrift fuer Lebensmittel-Untersuchung und -Forschung A: European Food Research and Technology',
  'Z. Naturforsch., A: Phys. Sci.': 'Zeitschrift fuer Naturforschung, A: Physical Sciences',
  'Z. Naturforsch., B: J. Chem. Sci.': 'Zeitschrift fuer Naturforschung, B: A Journal of Chemical Sciences',
  'Z. Naturforsch., C: J. Biosci.': 'Zeitschrift fuer Naturforschung, C: Journal of Biosciences',
  'Z. Phys. Chem. (Muenchen, Ger.)': 'Zeitschrift fuer Physikalische Chemie (Muenchen, Germany)',
  'Zh. Org. Farm. Khim.': 'Zhurnal Organichnoi ta Farmatsevtichnoi Khimii',
}
