"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetricFilter = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const logs_generated_1 = require("./logs.generated");
/**
 * A filter that extracts information from CloudWatch Logs and emits to CloudWatch Metrics.
 *
 * @stability stable
 */
class MetricFilter extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.metricName = props.metricName;
        this.metricNamespace = props.metricNamespace;
        // It looks odd to map this object to a singleton list, but that's how
        // we're supposed to do it according to the docs.
        //
        // > Currently, you can specify only one metric transformation for
        // > each metric filter. If you want to specify multiple metric
        // > transformations, you must specify multiple metric filters.
        //
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-logs-metricfilter.html
        new logs_generated_1.CfnMetricFilter(this, 'Resource', {
            logGroupName: props.logGroup.logGroupName,
            filterPattern: props.filterPattern.logPatternString,
            metricTransformations: [{
                    metricNamespace: props.metricNamespace,
                    metricName: props.metricName,
                    metricValue: props.metricValue !== undefined ? props.metricValue : '1',
                    defaultValue: props.defaultValue,
                }],
        });
    }
    /**
     * Return the given named metric for this Metric Filter.
     *
     * @default avg over 5 minutes
     * @stability stable
     */
    metric(props) {
        return new aws_cloudwatch_1.Metric({
            metricName: this.metricName,
            namespace: this.metricNamespace,
            statistic: 'avg',
            ...props,
        }).attachTo(this);
    }
}
exports.MetricFilter = MetricFilter;
//# sourceMappingURL=data:application/json;base64,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