"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkLoadBalancer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancingv2_canned_metrics_generated_1 = require("../elasticloadbalancingv2-canned-metrics.generated");
const base_load_balancer_1 = require("../shared/base-load-balancer");
const network_listener_1 = require("./network-listener");
/**
 * Define a new network load balancer.
 *
 * @stability stable
 * @resource AWS::ElasticLoadBalancingV2::LoadBalancer
 */
class NetworkLoadBalancer extends base_load_balancer_1.BaseLoadBalancer {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'network',
        });
        if (props.crossZoneEnabled) {
            this.setAttribute('load_balancing.cross_zone.enabled', 'true');
        }
    }
    /**
     * Looks up the network load balancer.
     *
     * @stability stable
     */
    static fromLookup(scope, id, options) {
        const props = base_load_balancer_1.BaseLoadBalancer._queryContextProvider(scope, {
            userOptions: options,
            loadBalancerType: cxschema.LoadBalancerType.NETWORK,
        });
        return new LookedUpNetworkLoadBalancer(scope, id, props);
    }
    /**
     * @stability stable
     */
    static fromNetworkLoadBalancerAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.loadBalancerArn = attrs.loadBalancerArn;
                this.vpc = attrs.vpc;
            }
            addListener(lid, props) {
                return new network_listener_1.NetworkListener(this, lid, {
                    loadBalancer: this,
                    ...props,
                });
            }
            get loadBalancerCanonicalHostedZoneId() {
                if (attrs.loadBalancerCanonicalHostedZoneId) {
                    return attrs.loadBalancerCanonicalHostedZoneId;
                }
                // eslint-disable-next-line max-len
                throw new Error(`'loadBalancerCanonicalHostedZoneId' was not provided when constructing Network Load Balancer ${this.node.path} from attributes`);
            }
            get loadBalancerDnsName() {
                if (attrs.loadBalancerDnsName) {
                    return attrs.loadBalancerDnsName;
                }
                // eslint-disable-next-line max-len
                throw new Error(`'loadBalancerDnsName' was not provided when constructing Network Load Balancer ${this.node.path} from attributes`);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a listener to this load balancer.
     *
     * @returns The newly created listener
     * @stability stable
     */
    addListener(id, props) {
        return new network_listener_1.NetworkListener(this, id, {
            loadBalancer: this,
            ...props,
        });
    }
    /**
     * Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     *
     * This is extending the BaseLoadBalancer.logAccessLogs method to match the bucket permissions described
     * at https://docs.aws.amazon.com/elasticloadbalancing/latest/network/load-balancer-access-logs.html#access-logging-bucket-requirements
     *
     * @stability stable
     */
    logAccessLogs(bucket, prefix) {
        super.logAccessLogs(bucket, prefix);
        const logsDeliveryServicePrincipal = new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com');
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:PutObject'],
            principals: [logsDeliveryServicePrincipal],
            resources: [
                bucket.arnForObjects(`${prefix ? prefix + '/' : ''}AWSLogs/${this.stack.account}/*`),
            ],
            conditions: {
                StringEquals: { 's3:x-amz-acl': 'bucket-owner-full-control' },
            },
        }));
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:GetBucketAcl'],
            principals: [logsDeliveryServicePrincipal],
            resources: [bucket.bucketArn],
        }));
    }
    /**
     * Return the given named metric for this Network Load Balancer.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/NetworkELB',
            metricName,
            dimensions: { LoadBalancer: this.loadBalancerFullName },
            ...props,
        }).attachTo(this);
    }
    /**
     * The total number of concurrent TCP flows (or connections) from clients to targets.
     *
     * This metric includes connections in the SYN_SENT and ESTABLISHED states.
     * TCP connections are not terminated at the load balancer, so a client
     * opening a TCP connection to a target counts as a single flow.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricActiveFlowCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.activeFlowCountSum, {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of load balancer capacity units (LCU) used by your load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricConsumedLCUs(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.consumedLcUsAverage, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (deprecated) The number of targets that are considered healthy.
     *
     * @default Average over 5 minutes
     * @deprecated use ``NetworkTargetGroup.metricHealthyHostCount`` instead
     */
    metricHealthyHostCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.healthyHostCountAverage, props);
    }
    /**
     * (deprecated) The number of targets that are considered unhealthy.
     *
     * @default Average over 5 minutes
     * @deprecated use ``NetworkTargetGroup.metricUnHealthyHostCount`` instead
     */
    metricUnHealthyHostCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.unHealthyHostCountAverage, props);
    }
    /**
     * The total number of new TCP flows (or connections) established from clients to targets in the time period.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricNewFlowCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.newFlowCountSum, props);
    }
    /**
     * The total number of bytes processed by the load balancer, including TCP/IP headers.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricProcessedBytes(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.processedBytesSum, props);
    }
    /**
     * The total number of reset (RST) packets sent from a client to a target.
     *
     * These resets are generated by the client and forwarded by the load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricTcpClientResetCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.tcpClientResetCountSum, props);
    }
    /**
     * The total number of reset (RST) packets generated by the load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricTcpElbResetCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.tcpElbResetCountSum, props);
    }
    /**
     * The total number of reset (RST) packets sent from a target to a client.
     *
     * These resets are generated by the target and forwarded by the load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricTcpTargetResetCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.tcpTargetResetCountSum, props);
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ LoadBalancer: this.loadBalancerFullName }),
            ...props,
        }).attachTo(this);
    }
}
exports.NetworkLoadBalancer = NetworkLoadBalancer;
_a = JSII_RTTI_SYMBOL_1;
NetworkLoadBalancer[_a] = { fqn: "@aws-cdk/aws-elasticloadbalancingv2.NetworkLoadBalancer", version: "1.95.1" };
class LookedUpNetworkLoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.loadBalancerArn = props.loadBalancerArn;
        this.loadBalancerCanonicalHostedZoneId = props.loadBalancerCanonicalHostedZoneId;
        this.loadBalancerDnsName = props.loadBalancerDnsName;
        this.vpc = ec2.Vpc.fromLookup(this, 'Vpc', {
            vpcId: props.vpcId,
        });
    }
    addListener(lid, props) {
        return new network_listener_1.NetworkListener(this, lid, {
            loadBalancer: this,
            ...props,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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