"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const crypto = require("crypto");
const apigateway_generated_1 = require("./apigateway.generated");
const restapi_1 = require("./restapi");
/**
 * A Deployment of a REST API.
 *
 * An immutable representation of a RestApi resource that can be called by users
 * using Stages. A deployment must be associated with a Stage for it to be
 * callable over the Internet.
 *
 * Normally, you don't need to define deployments manually. The RestApi
 * construct manages a Deployment resource that represents the latest model. It
 * can be accessed through `restApi.latestDeployment` (unless `deploy: false` is
 * set when defining the `RestApi`).
 *
 * If you manually define this resource, you will need to know that since
 * deployments are immutable, as long as the resource's logical ID doesn't
 * change, the deployment will represent the snapshot in time in which the
 * resource was created. This means that if you modify the RestApi model (i.e.
 * add methods or resources), these changes will not be reflected unless a new
 * deployment resource is created.
 *
 * To achieve this behavior, the method `addToLogicalId(data)` can be used to
 * augment the logical ID generated for the deployment resource such that it
 * will include arbitrary data. This is done automatically for the
 * `restApi.latestDeployment` deployment.
 *
 * Furthermore, since a deployment does not reference any of the REST API
 * resources and methods, CloudFormation will likely provision it before these
 * resources are created, which means that it will represent a "half-baked"
 * model. Use the `node.addDependency(dep)` method to circumvent that. This is done
 * automatically for the `restApi.latestDeployment` deployment.
 */
class Deployment extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new LatestDeploymentResource(this, 'Resource', {
            description: props.description,
            restApi: props.api,
        });
        if (props.retainDeployments) {
            this.resource.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        this.api = props.api;
        this.deploymentId = core_1.Lazy.stringValue({ produce: () => this.resource.ref });
    }
    /**
     * Adds a component to the hash that determines this Deployment resource's
     * logical ID.
     *
     * This should be called by constructs of the API Gateway model that want to
     * invalidate the deployment when their settings change. The component will
     * be resolve()ed during synthesis so tokens are welcome.
     */
    addToLogicalId(data) {
        this.resource.addToLogicalId(data);
    }
    /**
     * Hook into synthesis before it occurs and make any final adjustments.
     */
    prepare() {
        if (this.api instanceof restapi_1.RestApi) {
            // Ignore IRestApi that are imported
            /*
             * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
             * Quoting from CloudFormation's docs - "If you create an AWS::ApiGateway::RestApi resource and its methods (using AWS::ApiGateway::Method) in
             * the same template as your deployment, the deployment must depend on the RestApi's methods. To create a dependency, add a DependsOn attribute
             * to the deployment. If you don't, AWS CloudFormation creates the deployment right after it creates the RestApi resource that doesn't contain
             * any methods, and AWS CloudFormation encounters the following error: The REST API doesn't contain any methods."
             */
            /*
             * Adding a dependency between LatestDeployment and Method construct, using ConstructNode.addDependencies(), creates additional dependencies
             * between AWS::ApiGateway::Deployment and the AWS::Lambda::Permission nodes (children under Method), causing cyclic dependency errors. Hence,
             * falling back to declaring dependencies between the underlying CfnResources.
             */
            this.api.methods.map(m => m.node.defaultChild).forEach(m => this.resource.addDependsOn(m));
        }
    }
}
exports.Deployment = Deployment;
class LatestDeploymentResource extends apigateway_generated_1.CfnDeployment {
    constructor(scope, id, props) {
        super(scope, id, {
            description: props.description,
            restApiId: props.restApi.restApiId,
        });
        this.hashComponents = new Array();
        this.api = props.restApi;
        this.originalLogicalId = core_1.Stack.of(this).getLogicalId(this);
    }
    /**
     * Allows adding arbitrary data to the hashed logical ID of this deployment.
     * This can be used to couple the deployment to the API Gateway model.
     */
    addToLogicalId(data) {
        // if the construct is locked, it means we are already synthesizing and then
        // we can't modify the hash because we might have already calculated it.
        if (this.node.locked) {
            throw new Error('Cannot modify the logical ID when the construct is locked');
        }
        this.hashComponents.push(data);
    }
    /**
     * Hooks into synthesis to calculate a logical ID that hashes all the components
     * add via `addToLogicalId`.
     */
    prepare() {
        if (this.api instanceof restapi_1.RestApi || this.api instanceof restapi_1.SpecRestApi) { // Ignore IRestApi that are imported
            // Add CfnRestApi to the logical id so a new deployment is triggered when any of its properties change.
            const cfnRestApiCF = this.api.node.defaultChild._toCloudFormation();
            this.addToLogicalId(core_1.Stack.of(this).resolve(cfnRestApiCF));
        }
        const stack = core_1.Stack.of(this);
        // if hash components were added to the deployment, we use them to calculate
        // a logical ID for the deployment resource.
        if (this.hashComponents.length > 0) {
            const md5 = crypto.createHash('md5');
            this.hashComponents.map(x => stack.resolve(x)).forEach(c => md5.update(JSON.stringify(c)));
            this.overrideLogicalId(this.originalLogicalId + md5.digest('hex'));
        }
        super.prepare();
    }
}
//# sourceMappingURL=data:application/json;base64,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