"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const cors_1 = require("./cors");
const integrations_1 = require("./integrations");
const method_1 = require("./method");
class ResourceBase extends core_1.Resource {
    constructor(scope, id) {
        super(scope, id);
        this.children = {};
    }
    addResource(pathPart, options) {
        return new Resource(this, pathPart, { parent: this, pathPart, ...options });
    }
    addMethod(httpMethod, integration, options) {
        return new method_1.Method(this, httpMethod, { resource: this, httpMethod, integration, options });
    }
    addProxy(options) {
        return new ProxyResource(this, '{proxy+}', { parent: this, ...options });
    }
    addCorsPreflight(options) {
        const headers = {};
        //
        // Access-Control-Allow-Headers
        const allowHeaders = options.allowHeaders || cors_1.Cors.DEFAULT_HEADERS;
        headers['Access-Control-Allow-Headers'] = `'${allowHeaders.join(',')}'`;
        //
        // Access-Control-Allow-Origin
        if (options.allowOrigins.length === 0) {
            throw new Error('allowOrigins must contain at least one origin');
        }
        if (options.allowOrigins.includes('*') && options.allowOrigins.length > 1) {
            throw new Error(`Invalid "allowOrigins" - cannot mix "*" with specific origins: ${options.allowOrigins.join(',')}`);
        }
        // we use the first origin here and if there are more origins in the list, we
        // will match against them in the response velocity template
        const initialOrigin = options.allowOrigins[0];
        headers['Access-Control-Allow-Origin'] = `'${initialOrigin}'`;
        // the "Vary" header is required if we allow a specific origin
        // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin#CORS_and_caching
        if (initialOrigin !== '*') {
            headers.Vary = '\'Origin\'';
        }
        //
        // Access-Control-Allow-Methods
        let allowMethods = options.allowMethods || cors_1.Cors.ALL_METHODS;
        if (allowMethods.includes('ANY')) {
            if (allowMethods.length > 1) {
                throw new Error(`ANY cannot be used with any other method. Received: ${allowMethods.join(',')}`);
            }
            allowMethods = cors_1.Cors.ALL_METHODS;
        }
        headers['Access-Control-Allow-Methods'] = `'${allowMethods.join(',')}'`;
        //
        // Access-Control-Allow-Credentials
        if (options.allowCredentials) {
            headers['Access-Control-Allow-Credentials'] = '\'true\'';
        }
        //
        // Access-Control-Max-Age
        let maxAgeSeconds;
        if (options.maxAge && options.disableCache) {
            throw new Error('The options "maxAge" and "disableCache" are mutually exclusive');
        }
        if (options.maxAge) {
            maxAgeSeconds = options.maxAge.toSeconds();
        }
        if (options.disableCache) {
            maxAgeSeconds = -1;
        }
        if (maxAgeSeconds) {
            headers['Access-Control-Max-Age'] = `'${maxAgeSeconds}'`;
        }
        //
        // Access-Control-Expose-Headers
        //
        if (options.exposeHeaders) {
            headers['Access-Control-Expose-Headers'] = `'${options.exposeHeaders.join(',')}'`;
        }
        //
        // statusCode
        const statusCode = options.statusCode !== undefined ? options.statusCode : 204;
        //
        // prepare responseParams
        const integrationResponseParams = {};
        const methodReponseParams = {};
        for (const [name, value] of Object.entries(headers)) {
            const key = `method.response.header.${name}`;
            integrationResponseParams[key] = value;
            methodReponseParams[key] = true;
        }
        return this.addMethod('OPTIONS', new integrations_1.MockIntegration({
            requestTemplates: { 'application/json': '{ statusCode: 200 }' },
            integrationResponses: [
                { statusCode: `${statusCode}`, responseParameters: integrationResponseParams, responseTemplates: renderResponseTemplate() },
            ],
        }), {
            methodResponses: [
                { statusCode: `${statusCode}`, responseParameters: methodReponseParams },
            ],
        });
        // renders the response template to match all possible origins (if we have more than one)
        function renderResponseTemplate() {
            const origins = options.allowOrigins.slice(1);
            if (origins.length === 0) {
                return undefined;
            }
            const template = new Array();
            template.push('#set($origin = $input.params("Origin"))');
            template.push('#if($origin == "") #set($origin = $input.params("origin")) #end');
            const condition = origins.map(o => `$origin.matches("${o}")`).join(' || ');
            template.push(`#if(${condition})`);
            template.push('  #set($context.responseOverride.header.Access-Control-Allow-Origin = $origin)');
            template.push('#end');
            return {
                'application/json': template.join('\n'),
            };
        }
    }
    getResource(pathPart) {
        return this.children[pathPart];
    }
    /**
     * @internal
     */
    _trackChild(pathPart, resource) {
        this.children[pathPart] = resource;
    }
    resourceForPath(path) {
        if (!path) {
            return this;
        }
        if (path.startsWith('/')) {
            if (this.path !== '/') {
                throw new Error(`Path may start with "/" only for the resource, but we are at: ${this.path}`);
            }
            // trim trailing "/"
            return this.resourceForPath(path.substr(1));
        }
        const parts = path.split('/');
        const next = parts.shift();
        if (!next || next === '') {
            throw new Error('resourceForPath cannot be called with an empty path');
        }
        let resource = this.getResource(next);
        if (!resource) {
            resource = this.addResource(next);
        }
        return resource.resourceForPath(parts.join('/'));
    }
    get url() {
        return this.restApi.urlForPath(this.path);
    }
}
exports.ResourceBase = ResourceBase;
class Resource extends ResourceBase {
    constructor(scope, id, props) {
        super(scope, id);
        validateResourcePathPart(props.pathPart);
        this.parentResource = props.parent;
        if (props.parent instanceof ResourceBase) {
            props.parent._trackChild(props.pathPart, this);
        }
        const resourceProps = {
            restApiId: props.parent.restApi.restApiId,
            parentId: props.parent.resourceId,
            pathPart: props.pathPart,
        };
        const resource = new apigateway_generated_1.CfnResource(this, 'Resource', resourceProps);
        this.resourceId = resource.ref;
        this.restApi = props.parent.restApi;
        // render resource path (special case for root)
        this.path = props.parent.path;
        if (!this.path.endsWith('/')) {
            this.path += '/';
        }
        this.path += props.pathPart;
        const deployment = props.parent.restApi.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ resource: resourceProps });
        }
        // setup defaults based on properties and inherit from parent. method defaults
        // are inherited per property, so children can override piecemeal.
        this.defaultIntegration = props.defaultIntegration || props.parent.defaultIntegration;
        this.defaultMethodOptions = {
            ...props.parent.defaultMethodOptions,
            ...props.defaultMethodOptions,
        };
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions || props.parent.defaultCorsPreflightOptions;
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
}
exports.Resource = Resource;
/**
 * Defines a {proxy+} greedy resource and an ANY method on a route.
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-set-up-simple-proxy.html
 */
class ProxyResource extends Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            parent: props.parent,
            pathPart: '{proxy+}',
            defaultIntegration: props.defaultIntegration,
            defaultMethodOptions: props.defaultMethodOptions,
        });
        const anyMethod = props.anyMethod !== undefined ? props.anyMethod : true;
        if (anyMethod) {
            this.anyMethod = this.addMethod('ANY');
        }
    }
    addMethod(httpMethod, integration, options) {
        // In case this proxy is mounted under the root, also add this method to
        // the root so that empty paths are proxied as well.
        if (this.parentResource && this.parentResource.path === '/') {
            // skip if the root resource already has this method defined
            if (!(this.parentResource.node.tryFindChild(httpMethod) instanceof method_1.Method)) {
                this.parentResource.addMethod(httpMethod, integration, options);
            }
        }
        return super.addMethod(httpMethod, integration, options);
    }
}
exports.ProxyResource = ProxyResource;
function validateResourcePathPart(part) {
    // strip {} which indicate this is a parameter
    if (part.startsWith('{') && part.endsWith('}')) {
        part = part.substr(1, part.length - 2);
        // proxy resources are allowed to end with a '+'
        if (part.endsWith('+')) {
            part = part.substr(0, part.length - 1);
        }
    }
    if (!/^[a-zA-Z0-9\.\_\-]+$/.test(part)) {
        throw new Error(`Resource's path part only allow [a-zA-Z0-9._-], an optional trailing '+'
      and curly braces at the beginning and the end: ${part}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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