"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
/**
 * Time period for which quota settings apply.
 */
var Period;
(function (Period) {
    Period["DAY"] = "DAY";
    Period["WEEK"] = "WEEK";
    Period["MONTH"] = "MONTH";
})(Period = exports.Period || (exports.Period = {}));
class UsagePlan extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.apiStages = new Array();
        let resource;
        resource = new apigateway_generated_1.CfnUsagePlan(this, 'Resource', {
            apiStages: core_1.Lazy.anyValue({ produce: () => this.renderApiStages(this.apiStages) }),
            description: props.description,
            quota: this.renderQuota(props),
            throttle: this.renderThrottle(props.throttle),
            usagePlanName: props.name,
        });
        this.apiStages.push(...(props.apiStages || []));
        this.usagePlanId = resource.ref;
        // Add ApiKey when
        if (props.apiKey) {
            this.addApiKey(props.apiKey);
        }
    }
    /**
     * Adds an ApiKey.
     *
     * @param apiKey
     */
    addApiKey(apiKey) {
        const prefix = 'UsagePlanKeyResource';
        // Postfixing apikey id only from the 2nd child, to keep physicalIds of UsagePlanKey for existing CDK apps unmodifed.
        const id = this.node.tryFindChild(prefix) ? `${prefix}:${apiKey.node.uniqueId}` : prefix;
        new apigateway_generated_1.CfnUsagePlanKey(this, id, {
            keyId: apiKey.keyId,
            keyType: "API_KEY" /* API_KEY */,
            usagePlanId: this.usagePlanId,
        });
    }
    /**
     * Adds an apiStage.
     * @param apiStage
     */
    addApiStage(apiStage) {
        this.apiStages.push(apiStage);
    }
    /**
     *
     * @param props
     */
    renderApiStages(apiStages) {
        if (apiStages && apiStages.length > 0) {
            const stages = [];
            apiStages.forEach((apiStage) => {
                stages.push(this.createStage(apiStage));
            });
            return stages;
        }
        return undefined;
    }
    createStage(apiStage) {
        const stage = apiStage.stage ? apiStage.stage.stageName.toString() : undefined;
        const apiId = apiStage.stage ? apiStage.stage.restApi.restApiId : undefined;
        const throttle = this.renderThrottlePerMethod(apiStage.throttle);
        return {
            apiId,
            stage,
            throttle,
        };
    }
    renderQuota(props) {
        if (props.quota === undefined) {
            return undefined;
        }
        else {
            const limit = props.quota ? props.quota.limit : undefined;
            util_1.validateInteger(limit, 'Throttle quota limit');
            const ret = {
                limit: limit ? limit : undefined,
                offset: props.quota ? props.quota.offset : undefined,
                period: props.quota ? props.quota.period : undefined,
            };
            return ret;
        }
    }
    renderThrottle(props) {
        let ret;
        if (props !== undefined) {
            const burstLimit = props.burstLimit;
            util_1.validateInteger(burstLimit, 'Throttle burst limit');
            const rateLimit = props.rateLimit;
            util_1.validateInteger(rateLimit, 'Throttle rate limit');
            ret = {
                burstLimit: burstLimit ? burstLimit : undefined,
                rateLimit: rateLimit ? rateLimit : undefined,
            };
        }
        return ret;
    }
    renderThrottlePerMethod(throttlePerMethod) {
        const ret = {};
        if (throttlePerMethod && throttlePerMethod.length > 0) {
            throttlePerMethod.forEach((value) => {
                const method = value.method;
                // this methodId is resource path and method for example /GET or /pets/GET
                const methodId = `${method.resource.path}/${method.httpMethod}`;
                ret[methodId] = this.renderThrottle(value.throttle);
            });
        }
        return ret;
    }
}
exports.UsagePlan = UsagePlan;
//# sourceMappingURL=data:application/json;base64,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