"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticWebsiteOrigin = exports.StaticWebsite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const aws_cloudfront_origins_1 = require("aws-cdk-lib/aws-cloudfront-origins");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const cloudfront_web_acl_1 = require("./cloudfront-web-acl");
const DEFAULT_RUNTIME_CONFIG_FILENAME = "runtime-config.json";
/**
 * Deploys a Static Website using by default a private S3 bucket as an origin and Cloudfront as the entrypoint.
 *
 * This construct configures a webAcl containing rules that are generally applicable to web applications. This
 * provides protection against exploitation of a wide range of vulnerabilities, including some of the high risk
 * and commonly occurring vulnerabilities described in OWASP publications such as OWASP Top 10.
 *
 */
class StaticWebsite extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.validateProps = (props) => {
            this.validateEncryptionSettings(props);
            props.runtimeOptions && this.validateRuntimeConfig(props.runtimeOptions);
            props.websiteBucket && this.validateBucketConfig(props.websiteBucket);
        };
        this.validateRuntimeConfig = (config) => {
            if (!config) {
                throw new Error("validateRuntimeConfig only accepts non-null RuntimeOptions.");
            }
            if (config.jsonFileName && !config.jsonFileName.endsWith(".json")) {
                throw new Error("RuntimeOptions.jsonFileName must be a json file.");
            }
        };
        this.validateBucketConfig = (bucket) => {
            if (bucket.isWebsite) {
                throw new Error("Website buckets cannot be configured as websites as this will break Cloudfront hosting!");
            }
        };
        this.validateEncryptionSettings = ({ defaultWebsiteBucketEncryption, defaultWebsiteBucketEncryptionKey, }) => {
            if (defaultWebsiteBucketEncryptionKey &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS) {
                throw new Error("Bucket encryption should be set to KMS if providing a defaultWebsiteBucketEncryptionKey.");
            }
            if (defaultWebsiteBucketEncryption &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.S3_MANAGED) {
                throw new Error("Only KMS and S3_MANAGED encryption are supported on the default bucket.");
            }
        };
        this.suppressCDKNagViolations = (props) => {
            const stack = aws_cdk_lib_1.Stack.of(this);
            !props.distributionProps?.certificate &&
                cdk_nag_1.NagSuppressions.addResourceSuppressions(this.cloudFrontDistribution, [
                    {
                        id: "AwsSolutions-CFR4",
                        reason: "Certificate is not mandatory therefore the Cloudfront certificate will be used.",
                    },
                ]);
            cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                {
                    id: "AwsSolutions-L1",
                    reason: "Latest runtime cannot be configured. CDK will need to upgrade the BucketDeployment construct accordingly.",
                },
            ]);
            cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                {
                    id: "AwsSolutions-IAM5",
                    reason: "All Policies have been scoped to a Bucket. Given Buckets can contain arbitrary content, wildcard resources with bucket scope are required.",
                    appliesTo: [
                        {
                            regex: "/^Action::s3:.*$/g",
                        },
                        {
                            regex: `/^Resource::.*$/g`,
                        },
                    ],
                },
            ]);
            cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                {
                    id: "AwsSolutions-IAM4",
                    reason: "Buckets can contain arbitrary content, therefore wildcard resources under a bucket are required.",
                    appliesTo: [
                        {
                            regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole$/g`,
                        },
                    ],
                },
            ]);
        };
        this.validateProps(props);
        // S3 Bucket to hold website files
        this.websiteBucket =
            props.websiteBucket ??
                new aws_s3_1.Bucket(this, "WebsiteBucket", {
                    versioned: true,
                    enforceSSL: true,
                    autoDeleteObjects: true,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                    encryption: props.defaultWebsiteBucketEncryption ?? aws_s3_1.BucketEncryption.S3_MANAGED,
                    encryptionKey: props.defaultWebsiteBucketEncryptionKey,
                    publicReadAccess: false,
                    blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                    serverAccessLogsPrefix: "access-logs",
                });
        // Web ACL
        const { distributionProps } = props;
        const webAclArn = distributionProps?.webAclId ??
            new cloudfront_web_acl_1.CloudfrontWebAcl(this, "WebsiteAcl", props.webAclProps).webAclArn;
        // Cloudfront Distribution
        const logBucket = props.distributionProps?.logBucket ||
            new aws_s3_1.Bucket(this, "DistributionLogBucket", {
                enforceSSL: true,
                autoDeleteObjects: true,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                encryption: props.defaultWebsiteBucketEncryption ?? aws_s3_1.BucketEncryption.S3_MANAGED,
                encryptionKey: props.defaultWebsiteBucketEncryptionKey,
                publicReadAccess: false,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                serverAccessLogsPrefix: "access-logs",
            });
        const defaultRootObject = distributionProps?.defaultRootObject ?? "index.html";
        this.cloudFrontDistribution = new aws_cloudfront_1.Distribution(this, "CloudfrontDistribution", {
            webAclId: webAclArn,
            enableLogging: true,
            logBucket: logBucket,
            defaultBehavior: {
                ...distributionProps?.defaultBehavior,
                origin: new aws_cloudfront_origins_1.S3Origin(this.websiteBucket),
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            defaultRootObject,
            errorResponses: distributionProps?.errorResponses ?? [
                {
                    httpStatus: 404,
                    responseHttpStatus: 200,
                    responsePagePath: `/${defaultRootObject}`,
                },
            ],
            ...distributionProps,
        });
        // Deploy Website
        this.bucketDeployment = new aws_s3_deployment_1.BucketDeployment(this, "WebsiteDeployment", {
            sources: [
                aws_s3_deployment_1.Source.asset(props.websiteContentPath),
                ...(props.runtimeOptions
                    ? [
                        aws_s3_deployment_1.Source.jsonData(props.runtimeOptions?.jsonFileName ||
                            DEFAULT_RUNTIME_CONFIG_FILENAME, props.runtimeOptions?.jsonPayload),
                    ]
                    : []),
            ],
            destinationBucket: this.websiteBucket,
            // Files in the distribution's edge caches will be invalidated after files are uploaded to the destination bucket.
            distribution: this.cloudFrontDistribution,
        });
        this.suppressCDKNagViolations(props);
    }
}
exports.StaticWebsite = StaticWebsite;
_a = JSII_RTTI_SYMBOL_1;
StaticWebsite[_a] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsite", version: "0.5.1" };
/**
 * If passing in distributionProps, the default behaviour.origin is a required parameter. An instance of this class can be passed in
 * to make the compiler happy.
 */
class StaticWebsiteOrigin {
    bind(_scope, _options) {
        throw new Error("This should never be called");
    }
}
exports.StaticWebsiteOrigin = StaticWebsiteOrigin;
_b = JSII_RTTI_SYMBOL_1;
StaticWebsiteOrigin[_b] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsiteOrigin", version: "0.5.1" };
//# sourceMappingURL=data:application/json;base64,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