"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcProvider = exports.AmiProvider = exports.Cluster = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const DEFAULT_INSTANCE_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M6G, ec2.InstanceSize.MEDIUM);
/**
 * Represents the k3sCluster construct.
 *
 * @stability stable
 */
class Cluster extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _d, _e, _f, _g;
        super(scope, id);
        // VPC configuration
        const vpc = (_d = props.vpc) !== null && _d !== void 0 ? _d : new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        // S3 bucket to host K3s token + kubeconfig file
        // support s3 bucket autoDeleteObjects native.
        // see - https://docs.aws.amazon.com/cdk/api/latest/docs/aws-s3-readme.html#bucket-deletion
        // PR - https://github.com/aws/aws-cdk/commit/32e9c23be2852cfca79a57c90e52b9301b1c7081
        let k3sBucket;
        if (props.bucketRemovalPolicy === cdk.RemovalPolicy.DESTROY) {
            k3sBucket = new s3.Bucket(this, 'k3sBucket', {
                removalPolicy: props.bucketRemovalPolicy,
                autoDeleteObjects: true,
            });
        }
        else {
            k3sBucket = new s3.Bucket(this, 'k3sBucket', {
                removalPolicy: cdk.RemovalPolicy.RETAIN,
            });
        }
        // control plane node Security Group
        const k3scontrolplanesg = new ec2.SecurityGroup(this, 'k3s-controlplane-SG', { vpc });
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // worker nodes Security Group
        const k3sworkersg = new ec2.SecurityGroup(this, 'k3s-worker-SG', { vpc });
        // for this prototype the workers are being placed in a public subnet
        // ideally they should land on a private subnet
        /// also ingress traffic - ssh (bastion style) or 6443 - should come from the control plane node only
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // check if the user requires a particular instance type for workers and control plane
        // if not, the default instance type is used
        this.controlPlaneInstanceType = (_e = props.controlPlaneInstanceType) !== null && _e !== void 0 ? _e : DEFAULT_INSTANCE_TYPE;
        this.workerInstanceType = (_f = props.workerInstanceType) !== null && _f !== void 0 ? _f : DEFAULT_INSTANCE_TYPE;
        // create control plane node
        const k3scontrolplane = new ec2.Instance(this, 'k3s-controlplane', {
            instanceType: this.controlPlaneInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnets: vpc.publicSubnets,
            },
            instanceName: 'k3s-controlplane',
            securityGroup: k3scontrolplanesg,
        });
        k3scontrolplane.addUserData(`
       #!/bin/bash
       curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.9%2Bk3s1/k3s-arm64
       chmod +x k3s
       ./k3s server &
       sleep 30
       ENDPOINT=$(curl http://169.254.169.254/latest/meta-data/public-hostname) 
       cp /etc/rancher/k3s/k3s.yaml /etc/rancher/k3s/kubeconfig.yaml
       sed -i s/127.0.0.1/$ENDPOINT/ /etc/rancher/k3s/kubeconfig.yaml
       aws s3 cp /var/lib/rancher/k3s/server/node-token s3://${k3sBucket.bucketName}/node-token
       aws s3 cp /etc/rancher/k3s/kubeconfig.yaml s3://${k3sBucket.bucketName}/kubeconfig.yaml
     `);
        this.endpointUri = k3scontrolplane.instancePublicIp;
        // create launch template for worker ASG
        // prepare the userData
        const userData = ec2.UserData.forLinux();
        userData.addCommands(`
          #!/bin/bash
          LOGFILE='/var/log/k3s.log'
          curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.13%2Bk3s1/k3s-arm64
          chmod +x k3s
          echo the bucket name is ${k3sBucket.bucketName} 
          aws s3 cp s3://${k3sBucket.bucketName}/node-token /node-token 
          (./k3s agent --server https://${k3scontrolplane.instancePrivateIp}:6443 \
          --token $(cat /node-token) 2>&1 | tee -a $LOGFILE || echo "failed" > $LOGFILE &)
    `);
        // create worker ASG
        const workerAsg = new autoscaling.AutoScalingGroup(this, 'WorkerAsg', {
            instanceType: this.workerInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnetType: ec2.SubnetType.PUBLIC,
            },
            minCapacity: (_g = props.workerMinCapacity) !== null && _g !== void 0 ? _g : 3,
        });
        const cfnInstanceProfile = workerAsg.node.tryFindChild('InstanceProfile');
        const lt = new ec2.CfnLaunchTemplate(this, 'WorkerLaunchTemplate', {
            launchTemplateData: {
                imageId: new AmiProvider().amiId.getImage(this).imageId,
                instanceType: this.workerInstanceType.toString(),
                instanceMarketOptions: {
                    marketType: props.spotWorkerNodes ? 'spot' : undefined,
                    spotOptions: props.spotWorkerNodes ? {
                        spotInstanceType: 'one-time',
                    } : undefined,
                },
                userData: cdk.Fn.base64(userData.render()),
                iamInstanceProfile: {
                    arn: cfnInstanceProfile.attrArn,
                },
            },
        });
        const cfnAsg = workerAsg.node.tryFindChild('ASG');
        cfnAsg.addPropertyDeletionOverride('LaunchConfigurationName');
        cfnAsg.addPropertyOverride('LaunchTemplate', {
            LaunchTemplateId: lt.ref,
            Version: lt.attrLatestVersionNumber,
        });
        workerAsg.addSecurityGroup(k3sworkersg);
        // enable the SSM session manager
        workerAsg.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        // grant the S3 write permission to the control plane node and read permissions to the worker nodes
        k3sBucket.grantWrite(k3scontrolplane.role);
        k3sBucket.grantRead(workerAsg.role);
        // endpoint info
        new cdk.CfnOutput(this, 'Endpoint', { value: `https://${k3scontrolplane.instancePublicIp}:6443` });
        // kubeconfig.yaml path
        new cdk.CfnOutput(this, 'Kubernetes configuration file', { value: `s3://${k3sBucket.bucketName}/kubeconfig.yaml` });
        workerAsg.node.addDependency(k3scontrolplane);
    }
}
exports.Cluster = Cluster;
_a = JSII_RTTI_SYMBOL_1;
Cluster[_a] = { fqn: "cdk-k3s-cluster.Cluster", version: "0.0.52" };
/**
 * The AMI provider to get the latest Amazon Linux 2 AMI for ARM64.
 *
 * @stability stable
 */
class AmiProvider {
    /**
     * @stability stable
     */
    get amiId() {
        return ec2.MachineImage.latestAmazonLinux({
            cpuType: ec2.AmazonLinuxCpuType.ARM_64,
            generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
        });
    }
}
exports.AmiProvider = AmiProvider;
_b = JSII_RTTI_SYMBOL_1;
AmiProvider[_b] = { fqn: "cdk-k3s-cluster.AmiProvider", version: "0.0.52" };
/**
 * The VPC provider to create or import the VPC.
 *
 * @stability stable
 */
class VpcProvider {
    /**
     * @stability stable
     */
    static getOrCreate(scope) {
        const vpc = scope.node.tryGetContext('use_default_vpc') === '1' ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
            scope.node.tryGetContext('use_vpc_id') ?
                ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
                new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
        return vpc;
    }
}
exports.VpcProvider = VpcProvider;
_c = JSII_RTTI_SYMBOL_1;
VpcProvider[_c] = { fqn: "cdk-k3s-cluster.VpcProvider", version: "0.0.52" };
//# sourceMappingURL=data:application/json;base64,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