"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKNag = exports.PDKNagApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const CDK_NAG_MESSAGE_TYPES = {
    ERROR: "aws:cdk:error",
    WARNING: "aws:cdk:warning",
};
const CDK_NAG_MESSAGE_TYPES_SET = new Set(Object.values(CDK_NAG_MESSAGE_TYPES));
/**
 * @inheritDoc
 */
class PDKNagApp extends aws_cdk_lib_1.App {
    constructor(props) {
        super(props);
        this._nagResults = [];
        this.failOnError = props?.failOnError ?? false;
        this.failOnWarning = props?.failOnWarning ?? false;
    }
    synth(options) {
        const assembly = super.synth(options);
        const typesToFail = new Set([
            this.failOnError && CDK_NAG_MESSAGE_TYPES.ERROR,
            this.failOnWarning && CDK_NAG_MESSAGE_TYPES.WARNING,
        ].filter((t) => t));
        if (this._nagResults.find((r) => r.messages.find((m) => typesToFail.has(m.messageType)))) {
            throw new Error(JSON.stringify(this._nagResults, undefined, 2));
        }
        return assembly;
    }
    addNagResult(result) {
        this._nagResults.push(result);
    }
    /**
     * Returns a list of NagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    nagResults() {
        return this._nagResults;
    }
}
exports.PDKNagApp = PDKNagApp;
_a = JSII_RTTI_SYMBOL_1;
PDKNagApp[_a] = { fqn: "aws-prototyping-sdk.pdk_nag.PDKNagApp", version: "0.5.0" };
class PDKNagAspect extends cdk_nag_1.AwsSolutionsChecks {
    constructor(app) {
        super({
            verbose: true,
            reports: true,
        });
        this.app = app;
    }
    visit(node) {
        super.visit(node);
        const results = node.node.metadata.filter((m) => CDK_NAG_MESSAGE_TYPES_SET.has(m.type));
        results.length > 0 &&
            this.app.addNagResult({
                resource: node.node.path,
                messages: results.map((m) => ({
                    messageDescription: m.data,
                    messageType: m.type,
                })),
            });
    }
}
/**
 * Helper for create a Nag Enabled App.
 */
class PDKNag {
    /**
     * Returns an instance of an App with Nag enabled.
     *
     * @param props props to initialize the app with.
     */
    static app(props) {
        const app = new PDKNagApp(props);
        aws_cdk_lib_1.Aspects.of(app).add(new PDKNagAspect(app));
        return app;
    }
    /**
     * Wrapper around NagSuppressions which does not throw.
     *
     * @param stack stack instance
     * @param path resource path
     * @param suppressions list of suppressions to apply.
     * @param applyToChildren whether to apply to children.
     */
    static addResourceSuppressionsByPathNoThrow(stack, path, suppressions, applyToChildren = false) {
        try {
            cdk_nag_1.NagSuppressions.addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren);
        }
        catch (e) {
            console.info(`Unable to add Nag Suppression for path: ${path} as it does not exist.`);
        }
    }
    /**
     * Returns a prefix comprising of a delimited set of Stack Ids.
     *
     * For example: StackA/NestedStackB/
     *
     * @param stack stack instance.
     */
    static getStackPrefix(stack) {
        if (stack.nested) {
            return `${PDKNag.getStackPrefix(stack.nestedStackParent)}${stack.node.id}/`;
        }
        else {
            return `${stack.stackName}/`;
        }
    }
    /**
     * Returns a stack partition regex.
     *
     * @param stack stack instance.
     */
    static getStackPartitionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackPartitionRegex(stack.nestedStackParent);
        }
        else {
            return stack.partition.startsWith("${Token")
                ? "<AWS::Partition>"
                : `(<AWS::Partition>|${stack.partition})`;
        }
    }
    /**
     * Returns a stack region regex.
     *
     * @param stack stack instance.
     */
    static getStackRegionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackRegionRegex(stack.nestedStackParent);
        }
        else {
            return stack.region.startsWith("${Token")
                ? "<AWS::Region>"
                : `(<AWS::Region>|${stack.region})`;
        }
    }
    /**
     * Returns a stack account regex.
     *
     * @param stack stack instance.
     */
    static getStackAccountRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackAccountRegex(stack.nestedStackParent);
        }
        else {
            return stack.account.startsWith("${Token")
                ? "<AWS::AccountId>"
                : `(<AWS::AccountId>|${stack.account})`;
        }
    }
}
exports.PDKNag = PDKNag;
_b = JSII_RTTI_SYMBOL_1;
PDKNag[_b] = { fqn: "aws-prototyping-sdk.pdk_nag.PDKNag", version: "0.5.0" };
//# sourceMappingURL=data:application/json;base64,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