from typing import Dict, List, Optional

from pydantic import BaseModel, Field

from metaphor.models.metadata_change_event import DataPlatform

connection_type_map: Dict[str, DataPlatform] = {
    "bigquery": DataPlatform.BIGQUERY,
    "postgres": DataPlatform.POSTGRESQL,
    "redshift": DataPlatform.REDSHIFT,
    "snowflake": DataPlatform.SNOWFLAKE,
}

# NOTE!!! the id (uuid) of an entity from graphql api is different from
# the id of the same entity from the REST api, use luid instead
workbooks_graphql_query = """
query {
  workbooks {
    luid
    name
    projectName
    vizportalUrlId
    upstreamDatasources {
      id
      luid
      vizportalUrlId
      name
      description
      fields {
        name
        description
      }
      upstreamTables {
        luid
        name
        fullName
        schema
        database {
          name
          connectionType
        }
      }
    }
    embeddedDatasources {
      id
      name
      fields {
        name
        description
      }
      upstreamTables {
        luid
        name
        fullName
        schema
        database {
          name
          connectionType
        }
      }
    }
  }
}
"""


class Database(BaseModel):
    name: str
    description: Optional[str]
    connectionType: str


class DatabaseTable(BaseModel):
    luid: str
    name: str
    fullName: Optional[str]
    schema_: Optional[str] = Field(alias="schema")
    description: Optional[str]
    database: Optional[Database]


class DatasourceField(BaseModel):
    name: str
    description: Optional[str]


class PublishedDatasource(BaseModel):
    id: str
    luid: str
    vizportalUrlId: str
    name: str
    description: Optional[str]
    fields: List[DatasourceField]
    upstreamTables: List[DatabaseTable]


class EmbeddedDatasource(BaseModel):
    id: str
    name: str
    fields: List[DatasourceField]
    upstreamTables: List[DatabaseTable]


class WorkbookQueryResponse(BaseModel):
    """Modeling Metadata Graphql API response for a workbook"""

    luid: str
    name: str
    projectName: str
    vizportalUrlId: str
    upstreamDatasources: List[PublishedDatasource]
    embeddedDatasources: List[EmbeddedDatasource]
