"This module contains methods for rooting TreeNode objects."

def outgroup(tree, outgroups):
    """Root the provided tree by the provided outgroups, given that at least
    one of the outgroups are present within the tree and that the OTUs that are
    present are non-repetetive and form a monophyletic group. Returns the
    rooted tree and True if the conditions are met or the input tree and False
    if the conditions were not met.

    Parameters
    ----------
    tree : TreeNode object
        The node you wish to root.
    outgroups : list of strings
        Root the provided TreeNode object using the OTUs within this list.

    Returns
    -------
    tree : TreeNode object
        Tree rerooted at outgroups if conditions are met, input tree if not.
    rooted : bool
        True if conditions are met, False if not.
    """
    rooted = False

    outgroups_in_tree = tree.outgroups_present(outgroups)

    if not outgroups_in_tree:
        # no outgroups found
        return tree, rooted

    if len(set(outgroups_in_tree)) < len(outgroups_in_tree):
        # repetetive outgroups found
        return tree, rooted

    if len(outgroups_in_tree) == 1:
        outgroup_otu = outgroups_in_tree[0]
        for leaf in tree.iter_leaves():
            otu = leaf.otu()
            if otu == outgroup_otu:
                rooted = True
                return tree.reroot(leaf), rooted

    for branch in tree.iter_branches():
        if branch.is_monophyletic_outgroup(outgroups_in_tree):
            rooted = True
            return tree.reroot(branch), rooted
    return tree, rooted

def midpoint(tree):
    """Calculate tip to tip distances of the provided tree and root it halfway
    between the two longest tips.

    Parameters
    ----------
    tree : TreeNode object
        The tree to be rooted.

    Returns
    -------
    tree_rooted : TreeNode object
        Midpoint rooted input tree.
    """
    longest_dist = None
    dists = tree.distances()
    for pair in dists:
        if not longest_dist or longest_dist < dists[pair]:
            longest_dist = dists[pair]
            node_a, node_b = pair

    if not longest_dist or not node_a or not node_b:
        return tree

    mid_dist = longest_dist / 2.0
    midpoint_node = None
    parent = node_a
    dist_walked = 0
    current_child = None

    # walk up
    while parent:
        dist_walked += parent.dist
        if dist_walked > mid_dist:
            midpoint_node = parent
            break
        for child in parent.children:
            if node_b in child.iter_leaves():
                current_child = child
                break
        parent = parent.parent

    # walk down
    if current_child:
        while current_child:
            dist_walked += current_child.dist
            if dist_walked > mid_dist:
                midpoint_node = current_child
                break
            for child in current_child.children:
                if node_b in child.iter_leaves():
                    current_child = child

    if midpoint_node:
        return tree.reroot(midpoint_node)
    else:
        return tree
