from typing import Dict

from mindfoundry.optaas.client.utils import _pprint


class Configuration:
    """A set of values assigned to each :class:`.Parameter`, as generated by OPTaaS or defined by the user.

    Attributes:
        json (Dict): The full Configuration object as stored in OPTaaS in JSON format.
        id (str): Unique id for the Configuration.
        values (Dict): Values assigned to each Parameter.
            Dict keys will be the Parameter names, and values will correspond to the Parameter type.
            :class:`GroupParameters <.GroupParameter>` will appear as nested Dicts.
        type (str): One of

            * `default`: all parameters are set to their default values
            * `exploration`: values are chosen so as to explore a new region of the parameter space
            * `exploitation`: values are chosen so as to improve upon a previously provided result
            * `user-defined`: values are provided by the user
    """

    def __init__(self, json: Dict) -> None:
        self.json = json
        self.id = json['id']  # pylint: disable=invalid-name
        self.values = json['values']
        self.type = json.get('type')
        self.results_url = json['_links']['results']['href']

    def __eq__(self, other):
        return self.__class__ == other.__class__ and self.__dict__ == other.__dict__

    def __repr__(self):
        return _pprint(self, 'type', 'values')
