"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.A4b = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [a4b](https://docs.aws.amazon.com/service-authorization/latest/reference/list_alexaforbusiness.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class A4b extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [a4b](https://docs.aws.amazon.com/service-authorization/latest/reference/list_alexaforbusiness.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'a4b';
        this.accessLevelList = {
            "Write": [
                "ApproveSkill",
                "AssociateContactWithAddressBook",
                "AssociateDeviceWithRoom",
                "AssociateSkillGroupWithRoom",
                "AssociateSkillWithSkillGroup",
                "AssociateSkillWithUsers",
                "CompleteRegistration",
                "CreateAddressBook",
                "CreateBusinessReportSchedule",
                "CreateConferenceProvider",
                "CreateContact",
                "CreateProfile",
                "CreateRoom",
                "CreateSkillGroup",
                "CreateUser",
                "DeleteAddressBook",
                "DeleteBusinessReportSchedule",
                "DeleteConferenceProvider",
                "DeleteContact",
                "DeleteDevice",
                "DeleteProfile",
                "DeleteRoom",
                "DeleteRoomSkillParameter",
                "DeleteSkillAuthorization",
                "DeleteSkillGroup",
                "DeleteUser",
                "DisassociateContactFromAddressBook",
                "DisassociateDeviceFromRoom",
                "DisassociateSkillFromSkillGroup",
                "DisassociateSkillFromUsers",
                "DisassociateSkillGroupFromRoom",
                "ForgetSmartHomeAppliances",
                "PutConferencePreference",
                "PutDeviceSetupEvents",
                "PutRoomSkillParameter",
                "PutSkillAuthorization",
                "RegisterAVSDevice",
                "RegisterDevice",
                "RejectSkill",
                "RevokeInvitation",
                "SendInvitation",
                "StartDeviceSync",
                "UpdateAddressBook",
                "UpdateBusinessReportSchedule",
                "UpdateConferenceProvider",
                "UpdateContact",
                "UpdateDevice",
                "UpdateProfile",
                "UpdateRoom",
                "UpdateSkillGroup"
            ],
            "Read": [
                "GetAddressBook",
                "GetConferencePreference",
                "GetConferenceProvider",
                "GetContact",
                "GetDevice",
                "GetNetworkProfile",
                "GetProfile",
                "GetRoom",
                "GetRoomSkillParameter",
                "GetSkillGroup",
                "ListTags",
                "ResolveRoom",
                "StartSmartHomeApplianceDiscovery"
            ],
            "List": [
                "ListBusinessReportSchedules",
                "ListConferenceProviders",
                "ListDeviceEvents",
                "ListSkills",
                "ListSkillsStoreCategories",
                "ListSkillsStoreSkillsByCategory",
                "ListSmartHomeAppliances",
                "SearchAddressBooks",
                "SearchContacts",
                "SearchDevices",
                "SearchNetworkProfiles",
                "SearchProfiles",
                "SearchRooms",
                "SearchSkillGroups",
                "SearchUsers"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Associates a skill with the organization under the customer's AWS account. If a skill is private, the user implicitly accepts access to this skill during enablement.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_ApproveSkill.html
     */
    toApproveSkill() {
        this.to('a4b:ApproveSkill');
        return this;
    }
    /**
     * Associates a contact with a given address book.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_AssociateContactWithAddressBook.html
     */
    toAssociateContactWithAddressBook() {
        this.to('a4b:AssociateContactWithAddressBook');
        return this;
    }
    /**
     * Associates device with given room.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_AssociateDeviceWithRoom.html
     */
    toAssociateDeviceWithRoom() {
        this.to('a4b:AssociateDeviceWithRoom');
        return this;
    }
    /**
     * Associates the skill group with given room. SkillGroup ARN and Room ARN must be specified.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_AssociateSkillGroupWithRoom.html
     */
    toAssociateSkillGroupWithRoom() {
        this.to('a4b:AssociateSkillGroupWithRoom');
        return this;
    }
    /**
     * Associates a skill with a skill group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_AssociateSkillWithSkillGroup.html
     */
    toAssociateSkillWithSkillGroup() {
        this.to('a4b:AssociateSkillWithSkillGroup');
        return this;
    }
    /**
     * Makes a private skill available for enrolled users to enable on their devices.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_AssociateSkillWithUsers.html
     */
    toAssociateSkillWithUsers() {
        this.to('a4b:AssociateSkillWithUsers');
        return this;
    }
    /**
     * Completes the operation of registering an Alexa device.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/ag/manage-devices.html
     */
    toCompleteRegistration() {
        this.to('a4b:CompleteRegistration');
        return this;
    }
    /**
     * Creates an address book with the specified details.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_CreateAddressBook.html
     */
    toCreateAddressBook() {
        this.to('a4b:CreateAddressBook');
        return this;
    }
    /**
     * Creates a recurring schedule for usage reports to deliver to the specified S3 location with a specified daily or weekly interval.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_CreateBusinessReportSchedule.html
     */
    toCreateBusinessReportSchedule() {
        this.to('a4b:CreateBusinessReportSchedule');
        return this;
    }
    /**
     * Adds a new conference provider under the user's AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_CreateConferenceProvider.html
     */
    toCreateConferenceProvider() {
        this.to('a4b:CreateConferenceProvider');
        return this;
    }
    /**
     * Creates a contact with the specified details.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_CreateContact.html
     */
    toCreateContact() {
        this.to('a4b:CreateContact');
        return this;
    }
    /**
     * Creates a new profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_CreateProfile.html
     */
    toCreateProfile() {
        this.to('a4b:CreateProfile');
        return this;
    }
    /**
     * Create room with the specified details.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_CreateRoom.html
     */
    toCreateRoom() {
        this.to('a4b:CreateRoom');
        return this;
    }
    /**
     * Creates a skill group with given name and description.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_CreateSkillGroup.html
     */
    toCreateSkillGroup() {
        this.to('a4b:CreateSkillGroup');
        return this;
    }
    /**
     * Creates a user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        this.to('a4b:CreateUser');
        return this;
    }
    /**
     * Deletes an address book by the address book ARN.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DeleteAddressBook.html
     */
    toDeleteAddressBook() {
        this.to('a4b:DeleteAddressBook');
        return this;
    }
    /**
     * Deletes the recurring report delivery schedule with the specified schedule ARN.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DeleteBusinessReportSchedule.html
     */
    toDeleteBusinessReportSchedule() {
        this.to('a4b:DeleteBusinessReportSchedule');
        return this;
    }
    /**
     * Deletes a conference provider.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DeleteConferenceProvider.html
     */
    toDeleteConferenceProvider() {
        this.to('a4b:DeleteConferenceProvider');
        return this;
    }
    /**
     * Deletes a contact by the contact ARN.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DeleteContact.html
     */
    toDeleteContact() {
        this.to('a4b:DeleteContact');
        return this;
    }
    /**
     * Removes a device from Alexa For Business.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DeleteDevice.html
     */
    toDeleteDevice() {
        this.to('a4b:DeleteDevice');
        return this;
    }
    /**
     * Delete profile by profile ARN.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DeleteProfile.html
     */
    toDeleteProfile() {
        this.to('a4b:DeleteProfile');
        return this;
    }
    /**
     * Delete room.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DeleteRoom.html
     */
    toDeleteRoom() {
        this.to('a4b:DeleteRoom');
        return this;
    }
    /**
     * Delete a parameter from a skill and room.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DeleteRoomSkillParameter.html
     */
    toDeleteRoomSkillParameter() {
        this.to('a4b:DeleteRoomSkillParameter');
        return this;
    }
    /**
     * Unlinks a third-party account from a skill.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DeleteSkillAuthorization.html
     */
    toDeleteSkillAuthorization() {
        this.to('a4b:DeleteSkillAuthorization');
        return this;
    }
    /**
     * Deletes skill group with skill group ARN. Skillgroup ARN must be specified.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DeleteSkillGroup.html
     */
    toDeleteSkillGroup() {
        this.to('a4b:DeleteSkillGroup');
        return this;
    }
    /**
     * Delete a user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        this.to('a4b:DeleteUser');
        return this;
    }
    /**
     * Disassociates a contact from a given address book.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DisassociateContactFromAddressBook.html
     */
    toDisassociateContactFromAddressBook() {
        this.to('a4b:DisassociateContactFromAddressBook');
        return this;
    }
    /**
     * Disassociates device from its current room.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DisassociateDeviceFromRoom.html
     */
    toDisassociateDeviceFromRoom() {
        this.to('a4b:DisassociateDeviceFromRoom');
        return this;
    }
    /**
     * Disassociates a skill from a skill group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DisassociateSkillFromSkillGroup.html
     */
    toDisassociateSkillFromSkillGroup() {
        this.to('a4b:DisassociateSkillFromSkillGroup');
        return this;
    }
    /**
     * Makes a private skill unavailable for enrolled users and prevents them from enabling it on their devices.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DisassociateSkillFromUsers.html
     */
    toDisassociateSkillFromUsers() {
        this.to('a4b:DisassociateSkillFromUsers');
        return this;
    }
    /**
     * Disassociates the skill group from given room. SkillGroup ARN and Room ARN must be specified.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_DisassociateSkillGroupFromRoom.html
     */
    toDisassociateSkillGroupFromRoom() {
        this.to('a4b:DisassociateSkillGroupFromRoom');
        return this;
    }
    /**
     * Forgets smart home appliances associated to a room.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_ForgetSmartHomeAppliances.html
     */
    toForgetSmartHomeAppliances() {
        this.to('a4b:ForgetSmartHomeAppliances');
        return this;
    }
    /**
     * Gets the address book details by the address book ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_GetAddressBook.html
     */
    toGetAddressBook() {
        this.to('a4b:GetAddressBook');
        return this;
    }
    /**
     * Retrieves the existing conference preferences.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_GetConferencePreference.html
     */
    toGetConferencePreference() {
        this.to('a4b:GetConferencePreference');
        return this;
    }
    /**
     * Gets details about a specific conference provider.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_GetConferenceProvider.html
     */
    toGetConferenceProvider() {
        this.to('a4b:GetConferenceProvider');
        return this;
    }
    /**
     * Gets the contact details by the contact ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_GetContact.html
     */
    toGetContact() {
        this.to('a4b:GetContact');
        return this;
    }
    /**
     * Get device details.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_GetDevice.html
     */
    toGetDevice() {
        this.to('a4b:GetDevice');
        return this;
    }
    /**
     * Gets the network profile details by the network profile ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_GetNetworkProfile.html
     */
    toGetNetworkProfile() {
        this.to('a4b:GetNetworkProfile');
        return this;
    }
    /**
     * Gets profile when provided with Profile ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_GetProfile.html
     */
    toGetProfile() {
        this.to('a4b:GetProfile');
        return this;
    }
    /**
     * Get room details.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_GetRoom.html
     */
    toGetRoom() {
        this.to('a4b:GetRoom');
        return this;
    }
    /**
     * Get an existing parameter that has been set for a skill and room.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_GetRoomSkillParameter.html
     */
    toGetRoomSkillParameter() {
        this.to('a4b:GetRoomSkillParameter');
        return this;
    }
    /**
     * Gets skill group details with skill group ARN. Skillgroup ARN must be specified.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_GetSkillGroup.html
     */
    toGetSkillGroup() {
        this.to('a4b:GetSkillGroup');
        return this;
    }
    /**
     * Lists the details of the schedules that a user configured.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_ListBusinessReportSchedules.html
     */
    toListBusinessReportSchedules() {
        this.to('a4b:ListBusinessReportSchedules');
        return this;
    }
    /**
     * Lists conference providers under a specific AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_ListConferenceProviders.html
     */
    toListConferenceProviders() {
        this.to('a4b:ListConferenceProviders');
        return this;
    }
    /**
     * Lists the device event history, including device connection status, for up to 30 days.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_ListDeviceEvents.html
     */
    toListDeviceEvents() {
        this.to('a4b:ListDeviceEvents');
        return this;
    }
    /**
     * Lists skills.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_ListSkills.html
     */
    toListSkills() {
        this.to('a4b:ListSkills');
        return this;
    }
    /**
     * Lists all categories in the Alexa skill store.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_ListSkillsStoreCategories.html
     */
    toListSkillsStoreCategories() {
        this.to('a4b:ListSkillsStoreCategories');
        return this;
    }
    /**
     * Lists all skills in the Alexa skill store by category.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_ListSkillsStoreSkillsByCategory.html
     */
    toListSkillsStoreSkillsByCategory() {
        this.to('a4b:ListSkillsStoreSkillsByCategory');
        return this;
    }
    /**
     * Lists all of the smart home appliances associated with a room.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_ListSmartHomeAppliances.html
     */
    toListSmartHomeAppliances() {
        this.to('a4b:ListSmartHomeAppliances');
        return this;
    }
    /**
     * Lists all tags on a resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_ListTags.html
     */
    toListTags() {
        this.to('a4b:ListTags');
        return this;
    }
    /**
     * Sets the conference preferences on a specific conference provider at the account level.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_PutConferencePreference.html
     */
    toPutConferencePreference() {
        this.to('a4b:PutConferencePreference');
        return this;
    }
    /**
     * Publishes Alexa device setup events.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/ag/manage-devices.html
     */
    toPutDeviceSetupEvents() {
        this.to('a4b:PutDeviceSetupEvents');
        return this;
    }
    /**
     * Put a room specific parameter for a skill.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_PutRoomSkillParameter.html
     */
    toPutRoomSkillParameter() {
        this.to('a4b:PutRoomSkillParameter');
        return this;
    }
    /**
     * Links a user's account to a third-party skill provider. If this API operation is called by an assumed IAM role, the skill being linked must be a private skill. Also, the skill must be owned by the AWS account that assumed the IAM role.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_PutSkillAuthorization.html
     */
    toPutSkillAuthorization() {
        this.to('a4b:PutSkillAuthorization');
        return this;
    }
    /**
     * Registers an Alexa-enabled device built by an Original Equipment Manufacturer (OEM) using Alexa Voice Service (AVS).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_RegisterAVSDevice.html
     */
    toRegisterAVSDevice() {
        this.to('a4b:RegisterAVSDevice');
        return this;
    }
    /**
     * Registers an Alexa device.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/ag/manage-devices.html
     */
    toRegisterDevice() {
        this.to('a4b:RegisterDevice');
        return this;
    }
    /**
     * Disassociates a skill from the organization under a user's AWS account. If the skill is a private skill, it moves to an AcceptStatus of PENDING.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_RejectSkill.html
     */
    toRejectSkill() {
        this.to('a4b:RejectSkill');
        return this;
    }
    /**
     * Returns resolved room information.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_ResolveRoom.html
     */
    toResolveRoom() {
        this.to('a4b:ResolveRoom');
        return this;
    }
    /**
     * Revoke an invitation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_RevokeInvitation.html
     */
    toRevokeInvitation() {
        this.to('a4b:RevokeInvitation');
        return this;
    }
    /**
     * Searches address books and lists the ones that meet a set of filter and sort criteria.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_SearchAddressBooks.html
     */
    toSearchAddressBooks() {
        this.to('a4b:SearchAddressBooks');
        return this;
    }
    /**
     * Searches contacts and lists the ones that meet a set of filter and sort criteria.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_SearchContacts.html
     */
    toSearchContacts() {
        this.to('a4b:SearchContacts');
        return this;
    }
    /**
     * Search for devices.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_SearchDevices.html
     */
    toSearchDevices() {
        this.to('a4b:SearchDevices');
        return this;
    }
    /**
     * Searches network profiles and lists the ones that meet a set of filter and sort criteria.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_SearchNetworkProfiles.html
     */
    toSearchNetworkProfiles() {
        this.to('a4b:SearchNetworkProfiles');
        return this;
    }
    /**
     * Search for profiles.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_SearchProfiles.html
     */
    toSearchProfiles() {
        this.to('a4b:SearchProfiles');
        return this;
    }
    /**
     * Search for rooms.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_SearchRooms.html
     */
    toSearchRooms() {
        this.to('a4b:SearchRooms');
        return this;
    }
    /**
     * Search for skill groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_SearchSkillGroups.html
     */
    toSearchSkillGroups() {
        this.to('a4b:SearchSkillGroups');
        return this;
    }
    /**
     * Search for users.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_SearchUsers.html
     */
    toSearchUsers() {
        this.to('a4b:SearchUsers');
        return this;
    }
    /**
     * Send an invitation to a user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_SendInvitation.html
     */
    toSendInvitation() {
        this.to('a4b:SendInvitation');
        return this;
    }
    /**
     * Restore the device and its account to its known, default settings by clearing all information and settings set by its previous users.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_StartDeviceSync.html
     */
    toStartDeviceSync() {
        this.to('a4b:StartDeviceSync');
        return this;
    }
    /**
     * Initiates the discovery of any smart home appliances associated with the room.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_StartSmartHomeApplianceDiscovery.html
     */
    toStartSmartHomeApplianceDiscovery() {
        this.to('a4b:StartSmartHomeApplianceDiscovery');
        return this;
    }
    /**
     * Adds metadata tags to a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('a4b:TagResource');
        return this;
    }
    /**
     * Removes metadata tags from a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('a4b:UntagResource');
        return this;
    }
    /**
     * Updates address book details by the address book ARN.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_UpdateAddressBook.html
     */
    toUpdateAddressBook() {
        this.to('a4b:UpdateAddressBook');
        return this;
    }
    /**
     * Updates the configuration of the report delivery schedule with the specified schedule ARN.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_UpdateBusinessReportSchedule.html
     */
    toUpdateBusinessReportSchedule() {
        this.to('a4b:UpdateBusinessReportSchedule');
        return this;
    }
    /**
     * Updates an existing conference provider's settings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_UpdateConferenceProvider.html
     */
    toUpdateConferenceProvider() {
        this.to('a4b:UpdateConferenceProvider');
        return this;
    }
    /**
     * Updates the contact details by the contact ARN.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_UpdateContact.html
     */
    toUpdateContact() {
        this.to('a4b:UpdateContact');
        return this;
    }
    /**
     * Updates device name.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_UpdateDevice.html
     */
    toUpdateDevice() {
        this.to('a4b:UpdateDevice');
        return this;
    }
    /**
     * Updates an existing profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_UpdateProfile.html
     */
    toUpdateProfile() {
        this.to('a4b:UpdateProfile');
        return this;
    }
    /**
     * Update room details.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_UpdateRoom.html
     */
    toUpdateRoom() {
        this.to('a4b:UpdateRoom');
        return this;
    }
    /**
     * Updates skill group details with skill group ARN. Skillgroup ARN must be specified.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_UpdateSkillGroup.html
     */
    toUpdateSkillGroup() {
        this.to('a4b:UpdateSkillGroup');
        return this;
    }
    /**
     * Adds a resource of type profile to the statement
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_Profile.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProfile(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:a4b:${Region}:${Account}:profile/${Resource_id}';
        arn = arn.replace('${Resource_id}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type room to the statement
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_Room.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRoom(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:a4b:${Region}:${Account}:room/${Resource_id}';
        arn = arn.replace('${Resource_id}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type device to the statement
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_Device.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDevice(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:a4b:${Region}:${Account}:device/${Resource_id}';
        arn = arn.replace('${Resource_id}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type skillgroup to the statement
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_SkillGroup.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSkillgroup(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:a4b:${Region}:${Account}:skill-group/${Resource_id}';
        arn = arn.replace('${Resource_id}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_UserData.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUser(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:a4b:${Region}:${Account}:user/${Resource_id}';
        arn = arn.replace('${Resource_id}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type addressbook to the statement
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_AddressBook.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAddressbook(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:a4b:${Region}:${Account}:address-book/${Resource_id}';
        arn = arn.replace('${Resource_id}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type conferenceprovider to the statement
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_ConferenceProvider.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConferenceprovider(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:a4b:${Region}:${Account}:conference-provider/${Resource_id}';
        arn = arn.replace('${Resource_id}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type contact to the statement
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_Contact.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onContact(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:a4b:${Region}:${Account}:contact/${Resource_id}';
        arn = arn.replace('${Resource_id}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type schedule to the statement
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_BusinessReportSchedule.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSchedule(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:a4b:${Region}:${Account}:schedule/${Resource_id}';
        arn = arn.replace('${Resource_id}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type networkprofile to the statement
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_NetworkProfile.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onNetworkprofile(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:a4b:${Region}:${Account}:network-profile/${Resource_id}';
        arn = arn.replace('${Resource_id}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the Amazon Id in the request
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_RegisterAVSDevice.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAmazonId(value, operator) {
        return this.if(`a4b:amazonId`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the device type in the request
     *
     * https://docs.aws.amazon.com/a4b/latest/APIReference/API_SearchDevices.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFiltersDeviceType(value, operator) {
        return this.if(`a4b:filters_deviceType`, value, operator || 'StringLike');
    }
}
exports.A4b = A4b;
//# sourceMappingURL=data:application/json;base64,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