"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExecuteApi = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [execute-api](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonapigateway.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class ExecuteApi extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [execute-api](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonapigateway.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'execute-api';
        this.accessLevelList = {
            "Write": [
                "InvalidateCache",
                "Invoke",
                "ManageConnections"
            ]
        };
    }
    /**
     * Used to invalidate API cache upon a client request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apigateway/api-reference/api-gateway-caching.html
     */
    toInvalidateCache() {
        this.to('execute-api:InvalidateCache');
        return this;
    }
    /**
     * Used to invoke an API upon a client request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apigateway/api-reference/how-to-call-api.html
     */
    toInvoke() {
        this.to('execute-api:Invoke');
        return this;
    }
    /**
     * ManageConnections controls access to the @connections API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apigateway/api-reference/apigateway-websocket-control-access-iam.html
     */
    toManageConnections() {
        this.to('execute-api:ManageConnections');
        return this;
    }
    /**
     * Adds a resource of type execute-api-general to the statement
     *
     * @param apiId - Identifier for the apiId.
     * @param stage - Identifier for the stage.
     * @param method - Identifier for the method.
     * @param apiSpecificResourcePath - Identifier for the apiSpecificResourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onExecuteApiGeneral(apiId, stage, method, apiSpecificResourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:execute-api:${Region}:${Account}:${ApiId}/${Stage}/${Method}/${ApiSpecificResourcePath}';
        arn = arn.replace('${ApiId}', apiId);
        arn = arn.replace('${Stage}', stage);
        arn = arn.replace('${Method}', method);
        arn = arn.replace('${ApiSpecificResourcePath}', apiSpecificResourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.ExecuteApi = ExecuteApi;
//# sourceMappingURL=data:application/json;base64,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