"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Appmesh = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [appmesh](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsappmesh.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Appmesh extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [appmesh](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsappmesh.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'appmesh';
        this.accessLevelList = {
            "Write": [
                "CreateGatewayRoute",
                "CreateMesh",
                "CreateRoute",
                "CreateVirtualGateway",
                "CreateVirtualNode",
                "CreateVirtualRouter",
                "CreateVirtualService",
                "DeleteGatewayRoute",
                "DeleteMesh",
                "DeleteRoute",
                "DeleteVirtualGateway",
                "DeleteVirtualNode",
                "DeleteVirtualRouter",
                "DeleteVirtualService",
                "TagResource",
                "UntagResource",
                "UpdateGatewayRoute",
                "UpdateMesh",
                "UpdateRoute",
                "UpdateVirtualGateway",
                "UpdateVirtualNode",
                "UpdateVirtualRouter",
                "UpdateVirtualService"
            ],
            "Read": [
                "DescribeGatewayRoute",
                "DescribeMesh",
                "DescribeRoute",
                "DescribeVirtualGateway",
                "DescribeVirtualNode",
                "DescribeVirtualRouter",
                "DescribeVirtualService",
                "StreamAggregatedResources"
            ],
            "List": [
                "ListGatewayRoutes",
                "ListMeshes",
                "ListRoutes",
                "ListTagsForResource",
                "ListVirtualGateways",
                "ListVirtualNodes",
                "ListVirtualRouters",
                "ListVirtualServices"
            ]
        };
    }
    /**
     * Creates a gateway route that is associated with a virtual gateway.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateGatewayRoute.html
     */
    toCreateGatewayRoute() {
        this.to('appmesh:CreateGatewayRoute');
        return this;
    }
    /**
     * Creates a service mesh.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateMesh.html
     */
    toCreateMesh() {
        this.to('appmesh:CreateMesh');
        return this;
    }
    /**
     * Creates a route that is associated with a virtual router.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateRoute.html
     */
    toCreateRoute() {
        this.to('appmesh:CreateRoute');
        return this;
    }
    /**
     * Creates a virtual gateway within a service mesh.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualGateway.html
     */
    toCreateVirtualGateway() {
        this.to('appmesh:CreateVirtualGateway');
        return this;
    }
    /**
     * Creates a virtual node within a service mesh.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualNode.html
     */
    toCreateVirtualNode() {
        this.to('appmesh:CreateVirtualNode');
        return this;
    }
    /**
     * Creates a virtual router within a service mesh.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualRouter.html
     */
    toCreateVirtualRouter() {
        this.to('appmesh:CreateVirtualRouter');
        return this;
    }
    /**
     * Creates a virtual service within a service mesh.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualService.html
     */
    toCreateVirtualService() {
        this.to('appmesh:CreateVirtualService');
        return this;
    }
    /**
     * Deletes an existing gateway route.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteGatewayRoute.html
     */
    toDeleteGatewayRoute() {
        this.to('appmesh:DeleteGatewayRoute');
        return this;
    }
    /**
     * Deletes an existing service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteMesh.html
     */
    toDeleteMesh() {
        this.to('appmesh:DeleteMesh');
        return this;
    }
    /**
     * Deletes an existing route.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteRoute.html
     */
    toDeleteRoute() {
        this.to('appmesh:DeleteRoute');
        return this;
    }
    /**
     * Deletes an existing virtual gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualGateway.html
     */
    toDeleteVirtualGateway() {
        this.to('appmesh:DeleteVirtualGateway');
        return this;
    }
    /**
     * Deletes an existing virtual node.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualNode.html
     */
    toDeleteVirtualNode() {
        this.to('appmesh:DeleteVirtualNode');
        return this;
    }
    /**
     * Deletes an existing virtual router.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualRouter.html
     */
    toDeleteVirtualRouter() {
        this.to('appmesh:DeleteVirtualRouter');
        return this;
    }
    /**
     * Deletes an existing virtual service.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualService.html
     */
    toDeleteVirtualService() {
        this.to('appmesh:DeleteVirtualService');
        return this;
    }
    /**
     * Describes an existing gateway route.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeGatewayRoute.html
     */
    toDescribeGatewayRoute() {
        this.to('appmesh:DescribeGatewayRoute');
        return this;
    }
    /**
     * Describes an existing service mesh.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeMesh.html
     */
    toDescribeMesh() {
        this.to('appmesh:DescribeMesh');
        return this;
    }
    /**
     * Describes an existing route.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeRoute.html
     */
    toDescribeRoute() {
        this.to('appmesh:DescribeRoute');
        return this;
    }
    /**
     * Describes an existing virtual gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualGateway.html
     */
    toDescribeVirtualGateway() {
        this.to('appmesh:DescribeVirtualGateway');
        return this;
    }
    /**
     * Describes an existing virtual node.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualNode.html
     */
    toDescribeVirtualNode() {
        this.to('appmesh:DescribeVirtualNode');
        return this;
    }
    /**
     * Describes an existing virtual router.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualRouter.html
     */
    toDescribeVirtualRouter() {
        this.to('appmesh:DescribeVirtualRouter');
        return this;
    }
    /**
     * Describes an existing virtual service.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualService.html
     */
    toDescribeVirtualService() {
        this.to('appmesh:DescribeVirtualService');
        return this;
    }
    /**
     * Returns a list of existing gateway routes in a service mesh.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListGatewayRoutes.html
     */
    toListGatewayRoutes() {
        this.to('appmesh:ListGatewayRoutes');
        return this;
    }
    /**
     * Returns a list of existing service meshes.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListMeshes.html
     */
    toListMeshes() {
        this.to('appmesh:ListMeshes');
        return this;
    }
    /**
     * Returns a list of existing routes in a service mesh.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListRoutes.html
     */
    toListRoutes() {
        this.to('appmesh:ListRoutes');
        return this;
    }
    /**
     * List the tags for an App Mesh resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('appmesh:ListTagsForResource');
        return this;
    }
    /**
     * Returns a list of existing virtual gateways in a service mesh.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualGateways.html
     */
    toListVirtualGateways() {
        this.to('appmesh:ListVirtualGateways');
        return this;
    }
    /**
     * Returns a list of existing virtual nodes.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualNodes.html
     */
    toListVirtualNodes() {
        this.to('appmesh:ListVirtualNodes');
        return this;
    }
    /**
     * Returns a list of existing virtual routers in a service mesh.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualRouters.html
     */
    toListVirtualRouters() {
        this.to('appmesh:ListVirtualRouters');
        return this;
    }
    /**
     * Returns a list of existing virtual services in a service mesh.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualServices.html
     */
    toListVirtualServices() {
        this.to('appmesh:ListVirtualServices');
        return this;
    }
    /**
     * Allows an Envoy Proxy to receive streamed resources for an App Mesh endpoint (VirtualNode or VirtualGateway).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/envoy.html
     */
    toStreamAggregatedResources() {
        this.to('appmesh:StreamAggregatedResources');
        return this;
    }
    /**
     * Associates the specified tags to a resource with the specified resourceArn.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('appmesh:TagResource');
        return this;
    }
    /**
     * Deletes specified tags from a resource.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('appmesh:UntagResource');
        return this;
    }
    /**
     * Updates an existing gateway route for a specified service mesh and virtual gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateGatewayRoute.html
     */
    toUpdateGatewayRoute() {
        this.to('appmesh:UpdateGatewayRoute');
        return this;
    }
    /**
     * Updates an existing service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateMesh.html
     */
    toUpdateMesh() {
        this.to('appmesh:UpdateMesh');
        return this;
    }
    /**
     * Updates an existing route for a specified service mesh and virtual router.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateRoute.html
     */
    toUpdateRoute() {
        this.to('appmesh:UpdateRoute');
        return this;
    }
    /**
     * Updates an existing virtual gateway in a specified service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualGateway.html
     */
    toUpdateVirtualGateway() {
        this.to('appmesh:UpdateVirtualGateway');
        return this;
    }
    /**
     * Updates an existing virtual node in a specified service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualNode.html
     */
    toUpdateVirtualNode() {
        this.to('appmesh:UpdateVirtualNode');
        return this;
    }
    /**
     * Updates an existing virtual router in a specified service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualRouter.html
     */
    toUpdateVirtualRouter() {
        this.to('appmesh:UpdateVirtualRouter');
        return this;
    }
    /**
     * Updates an existing virtual service in a specified service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualService.html
     */
    toUpdateVirtualService() {
        this.to('appmesh:UpdateVirtualService');
        return this;
    }
    /**
     * Adds a resource of type mesh to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html
     *
     * @param meshName - Identifier for the meshName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMesh(meshName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh:${Region}:${Account}:mesh/${MeshName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type virtualService to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualServiceName - Identifier for the virtualServiceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVirtualService(meshName, virtualServiceName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh:${Region}:${Account}:mesh/${MeshName}/virtualService/${VirtualServiceName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualServiceName}', virtualServiceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type virtualNode to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualNodeName - Identifier for the virtualNodeName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVirtualNode(meshName, virtualNodeName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh:${Region}:${Account}:mesh/${MeshName}/virtualNode/${VirtualNodeName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualNodeName}', virtualNodeName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type virtualRouter to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_routers.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualRouterName - Identifier for the virtualRouterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVirtualRouter(meshName, virtualRouterName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh:${Region}:${Account}:mesh/${MeshName}/virtualRouter/${VirtualRouterName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualRouterName}', virtualRouterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type route to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualRouterName - Identifier for the virtualRouterName.
     * @param routeName - Identifier for the routeName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRoute(meshName, virtualRouterName, routeName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh:${Region}:${Account}:mesh/${MeshName}/virtualRouter/${VirtualRouterName}/route/${RouteName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualRouterName}', virtualRouterName);
        arn = arn.replace('${RouteName}', routeName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type virtualGateway to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualGatewayName - Identifier for the virtualGatewayName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVirtualGateway(meshName, virtualGatewayName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh:${Region}:${Account}:mesh/${MeshName}/virtualGateway/${VirtualGatewayName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualGatewayName}', virtualGatewayName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type gatewayRoute to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualGatewayName - Identifier for the virtualGatewayName.
     * @param gatewayRouteName - Identifier for the gatewayRouteName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGatewayRoute(meshName, virtualGatewayName, gatewayRouteName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh:${Region}:${Account}:mesh/${MeshName}/virtualGateway/${VirtualGatewayName}/gatewayRoute/${GatewayRouteName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualGatewayName}', virtualGatewayName);
        arn = arn.replace('${GatewayRouteName}', gatewayRouteName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Appmesh = Appmesh;
//# sourceMappingURL=data:application/json;base64,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