"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Athena = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [athena](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonathena.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Athena extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [athena](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonathena.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'athena';
        this.accessLevelList = {
            "Read": [
                "BatchGetNamedQuery",
                "BatchGetQueryExecution",
                "GetDataCatalog",
                "GetDatabase",
                "GetNamedQuery",
                "GetQueryExecution",
                "GetQueryResults",
                "GetQueryResultsStream",
                "GetTableMetadata",
                "GetWorkGroup",
                "ListTagsForResource"
            ],
            "Tagging": [
                "CreateDataCatalog",
                "CreateWorkGroup",
                "TagResource",
                "UntagResource"
            ],
            "Write": [
                "CreateNamedQuery",
                "DeleteDataCatalog",
                "DeleteNamedQuery",
                "DeleteWorkGroup",
                "StartQueryExecution",
                "StopQueryExecution",
                "UpdateDataCatalog",
                "UpdateWorkGroup"
            ],
            "List": [
                "ListDataCatalogs",
                "ListDatabases",
                "ListNamedQueries",
                "ListQueryExecutions",
                "ListTableMetadata",
                "ListWorkGroups"
            ]
        };
    }
    /**
     * Grants permissions to get information about one or more named queries.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_BatchGetNamedQuery.html
     */
    toBatchGetNamedQuery() {
        this.to('athena:BatchGetNamedQuery');
        return this;
    }
    /**
     * Grants permissions to get information about one or more query executions.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_BatchGetQueryExecution.html
     */
    toBatchGetQueryExecution() {
        this.to('athena:BatchGetQueryExecution');
        return this;
    }
    /**
     * Grants permissions to create a datacatalog.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_CreateDataCatalog.html
     */
    toCreateDataCatalog() {
        this.to('athena:CreateDataCatalog');
        return this;
    }
    /**
     * Grants permissions to create a named query.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_CreateNamedQuery.html
     */
    toCreateNamedQuery() {
        this.to('athena:CreateNamedQuery');
        return this;
    }
    /**
     * Grants permissions to create a workgroup.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_CreateWorkGroup.html
     */
    toCreateWorkGroup() {
        this.to('athena:CreateWorkGroup');
        return this;
    }
    /**
     * Grants permissions to delete a datacatalog.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_DeleteDataCatalog.html
     */
    toDeleteDataCatalog() {
        this.to('athena:DeleteDataCatalog');
        return this;
    }
    /**
     * Grants permissions to delete a named query specified.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_DeleteNamedQuery.html
     */
    toDeleteNamedQuery() {
        this.to('athena:DeleteNamedQuery');
        return this;
    }
    /**
     * Grants permissions to delete a workgroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_DeleteWorkGroup.html
     */
    toDeleteWorkGroup() {
        this.to('athena:DeleteWorkGroup');
        return this;
    }
    /**
     * Grants permissions to get a datacatalog.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetDataCatalog.html
     */
    toGetDataCatalog() {
        this.to('athena:GetDataCatalog');
        return this;
    }
    /**
     * Grants permissions to get a database for a given datacatalog.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetDatabase.html
     */
    toGetDatabase() {
        this.to('athena:GetDatabase');
        return this;
    }
    /**
     * Grants permissions to get information about the specified named query.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetNamedQuery.html
     */
    toGetNamedQuery() {
        this.to('athena:GetNamedQuery');
        return this;
    }
    /**
     * Grants permissions to get information about the specified query execution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryExecution.html
     */
    toGetQueryExecution() {
        this.to('athena:GetQueryExecution');
        return this;
    }
    /**
     * Grants permissions to get the query results.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryResults.html
     */
    toGetQueryResults() {
        this.to('athena:GetQueryResults');
        return this;
    }
    /**
     * Grants permissions to get the query results stream.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/ug/connect-with-previous-jdbc.html#jdbc-prev-version-policies
     */
    toGetQueryResultsStream() {
        this.to('athena:GetQueryResultsStream');
        return this;
    }
    /**
     * Grants permissions to get a metadata about a table for a given datacatalog.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ GetTableMetadata.html
     */
    toGetTableMetadata() {
        this.to('athena:GetTableMetadata');
        return this;
    }
    /**
     * Grants permissions to get a workgroup.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetWorkGroup.html
     */
    toGetWorkGroup() {
        this.to('athena:GetWorkGroup');
        return this;
    }
    /**
     * Grants permissions to return a list of datacatalogs for the specified AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListDataCatalogs.html
     */
    toListDataCatalogs() {
        this.to('athena:ListDataCatalogs');
        return this;
    }
    /**
     * Grants permissions to return a list of databases for a given datacatalog.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListDatabases.html
     */
    toListDatabases() {
        this.to('athena:ListDatabases');
        return this;
    }
    /**
     * Grants permissions to return a list of named queries in Amazon Athena for the specified AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListNamedQueries.html
     */
    toListNamedQueries() {
        this.to('athena:ListNamedQueries');
        return this;
    }
    /**
     * Grants permissions to return a list of query executions for the specified AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListQueryExecutions.html
     */
    toListQueryExecutions() {
        this.to('athena:ListQueryExecutions');
        return this;
    }
    /**
     * Grants permissions to return a list of table metadata in a database for a given datacatalog.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListTableMetadata.html
     */
    toListTableMetadata() {
        this.to('athena:ListTableMetadata');
        return this;
    }
    /**
     * Grants permissions to return a list of tags for a resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('athena:ListTagsForResource');
        return this;
    }
    /**
     * Grants permissions to return a list of workgroups for the specified AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListWorkGroups.html
     */
    toListWorkGroups() {
        this.to('athena:ListWorkGroups');
        return this;
    }
    /**
     * Grants permissions to start a query execution using an SQL query provided as a string.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_StartQueryExecution.html
     */
    toStartQueryExecution() {
        this.to('athena:StartQueryExecution');
        return this;
    }
    /**
     * Grants permissions to stop the specified query execution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_StopQueryExecution.html
     */
    toStopQueryExecution() {
        this.to('athena:StopQueryExecution');
        return this;
    }
    /**
     * Grants permissions to add a tag to a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('athena:TagResource');
        return this;
    }
    /**
     * Grants permissions to remove a tag from a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('athena:UntagResource');
        return this;
    }
    /**
     * Grants permissions to update a datacatalog.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_UpdateDataCatalog.html
     */
    toUpdateDataCatalog() {
        this.to('athena:UpdateDataCatalog');
        return this;
    }
    /**
     * Grants permissions to update a workgroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_UpdateWorkGroup.html
     */
    toUpdateWorkGroup() {
        this.to('athena:UpdateWorkGroup');
        return this;
    }
    /**
     * Adds a resource of type datacatalog to the statement
     *
     * https://docs.aws.amazon.com/athena/latest/ug/datacatalogs-example-policies.html
     *
     * @param dataCatalogName - Identifier for the dataCatalogName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDatacatalog(dataCatalogName, account, region, partition) {
        var arn = 'arn:${Partition}:athena:${Region}:${Account}:datacatalog/${DataCatalogName}';
        arn = arn.replace('${DataCatalogName}', dataCatalogName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workgroup to the statement
     *
     * https://docs.aws.amazon.com/athena/latest/ug/example-policies-workgroup.html
     *
     * @param workGroupName - Identifier for the workGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkgroup(workGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:athena:${Region}:${Account}:workgroup/${WorkGroupName}';
        arn = arn.replace('${WorkGroupName}', workGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Athena = Athena;
//# sourceMappingURL=data:application/json;base64,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