"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Backup = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [backup](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbackup.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Backup extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [backup](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbackup.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'backup';
        this.accessLevelList = {
            "Write": [
                "CopyFromBackupVault",
                "CopyIntoBackupVault",
                "CreateBackupPlan",
                "CreateBackupSelection",
                "CreateBackupVault",
                "DeleteBackupPlan",
                "DeleteBackupSelection",
                "DeleteBackupVault",
                "DeleteBackupVaultAccessPolicy",
                "DeleteBackupVaultNotifications",
                "DeleteRecoveryPoint",
                "PutBackupVaultAccessPolicy",
                "PutBackupVaultNotifications",
                "StartBackupJob",
                "StartCopyJob",
                "StartRestoreJob",
                "StopBackupJob",
                "UpdateBackupPlan",
                "UpdateGlobalSettings",
                "UpdateRecoveryPointLifecycle",
                "UpdateRegionSettings"
            ],
            "Read": [
                "DescribeBackupJob",
                "DescribeBackupVault",
                "DescribeCopyJob",
                "DescribeGlobalSettings",
                "DescribeProtectedResource",
                "DescribeRecoveryPoint",
                "DescribeRegionSettings",
                "DescribeRestoreJob",
                "ExportBackupPlanTemplate",
                "GetBackupPlan",
                "GetBackupPlanFromJSON",
                "GetBackupPlanFromTemplate",
                "GetBackupSelection",
                "GetBackupVaultAccessPolicy",
                "GetBackupVaultNotifications",
                "GetRecoveryPointRestoreMetadata",
                "GetSupportedResourceTypes"
            ],
            "List": [
                "ListBackupJobs",
                "ListBackupPlanTemplates",
                "ListBackupPlanVersions",
                "ListBackupPlans",
                "ListBackupSelections",
                "ListBackupVaults",
                "ListCopyJobs",
                "ListProtectedResources",
                "ListRecoveryPointsByBackupVault",
                "ListRecoveryPointsByResource",
                "ListRestoreJobs",
                "ListTags"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Allows to copy from a backup vault
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCopyTargets()
     * - .ifCopyTargetOrgPaths()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartCopyJob.html
     */
    toCopyFromBackupVault() {
        this.to('backup:CopyFromBackupVault');
        return this;
    }
    /**
     * Allows to copy into a backup vault
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartCopyJob.html
     */
    toCopyIntoBackupVault() {
        this.to('backup:CopyIntoBackupVault');
        return this;
    }
    /**
     * Creates a new backup plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupPlan.html
     */
    toCreateBackupPlan() {
        this.to('backup:CreateBackupPlan');
        return this;
    }
    /**
     * Creates a new resource assignment in a backup plan.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupSelection.html
     */
    toCreateBackupSelection() {
        this.to('backup:CreateBackupSelection');
        return this;
    }
    /**
     * Creates a new backup vault.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupVault.html
     */
    toCreateBackupVault() {
        this.to('backup:CreateBackupVault');
        return this;
    }
    /**
     * Deletes a backup plan.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupPlan.html
     */
    toDeleteBackupPlan() {
        this.to('backup:DeleteBackupPlan');
        return this;
    }
    /**
     * Deletes a resource assignment from a backup plan.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupSelection.html
     */
    toDeleteBackupSelection() {
        this.to('backup:DeleteBackupSelection');
        return this;
    }
    /**
     * Deletes a backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVault.html
     */
    toDeleteBackupVault() {
        this.to('backup:DeleteBackupVault');
        return this;
    }
    /**
     * Deletes backup vault access policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultAccessPolicy.html
     */
    toDeleteBackupVaultAccessPolicy() {
        this.to('backup:DeleteBackupVaultAccessPolicy');
        return this;
    }
    /**
     * Removes notifications from backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultNotifications.html
     */
    toDeleteBackupVaultNotifications() {
        this.to('backup:DeleteBackupVaultNotifications');
        return this;
    }
    /**
     * Deletes a recovery point from a backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteRecoveryPoint.html
     */
    toDeleteRecoveryPoint() {
        this.to('backup:DeleteRecoveryPoint');
        return this;
    }
    /**
     * Describes a backup job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupJob.html
     */
    toDescribeBackupJob() {
        this.to('backup:DescribeBackupJob');
        return this;
    }
    /**
     * Describes a new backup vault with the specified name.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupVault.html
     */
    toDescribeBackupVault() {
        this.to('backup:DescribeBackupVault');
        return this;
    }
    /**
     * Describes a copy job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeCopyJob.html
     */
    toDescribeCopyJob() {
        this.to('backup:DescribeCopyJob');
        return this;
    }
    /**
     * Describes global settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeGlobalSettings.html
     */
    toDescribeGlobalSettings() {
        this.to('backup:DescribeGlobalSettings');
        return this;
    }
    /**
     * Describes a protected resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeProtectedResource.html
     */
    toDescribeProtectedResource() {
        this.to('backup:DescribeProtectedResource');
        return this;
    }
    /**
     * Describes a recovery point.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRecoveryPoint.html
     */
    toDescribeRecoveryPoint() {
        this.to('backup:DescribeRecoveryPoint');
        return this;
    }
    /**
     * Describes region settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRegionSettings.html
     */
    toDescribeRegionSettings() {
        this.to('backup:DescribeRegionSettings');
        return this;
    }
    /**
     * Describes a restore job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRestoreJob.html
     */
    toDescribeRestoreJob() {
        this.to('backup:DescribeRestoreJob');
        return this;
    }
    /**
     * Exports a backup plan as a JSON.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ExportBackupPlanTemplate.html
     */
    toExportBackupPlanTemplate() {
        this.to('backup:ExportBackupPlanTemplate');
        return this;
    }
    /**
     * Gets a backup plan.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlan.html
     */
    toGetBackupPlan() {
        this.to('backup:GetBackupPlan');
        return this;
    }
    /**
     * Transforms a JSON to a backup plan.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromJSON.html
     */
    toGetBackupPlanFromJSON() {
        this.to('backup:GetBackupPlanFromJSON');
        return this;
    }
    /**
     * Transforms a template to a backup plan.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromTemplate.html
     */
    toGetBackupPlanFromTemplate() {
        this.to('backup:GetBackupPlanFromTemplate');
        return this;
    }
    /**
     * Gets a backup plan resource assignment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupSelection.html
     */
    toGetBackupSelection() {
        this.to('backup:GetBackupSelection');
        return this;
    }
    /**
     * Gets backup vault access policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultAccessPolicy.html
     */
    toGetBackupVaultAccessPolicy() {
        this.to('backup:GetBackupVaultAccessPolicy');
        return this;
    }
    /**
     * Gets backup vault notifications.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultNotifications.html
     */
    toGetBackupVaultNotifications() {
        this.to('backup:GetBackupVaultNotifications');
        return this;
    }
    /**
     * Gets recovery point restore metadata.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetRecoveryPointRestoreMetadata.html
     */
    toGetRecoveryPointRestoreMetadata() {
        this.to('backup:GetRecoveryPointRestoreMetadata');
        return this;
    }
    /**
     * Gets supported resource types.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetSupportedResourceTypes.html
     */
    toGetSupportedResourceTypes() {
        this.to('backup:GetSupportedResourceTypes');
        return this;
    }
    /**
     * Lists backup jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupJobs.html
     */
    toListBackupJobs() {
        this.to('backup:ListBackupJobs');
        return this;
    }
    /**
     * Lists backup plan templates provided by AWS Backup.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanTemplates.html
     */
    toListBackupPlanTemplates() {
        this.to('backup:ListBackupPlanTemplates');
        return this;
    }
    /**
     * Lists backup plan versions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanVersions.html
     */
    toListBackupPlanVersions() {
        this.to('backup:ListBackupPlanVersions');
        return this;
    }
    /**
     * Lists backup plans.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlans.html
     */
    toListBackupPlans() {
        this.to('backup:ListBackupPlans');
        return this;
    }
    /**
     * Lists resource assignments for a specific backup plan.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupSelections.html
     */
    toListBackupSelections() {
        this.to('backup:ListBackupSelections');
        return this;
    }
    /**
     * Lists backup vaults.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupVaults.html
     */
    toListBackupVaults() {
        this.to('backup:ListBackupVaults');
        return this;
    }
    /**
     * Lists copy jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListCopyJobs.html
     */
    toListCopyJobs() {
        this.to('backup:ListCopyJobs');
        return this;
    }
    /**
     * Lists protected resources by AWS Backup.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListProtectedResources.html
     */
    toListProtectedResources() {
        this.to('backup:ListProtectedResources');
        return this;
    }
    /**
     * Lists recovery points inside a backup vault.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByBackupVault.html
     */
    toListRecoveryPointsByBackupVault() {
        this.to('backup:ListRecoveryPointsByBackupVault');
        return this;
    }
    /**
     * Lists recovery points for a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByResource.html
     */
    toListRecoveryPointsByResource() {
        this.to('backup:ListRecoveryPointsByResource');
        return this;
    }
    /**
     * Lists restore jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRestoreJobs.html
     */
    toListRestoreJobs() {
        this.to('backup:ListRestoreJobs');
        return this;
    }
    /**
     * Lists tags for a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListTags.html
     */
    toListTags() {
        this.to('backup:ListTags');
        return this;
    }
    /**
     * Adds an access policy to the backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultAccessPolicy.html
     */
    toPutBackupVaultAccessPolicy() {
        this.to('backup:PutBackupVaultAccessPolicy');
        return this;
    }
    /**
     * Adds an SNS topic to the backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultNotifications.html
     */
    toPutBackupVaultNotifications() {
        this.to('backup:PutBackupVaultNotifications');
        return this;
    }
    /**
     * Starts a new backup job.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartBackupJob.html
     */
    toStartBackupJob() {
        this.to('backup:StartBackupJob');
        return this;
    }
    /**
     * Copy a backup from a source backup vault to a destination backup vault.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartCopyJob.html
     */
    toStartCopyJob() {
        this.to('backup:StartCopyJob');
        return this;
    }
    /**
     * Starts a new restore job.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartRestoreJob.html
     */
    toStartRestoreJob() {
        this.to('backup:StartRestoreJob');
        return this;
    }
    /**
     * Stops a backup job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StopBackupJob.html
     */
    toStopBackupJob() {
        this.to('backup:StopBackupJob');
        return this;
    }
    /**
     * Tags a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_TagResource.html
     */
    toTagResource() {
        this.to('backup:TagResource');
        return this;
    }
    /**
     * Untags a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UntagResource.html
     */
    toUntagResource() {
        this.to('backup:UntagResource');
        return this;
    }
    /**
     * Updates a backup plan.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateBackupPlan.html
     */
    toUpdateBackupPlan() {
        this.to('backup:UpdateBackupPlan');
        return this;
    }
    /**
     * Updates global settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateGlobalSettings.html
     */
    toUpdateGlobalSettings() {
        this.to('backup:UpdateGlobalSettings');
        return this;
    }
    /**
     * Updates the lifecycle of the recovery point.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateRecoveryPointLifecycle.html
     */
    toUpdateRecoveryPointLifecycle() {
        this.to('backup:UpdateRecoveryPointLifecycle');
        return this;
    }
    /**
     * Describes region settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateRegionSettings.html
     */
    toUpdateRegionSettings() {
        this.to('backup:UpdateRegionSettings');
        return this;
    }
    /**
     * Adds a resource of type backupVault to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/vaults.html
     *
     * @param backupVaultName - Identifier for the backupVaultName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBackupVault(backupVaultName, account, region, partition) {
        var arn = 'arn:${Partition}:backup:${Region}:${Account}:backup-vault:${BackupVaultName}';
        arn = arn.replace('${BackupVaultName}', backupVaultName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type backupPlan to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/about-backup-plans.html
     *
     * @param backupPlanId - Identifier for the backupPlanId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBackupPlan(backupPlanId, account, region, partition) {
        var arn = 'arn:${Partition}:backup:${Region}:${Account}:backup-plan:${BackupPlanId}';
        arn = arn.replace('${BackupPlanId}', backupPlanId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type recoveryPoint to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/recovery-points.html
     *
     * @param recoveryPointId - Identifier for the recoveryPointId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRecoveryPoint(recoveryPointId, account, region, partition) {
        var arn = 'arn:${Partition}:backup:${Region}:${Account}:recovery-point:${RecoveryPointId}';
        arn = arn.replace('${RecoveryPointId}', recoveryPointId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the organization unit.
     *
     * Applies to actions:
     * - .toCopyFromBackupVault()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCopyTargetOrgPaths(value, operator) {
        return this.if(`backup:CopyTargetOrgPaths`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the ARN of an backup vault.
     *
     * Applies to actions:
     * - .toCopyFromBackupVault()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCopyTargets(value, operator) {
        return this.if(`backup:CopyTargets`, value, operator || 'StringLike');
    }
}
exports.Backup = Backup;
//# sourceMappingURL=data:application/json;base64,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