"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Chime = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [chime](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonchime.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Chime extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [chime](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonchime.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'chime';
        this.accessLevelList = {
            "Write": [
                "AcceptDelegate",
                "ActivateUsers",
                "AddDomain",
                "AddOrUpdateGroups",
                "AssociatePhoneNumberWithUser",
                "AssociatePhoneNumbersWithVoiceConnector",
                "AssociatePhoneNumbersWithVoiceConnectorGroup",
                "AssociateSigninDelegateGroupsWithAccount",
                "AuthorizeDirectory",
                "BatchCreateAttendee",
                "BatchCreateRoomMembership",
                "BatchDeletePhoneNumber",
                "BatchSuspendUser",
                "BatchUnsuspendUser",
                "BatchUpdatePhoneNumber",
                "BatchUpdateUser",
                "ConnectDirectory",
                "CreateAccount",
                "CreateApiKey",
                "CreateAttendee",
                "CreateBot",
                "CreateBotMembership",
                "CreateCDRBucket",
                "CreateMeeting",
                "CreateMeetingDialOut",
                "CreateMeetingWithAttendees",
                "CreatePhoneNumberOrder",
                "CreateProxySession",
                "CreateRoom",
                "CreateRoomMembership",
                "CreateSipMediaApplication",
                "CreateSipMediaApplicationCall",
                "CreateSipRule",
                "CreateUser",
                "CreateVoiceConnector",
                "CreateVoiceConnectorGroup",
                "DeleteAccount",
                "DeleteAccountOpenIdConfig",
                "DeleteApiKey",
                "DeleteAttendee",
                "DeleteCDRBucket",
                "DeleteDelegate",
                "DeleteDomain",
                "DeleteEventsConfiguration",
                "DeleteGroups",
                "DeleteMeeting",
                "DeletePhoneNumber",
                "DeleteProxySession",
                "DeleteRoom",
                "DeleteRoomMembership",
                "DeleteSipMediaApplication",
                "DeleteSipRule",
                "DeleteVoiceConnector",
                "DeleteVoiceConnectorEmergencyCallingConfiguration",
                "DeleteVoiceConnectorGroup",
                "DeleteVoiceConnectorOrigination",
                "DeleteVoiceConnectorProxy",
                "DeleteVoiceConnectorStreamingConfiguration",
                "DeleteVoiceConnectorTermination",
                "DeleteVoiceConnectorTerminationCredentials",
                "DisassociatePhoneNumberFromUser",
                "DisassociatePhoneNumbersFromVoiceConnector",
                "DisassociatePhoneNumbersFromVoiceConnectorGroup",
                "DisassociateSigninDelegateGroupsFromAccount",
                "DisconnectDirectory",
                "InviteDelegate",
                "InviteUsers",
                "InviteUsersFromProvider",
                "LogoutUser",
                "PutEventsConfiguration",
                "PutRetentionSettings",
                "PutSipMediaApplicationLoggingConfiguration",
                "PutVoiceConnectorEmergencyCallingConfiguration",
                "PutVoiceConnectorLoggingConfiguration",
                "PutVoiceConnectorOrigination",
                "PutVoiceConnectorProxy",
                "PutVoiceConnectorStreamingConfiguration",
                "PutVoiceConnectorTermination",
                "PutVoiceConnectorTerminationCredentials",
                "RedactConversationMessage",
                "RedactRoomMessage",
                "RegenerateSecurityToken",
                "RenameAccount",
                "RenewDelegate",
                "ResetAccountResource",
                "ResetPersonalPIN",
                "RestorePhoneNumber",
                "StartDataExport",
                "SubmitSupportRequest",
                "SuspendUsers",
                "UnauthorizeDirectory",
                "UpdateAccount",
                "UpdateAccountOpenIdConfig",
                "UpdateAccountResource",
                "UpdateAccountSettings",
                "UpdateBot",
                "UpdateCDRSettings",
                "UpdateGlobalSettings",
                "UpdatePhoneNumber",
                "UpdatePhoneNumberSettings",
                "UpdateProxySession",
                "UpdateRoom",
                "UpdateRoomMembership",
                "UpdateSipMediaApplication",
                "UpdateSipRule",
                "UpdateSupportedLicenses",
                "UpdateUser",
                "UpdateUserLicenses",
                "UpdateUserSettings",
                "UpdateVoiceConnector",
                "UpdateVoiceConnectorGroup"
            ],
            "Read": [
                "GetAccount",
                "GetAccountResource",
                "GetAccountSettings",
                "GetAccountWithOpenIdConfig",
                "GetAttendee",
                "GetBot",
                "GetCDRBucket",
                "GetDomain",
                "GetEventsConfiguration",
                "GetGlobalSettings",
                "GetMeeting",
                "GetMeetingDetail",
                "GetPhoneNumber",
                "GetPhoneNumberOrder",
                "GetPhoneNumberSettings",
                "GetProxySession",
                "GetRetentionSettings",
                "GetRoom",
                "GetSipMediaApplication",
                "GetSipMediaApplicationLoggingConfiguration",
                "GetSipRule",
                "GetTelephonyLimits",
                "GetUser",
                "GetUserActivityReportData",
                "GetUserByEmail",
                "GetUserSettings",
                "GetVoiceConnector",
                "GetVoiceConnectorEmergencyCallingConfiguration",
                "GetVoiceConnectorGroup",
                "GetVoiceConnectorLoggingConfiguration",
                "GetVoiceConnectorOrigination",
                "GetVoiceConnectorProxy",
                "GetVoiceConnectorStreamingConfiguration",
                "GetVoiceConnectorTermination",
                "GetVoiceConnectorTerminationHealth",
                "SearchAvailablePhoneNumbers",
                "ValidateAccountResource"
            ],
            "List": [
                "ListAccountUsageReportData",
                "ListAccounts",
                "ListApiKeys",
                "ListAttendeeTags",
                "ListAttendees",
                "ListBots",
                "ListCDRBucket",
                "ListCallingRegions",
                "ListDelegates",
                "ListDirectories",
                "ListDomains",
                "ListGroups",
                "ListMeetingEvents",
                "ListMeetingTags",
                "ListMeetings",
                "ListMeetingsReportData",
                "ListPhoneNumberOrders",
                "ListPhoneNumbers",
                "ListProxySessions",
                "ListRoomMemberships",
                "ListRooms",
                "ListSipMediaApplications",
                "ListSipRules",
                "ListTagsForResource",
                "ListUsers",
                "ListVoiceConnectorGroups",
                "ListVoiceConnectorTerminationCredentials",
                "ListVoiceConnectors",
                "RetrieveDataExports"
            ],
            "Tagging": [
                "TagAttendee",
                "TagMeeting",
                "TagResource",
                "UntagAttendee",
                "UntagMeeting",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to accept the delegate invitation to share management of an Amazon Chime account with another AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toAcceptDelegate() {
        this.to('chime:AcceptDelegate');
        return this;
    }
    /**
     * Grants permission to activate users in an Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    toActivateUsers() {
        this.to('chime:ActivateUsers');
        return this;
    }
    /**
     * Grants permission to add a domain to your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    toAddDomain() {
        this.to('chime:AddDomain');
        return this;
    }
    /**
     * Grants permission to add new or update existing Active Directory or Okta user groups associated with your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html
     */
    toAddOrUpdateGroups() {
        this.to('chime:AddOrUpdateGroups');
        return this;
    }
    /**
     * Grants permission to associate a phone number with an Amazon Chime user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumberWithUser.html
     */
    toAssociatePhoneNumberWithUser() {
        this.to('chime:AssociatePhoneNumberWithUser');
        return this;
    }
    /**
     * Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnector.html
     */
    toAssociatePhoneNumbersWithVoiceConnector() {
        this.to('chime:AssociatePhoneNumbersWithVoiceConnector');
        return this;
    }
    /**
     * Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnectorGroup.html
     */
    toAssociatePhoneNumbersWithVoiceConnectorGroup() {
        this.to('chime:AssociatePhoneNumbersWithVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to associate the specified sign-in delegate groups with the specified Amazon Chime account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociateSigninDelegateGroupsWithAccount.html
     */
    toAssociateSigninDelegateGroupsWithAccount() {
        this.to('chime:AssociateSigninDelegateGroupsWithAccount');
        return this;
    }
    /**
     * Grants permission to authorize an Active Directory for your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toAuthorizeDirectory() {
        this.to('chime:AuthorizeDirectory');
        return this;
    }
    /**
     * Grants permission to create new attendees for an active Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchCreateAttendee.html
     */
    toBatchCreateAttendee() {
        this.to('chime:BatchCreateAttendee');
        return this;
    }
    /**
     * Grants permission to batch add room members
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchCreateRoomMembership.html
     */
    toBatchCreateRoomMembership() {
        this.to('chime:BatchCreateRoomMembership');
        return this;
    }
    /**
     * Grants permission to move up to 50 phone numbers to the deletion queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchDeletePhoneNumber.html
     */
    toBatchDeletePhoneNumber() {
        this.to('chime:BatchDeletePhoneNumber');
        return this;
    }
    /**
     * Grants permission to suspend up to 50 users from a Team or EnterpriseLWA Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchSuspendUser.html
     */
    toBatchSuspendUser() {
        this.to('chime:BatchSuspendUser');
        return this;
    }
    /**
     * Grants permission to remove the suspension from up to 50 previously suspended users for the specified Amazon Chime EnterpriseLWA account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUnsuspendUser.html
     */
    toBatchUnsuspendUser() {
        this.to('chime:BatchUnsuspendUser');
        return this;
    }
    /**
     * Grants permission to update phone number details within the UpdatePhoneNumberRequestItem object for up to 50 phone numbers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdatePhoneNumber.html
     */
    toBatchUpdatePhoneNumber() {
        this.to('chime:BatchUpdatePhoneNumber');
        return this;
    }
    /**
     * Grants permission to update user details within the UpdateUserRequestItem object for up to 20 users for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdateUser.html
     */
    toBatchUpdateUser() {
        this.to('chime:BatchUpdateUser');
        return this;
    }
    /**
     * Grants permission to connect an Active Directory to your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:ConnectDirectory
     *
     * https://docs.aws.amazon.com/chime/latest/ag/active_directory.html
     */
    toConnectDirectory() {
        this.to('chime:ConnectDirectory');
        return this;
    }
    /**
     * Grants permission to create an Amazon Chime account under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAccount.html
     */
    toCreateAccount() {
        this.to('chime:CreateAccount');
        return this;
    }
    /**
     * Grants permission to create a new SCIM access key for your Amazon Chime account and Okta configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toCreateApiKey() {
        this.to('chime:CreateApiKey');
        return this;
    }
    /**
     * Grants permission to create a new attendee for an active Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAttendee.html
     */
    toCreateAttendee() {
        this.to('chime:CreateAttendee');
        return this;
    }
    /**
     * Grants permission to create a bot for an Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateBot.html
     */
    toCreateBot() {
        this.to('chime:CreateBot');
        return this;
    }
    /**
     * Grants permission to add a bot to a chat room in your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateBotMembership.html
     */
    toCreateBotMembership() {
        this.to('chime:CreateBotMembership');
        return this;
    }
    /**
     * Grants permission to create a new Call Detail Record S3 bucket
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:CreateBucket
     * - s3:ListAllMyBuckets
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    toCreateCDRBucket() {
        this.to('chime:CreateCDRBucket');
        return this;
    }
    /**
     * Grants permission to create a new Amazon Chime SDK meeting in the specified media Region, with no initial attendees
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateMeeting.html
     */
    toCreateMeeting() {
        this.to('chime:CreateMeeting');
        return this;
    }
    /**
     * Grants permission to call a phone number to join the specified Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateMeetingDialOut.html
     */
    toCreateMeetingDialOut() {
        this.to('chime:CreateMeetingDialOut');
        return this;
    }
    /**
     * Grants permission to create a new Amazon Chime SDK meeting in the specified media Region, with a set of attendees
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateMeetingWithAttendees.html
     */
    toCreateMeetingWithAttendees() {
        this.to('chime:CreateMeetingWithAttendees');
        return this;
    }
    /**
     * Grants permission to create a phone number order with the Carriers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreatePhoneNumberOrder.html
     */
    toCreatePhoneNumberOrder() {
        this.to('chime:CreatePhoneNumberOrder');
        return this;
    }
    /**
     * Grants permission to create a proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateProxySession.html
     */
    toCreateProxySession() {
        this.to('chime:CreateProxySession');
        return this;
    }
    /**
     * Grants permission to create a room
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateRoom.html
     */
    toCreateRoom() {
        this.to('chime:CreateRoom');
        return this;
    }
    /**
     * Grants permission to add a room member
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateRoomMembership.html
     */
    toCreateRoomMembership() {
        this.to('chime:CreateRoomMembership');
        return this;
    }
    /**
     * Grants permission to create an Amazon Chime SIP media application under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateSipMediaApplication.html
     */
    toCreateSipMediaApplication() {
        this.to('chime:CreateSipMediaApplication');
        return this;
    }
    /**
     * Grants permission to create outbound call for Amazon Chime SIP media application under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateSipMediaApplicationCall.html
     */
    toCreateSipMediaApplicationCall() {
        this.to('chime:CreateSipMediaApplicationCall');
        return this;
    }
    /**
     * Grants permission to create an Amazon Chime SIP rule under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateSipRule.html
     */
    toCreateSipRule() {
        this.to('chime:CreateSipRule');
        return this;
    }
    /**
     * Grants permission to create a user under the specified Amazon Chime account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        this.to('chime:CreateUser');
        return this;
    }
    /**
     * Grants permission to create a Amazon Chime Voice Connector under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnector.html
     */
    toCreateVoiceConnector() {
        this.to('chime:CreateVoiceConnector');
        return this;
    }
    /**
     * Grants permission to create a Amazon Chime Voice Connector Group under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnectorGroup.html
     */
    toCreateVoiceConnectorGroup() {
        this.to('chime:CreateVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAccount.html
     */
    toDeleteAccount() {
        this.to('chime:DeleteAccount');
        return this;
    }
    /**
     * Grants permission to delete the OpenIdConfig attributes from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toDeleteAccountOpenIdConfig() {
        this.to('chime:DeleteAccountOpenIdConfig');
        return this;
    }
    /**
     * Grants permission to delete the specified SCIM access key associated with your Amazon Chime account and Okta configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toDeleteApiKey() {
        this.to('chime:DeleteApiKey');
        return this;
    }
    /**
     * Grants permission to delete the specified attendee from an Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAttendee.html
     */
    toDeleteAttendee() {
        this.to('chime:DeleteAttendee');
        return this;
    }
    /**
     * Grants permission to delete a Call Detail Record S3 bucket from your Amazon Chime account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:DeleteBucket
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toDeleteCDRBucket() {
        this.to('chime:DeleteCDRBucket');
        return this;
    }
    /**
     * Grants permission to delete delegated AWS account management from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toDeleteDelegate() {
        this.to('chime:DeleteDelegate');
        return this;
    }
    /**
     * Grants permission to delete a domain from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    toDeleteDomain() {
        this.to('chime:DeleteDomain');
        return this;
    }
    /**
     * Grants permission to delete an events configuration for a bot to receive outgoing events
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteEventsConfiguration.html
     */
    toDeleteEventsConfiguration() {
        this.to('chime:DeleteEventsConfiguration');
        return this;
    }
    /**
     * Grants permission to delete Active Directory or Okta user groups from your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toDeleteGroups() {
        this.to('chime:DeleteGroups');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteMeeting.html
     */
    toDeleteMeeting() {
        this.to('chime:DeleteMeeting');
        return this;
    }
    /**
     * Grants permission to move a phone number to the deletion queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeletePhoneNumber.html
     */
    toDeletePhoneNumber() {
        this.to('chime:DeletePhoneNumber');
        return this;
    }
    /**
     * Grants permission to delete a proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteProxySession.html
     */
    toDeleteProxySession() {
        this.to('chime:DeleteProxySession');
        return this;
    }
    /**
     * Grants permission to delete a room
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteRoom.html
     */
    toDeleteRoom() {
        this.to('chime:DeleteRoom');
        return this;
    }
    /**
     * Grants permission to remove a room member
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteRoomMembership.html
     */
    toDeleteRoomMembership() {
        this.to('chime:DeleteRoomMembership');
        return this;
    }
    /**
     * Grants permission to delete Amazon Chime SIP media application under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteSipMediaApplication.html
     */
    toDeleteSipMediaApplication() {
        this.to('chime:DeleteSipMediaApplication');
        return this;
    }
    /**
     * Grants permission to delete Amazon Chime SIP rule under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteSipRule.html
     */
    toDeleteSipRule() {
        this.to('chime:DeleteSipRule');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnector.html
     */
    toDeleteVoiceConnector() {
        this.to('chime:DeleteVoiceConnector');
        return this;
    }
    /**
     * Grants permission to delete emergency calling configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorEmergencyCallingConfiguration.html
     */
    toDeleteVoiceConnectorEmergencyCallingConfiguration() {
        this.to('chime:DeleteVoiceConnectorEmergencyCallingConfiguration');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorGroup.html
     */
    toDeleteVoiceConnectorGroup() {
        this.to('chime:DeleteVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to delete the origination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorOrigination.html
     */
    toDeleteVoiceConnectorOrigination() {
        this.to('chime:DeleteVoiceConnectorOrigination');
        return this;
    }
    /**
     * Grants permission to delete proxy configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorProxy.html
     */
    toDeleteVoiceConnectorProxy() {
        this.to('chime:DeleteVoiceConnectorProxy');
        return this;
    }
    /**
     * Grants permission to delete streaming configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorStreamingConfiguration.html
     */
    toDeleteVoiceConnectorStreamingConfiguration() {
        this.to('chime:DeleteVoiceConnectorStreamingConfiguration');
        return this;
    }
    /**
     * Grants permission to delete the termination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTermination.html
     */
    toDeleteVoiceConnectorTermination() {
        this.to('chime:DeleteVoiceConnectorTermination');
        return this;
    }
    /**
     * Grants permission to delete SIP termination credentials for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTerminationCredentials.html
     */
    toDeleteVoiceConnectorTerminationCredentials() {
        this.to('chime:DeleteVoiceConnectorTerminationCredentials');
        return this;
    }
    /**
     * Grants permission to disassociate the primary provisioned number from the specified Amazon Chime user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumberFromUser.html
     */
    toDisassociatePhoneNumberFromUser() {
        this.to('chime:DisassociatePhoneNumberFromUser');
        return this;
    }
    /**
     * Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnector.html
     */
    toDisassociatePhoneNumbersFromVoiceConnector() {
        this.to('chime:DisassociatePhoneNumbersFromVoiceConnector');
        return this;
    }
    /**
     * Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnectorGroup.html
     */
    toDisassociatePhoneNumbersFromVoiceConnectorGroup() {
        this.to('chime:DisassociatePhoneNumbersFromVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to disassociate the specified sign-in delegate groups from the specified Amazon Chime account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociateSigninDelegateGroupsFromAccount.html
     */
    toDisassociateSigninDelegateGroupsFromAccount() {
        this.to('chime:DisassociateSigninDelegateGroupsFromAccount');
        return this;
    }
    /**
     * Grants permission to disconnect the Active Directory from your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toDisconnectDirectory() {
        this.to('chime:DisconnectDirectory');
        return this;
    }
    /**
     * Grants permission to get details for the specified Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccount.html
     */
    toGetAccount() {
        this.to('chime:GetAccount');
        return this;
    }
    /**
     * Grants permission to get details for the account resource associated with your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toGetAccountResource() {
        this.to('chime:GetAccountResource');
        return this;
    }
    /**
     * Grants permission to get account settings for the specified Amazon Chime account ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccountSettings.html
     */
    toGetAccountSettings() {
        this.to('chime:GetAccountSettings');
        return this;
    }
    /**
     * Grants permission to get the account details and OpenIdConfig attributes for your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toGetAccountWithOpenIdConfig() {
        this.to('chime:GetAccountWithOpenIdConfig');
        return this;
    }
    /**
     * Grants permission to get attendee details for a specified meeting ID and attendee ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAttendee.html
     */
    toGetAttendee() {
        this.to('chime:GetAttendee');
        return this;
    }
    /**
     * Grants permission to retrieve details for the specified bot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetBot.html
     */
    toGetBot() {
        this.to('chime:GetBot');
        return this;
    }
    /**
     * Grants permission to get details of a Call Detail Record S3 bucket associated with your Amazon Chime account
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - s3:GetBucketAcl
     * - s3:GetBucketLocation
     * - s3:GetBucketLogging
     * - s3:GetBucketVersioning
     * - s3:GetBucketWebsite
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toGetCDRBucket() {
        this.to('chime:GetCDRBucket');
        return this;
    }
    /**
     * Grants permission to get domain details for a domain associated with your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    toGetDomain() {
        this.to('chime:GetDomain');
        return this;
    }
    /**
     * Grants permission to retrieve details for an events configuration for a bot to receive outgoing events
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetEventsConfiguration.html
     */
    toGetEventsConfiguration() {
        this.to('chime:GetEventsConfiguration');
        return this;
    }
    /**
     * Grants permission to get global settings related to Amazon Chime for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetGlobalSettings.html
     */
    toGetGlobalSettings() {
        this.to('chime:GetGlobalSettings');
        return this;
    }
    /**
     * Grants permission to get the meeting record for a specified meeting ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetMeeting.html
     */
    toGetMeeting() {
        this.to('chime:GetMeeting');
        return this;
    }
    /**
     * Grants permission to get attendee, connection, and other details for a meeting
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toGetMeetingDetail() {
        this.to('chime:GetMeetingDetail');
        return this;
    }
    /**
     * Grants permission to get details for the specified phone number
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumber.html
     */
    toGetPhoneNumber() {
        this.to('chime:GetPhoneNumber');
        return this;
    }
    /**
     * Grants permission to get details for the specified phone number order
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberOrder.html
     */
    toGetPhoneNumberOrder() {
        this.to('chime:GetPhoneNumberOrder');
        return this;
    }
    /**
     * Grants permission to get phone number settings related to Amazon Chime for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberSettings.html
     */
    toGetPhoneNumberSettings() {
        this.to('chime:GetPhoneNumberSettings');
        return this;
    }
    /**
     * Grants permission to get details of the specified proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetProxySession.html
     */
    toGetProxySession() {
        this.to('chime:GetProxySession');
        return this;
    }
    /**
     * Gets the retention settings for the specified Amazon Chime account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetRetentionSettings.html
     */
    toGetRetentionSettings() {
        this.to('chime:GetRetentionSettings');
        return this;
    }
    /**
     * Grants permission to retrieve a room
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetRoom.html
     */
    toGetRoom() {
        this.to('chime:GetRoom');
        return this;
    }
    /**
     * Grants permission to get details of Amazon Chime SIP media application under the administrator's AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetSipMediaApplication.html
     */
    toGetSipMediaApplication() {
        this.to('chime:GetSipMediaApplication');
        return this;
    }
    /**
     * Grants permission to get logging configuration settings for Amazon Chime SIP media application under the administrator's AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetSipMediaApplicationLoggingConfiguration.html
     */
    toGetSipMediaApplicationLoggingConfiguration() {
        this.to('chime:GetSipMediaApplicationLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details of Amazon Chime SIP rule under the administrator's AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetSipRule.html
     */
    toGetSipRule() {
        this.to('chime:GetSipRule');
        return this;
    }
    /**
     * Grants permission to get telephony limits for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html
     */
    toGetTelephonyLimits() {
        this.to('chime:GetTelephonyLimits');
        return this;
    }
    /**
     * Grants permission to get details for the specified user ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUser.html
     */
    toGetUser() {
        this.to('chime:GetUser');
        return this;
    }
    /**
     * Grants permission to get a summary of user activity on the user details page
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/user-details.html
     */
    toGetUserActivityReportData() {
        this.to('chime:GetUserActivityReportData');
        return this;
    }
    /**
     * Grants permission to get user details for an Amazon Chime user based on the email address in an Amazon Chime Enterprise or Team account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/user-details.html
     */
    toGetUserByEmail() {
        this.to('chime:GetUserByEmail');
        return this;
    }
    /**
     * Grants permission to get user settings related to the specified Amazon Chime user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUserSettings.html
     */
    toGetUserSettings() {
        this.to('chime:GetUserSettings');
        return this;
    }
    /**
     * Grants permission to get details for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnector.html
     */
    toGetVoiceConnector() {
        this.to('chime:GetVoiceConnector');
        return this;
    }
    /**
     * Grants permission to get details of the emergency calling configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorEmergencyCallingConfiguration.html
     */
    toGetVoiceConnectorEmergencyCallingConfiguration() {
        this.to('chime:GetVoiceConnectorEmergencyCallingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details for the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorGroup.html
     */
    toGetVoiceConnectorGroup() {
        this.to('chime:GetVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to get details of the logging configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorLoggingConfiguration.html
     */
    toGetVoiceConnectorLoggingConfiguration() {
        this.to('chime:GetVoiceConnectorLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details of the origination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorOrigination.html
     */
    toGetVoiceConnectorOrigination() {
        this.to('chime:GetVoiceConnectorOrigination');
        return this;
    }
    /**
     * Grants permission to get details of the proxy configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorProxy.html
     */
    toGetVoiceConnectorProxy() {
        this.to('chime:GetVoiceConnectorProxy');
        return this;
    }
    /**
     * Grants permission to get details of the streaming configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorStreamingConfiguration.html
     */
    toGetVoiceConnectorStreamingConfiguration() {
        this.to('chime:GetVoiceConnectorStreamingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details of the termination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTermination.html
     */
    toGetVoiceConnectorTermination() {
        this.to('chime:GetVoiceConnectorTermination');
        return this;
    }
    /**
     * Grants permission to get details of the termination health for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTerminationHealth.html
     */
    toGetVoiceConnectorTerminationHealth() {
        this.to('chime:GetVoiceConnectorTerminationHealth');
        return this;
    }
    /**
     * Grants permission to send an invitation to accept a request for AWS account delegation for an Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toInviteDelegate() {
        this.to('chime:InviteDelegate');
        return this;
    }
    /**
     * Grants permission to invite as many as 50 users to the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_InviteUsers.html
     */
    toInviteUsers() {
        this.to('chime:InviteUsers');
        return this;
    }
    /**
     * Grants permission to invite users from a third party provider to your Amazon Chime account
     *
     * Access Level: Write
     */
    toInviteUsersFromProvider() {
        this.to('chime:InviteUsersFromProvider');
        return this;
    }
    /**
     * Grants permission to list Amazon Chime account usage reporting data
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/view-reports.html
     */
    toListAccountUsageReportData() {
        this.to('chime:ListAccountUsageReportData');
        return this;
    }
    /**
     * Grants permission to list the Amazon Chime accounts under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAccounts.html
     */
    toListAccounts() {
        this.to('chime:ListAccounts');
        return this;
    }
    /**
     * Grants permission to list the SCIM access keys defined for your Amazon Chime account and Okta configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toListApiKeys() {
        this.to('chime:ListApiKeys');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon Chime SDK attendee resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendeeTags.html
     */
    toListAttendeeTags() {
        this.to('chime:ListAttendeeTags');
        return this;
    }
    /**
     * Grants permission to list up to 100 attendees for a specified Amazon Chime SDK meeting
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendees.html
     */
    toListAttendees() {
        this.to('chime:ListAttendees');
        return this;
    }
    /**
     * Grants permission to list the bots associated with the administrator's Amazon Chime Enterprise account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListBots.html
     */
    toListBots() {
        this.to('chime:ListBots');
        return this;
    }
    /**
     * Grants permission to list Call Detail Record S3 buckets
     *
     * Access Level: List
     *
     * Dependent actions:
     * - s3:ListAllMyBuckets
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toListCDRBucket() {
        this.to('chime:ListCDRBucket');
        return this;
    }
    /**
     * Grants permission to list the calling regions available for the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html
     */
    toListCallingRegions() {
        this.to('chime:ListCallingRegions');
        return this;
    }
    /**
     * Grants permission to list account delegate information associated with your Amazon Chime account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toListDelegates() {
        this.to('chime:ListDelegates');
        return this;
    }
    /**
     * Grants permission to list active Active Directories hosted in the Directory Service of your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toListDirectories() {
        this.to('chime:ListDirectories');
        return this;
    }
    /**
     * Grants permission to list domains associated with your Amazon Chime account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    toListDomains() {
        this.to('chime:ListDomains');
        return this;
    }
    /**
     * Grants permission to list Active Directory or Okta user groups associated with your Amazon Chime Enterprise account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toListGroups() {
        this.to('chime:ListGroups');
        return this;
    }
    /**
     * Grants permission to list all events that occurred for a specified meeting
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/view-reports.html
     */
    toListMeetingEvents() {
        this.to('chime:ListMeetingEvents');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetingTags.html
     */
    toListMeetingTags() {
        this.to('chime:ListMeetingTags');
        return this;
    }
    /**
     * Grants permission to list up to 100 active Amazon Chime SDK meetings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetings.html
     */
    toListMeetings() {
        this.to('chime:ListMeetings');
        return this;
    }
    /**
     * Grants permission to list meetings ended during the specified date range
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/view-reports.html
     */
    toListMeetingsReportData() {
        this.to('chime:ListMeetingsReportData');
        return this;
    }
    /**
     * Grants permission to list the phone number orders under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumberOrders.html
     */
    toListPhoneNumberOrders() {
        this.to('chime:ListPhoneNumberOrders');
        return this;
    }
    /**
     * Grants permission to list the phone numbers under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumbers.html
     */
    toListPhoneNumbers() {
        this.to('chime:ListPhoneNumbers');
        return this;
    }
    /**
     * Grants permission to list proxy sessions for the specified Amazon Chime Voice Connector
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListProxySessions.html
     */
    toListProxySessions() {
        this.to('chime:ListProxySessions');
        return this;
    }
    /**
     * Grants permission to list all room members
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListRoomMemberships.html
     */
    toListRoomMemberships() {
        this.to('chime:ListRoomMemberships');
        return this;
    }
    /**
     * Grants permission to list rooms
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListRooms.html
     */
    toListRooms() {
        this.to('chime:ListRooms');
        return this;
    }
    /**
     * Grants permission to list all Amazon Chime SIP media applications under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListSipMediaApplications.html
     */
    toListSipMediaApplications() {
        this.to('chime:ListSipMediaApplications');
        return this;
    }
    /**
     * Grants permission to list all Amazon Chime SIP rules under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListSipRules.html
     */
    toListSipRules() {
        this.to('chime:ListSipRules');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('chime:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list the users that belong to the specified Amazon Chime account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListUsers.html
     */
    toListUsers() {
        this.to('chime:ListUsers');
        return this;
    }
    /**
     * Grants permission to list the Amazon Chime Voice Connector Groups under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorGroups.html
     */
    toListVoiceConnectorGroups() {
        this.to('chime:ListVoiceConnectorGroups');
        return this;
    }
    /**
     * Grants permission to list the SIP termination credentials for the specified Amazon Chime Voice Connector
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorTerminationCredentials.html
     */
    toListVoiceConnectorTerminationCredentials() {
        this.to('chime:ListVoiceConnectorTerminationCredentials');
        return this;
    }
    /**
     * Grants permission to list the Amazon Chime Voice Connectors under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectors.html
     */
    toListVoiceConnectors() {
        this.to('chime:ListVoiceConnectors');
        return this;
    }
    /**
     * Grants permission to log out the specified user from all of the devices they are currently logged into
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_LogoutUser.html
     */
    toLogoutUser() {
        this.to('chime:LogoutUser');
        return this;
    }
    /**
     * Grants permission to update details for an events configuration for a bot to receive outgoing events
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutEventsConfiguration.html
     */
    toPutEventsConfiguration() {
        this.to('chime:PutEventsConfiguration');
        return this;
    }
    /**
     * Puts retention settings for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutRetentionSettings.html
     */
    toPutRetentionSettings() {
        this.to('chime:PutRetentionSettings');
        return this;
    }
    /**
     * Grants permission to update logging configuration settings for Amazon Chime SIP media application under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutSipMediaApplicationLoggingConfiguration.html
     */
    toPutSipMediaApplicationLoggingConfiguration() {
        this.to('chime:PutSipMediaApplicationLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to add emergency calling configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorEmergencyCallingConfiguration.html
     */
    toPutVoiceConnectorEmergencyCallingConfiguration() {
        this.to('chime:PutVoiceConnectorEmergencyCallingConfiguration');
        return this;
    }
    /**
     * Grants permission to add logging configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - logs:CreateLogDelivery
     * - logs:CreateLogGroup
     * - logs:DeleteLogDelivery
     * - logs:DescribeLogGroups
     * - logs:GetLogDelivery
     * - logs:ListLogDeliveries
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorLoggingConfiguration.html
     */
    toPutVoiceConnectorLoggingConfiguration() {
        this.to('chime:PutVoiceConnectorLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to update the origination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorOrigination.html
     */
    toPutVoiceConnectorOrigination() {
        this.to('chime:PutVoiceConnectorOrigination');
        return this;
    }
    /**
     * Grants permission to add proxy configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorProxy.html
     */
    toPutVoiceConnectorProxy() {
        this.to('chime:PutVoiceConnectorProxy');
        return this;
    }
    /**
     * Grants permission to add streaming configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorStreamingConfiguration.html
     */
    toPutVoiceConnectorStreamingConfiguration() {
        this.to('chime:PutVoiceConnectorStreamingConfiguration');
        return this;
    }
    /**
     * Grants permission to update the termination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTermination.html
     */
    toPutVoiceConnectorTermination() {
        this.to('chime:PutVoiceConnectorTermination');
        return this;
    }
    /**
     * Grants permission to add SIP termination credentials for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTerminationCredentials.html
     */
    toPutVoiceConnectorTerminationCredentials() {
        this.to('chime:PutVoiceConnectorTerminationCredentials');
        return this;
    }
    /**
     * Redacts the specified Chime conversation Message
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_RedactConversationMessage.html
     */
    toRedactConversationMessage() {
        this.to('chime:RedactConversationMessage');
        return this;
    }
    /**
     * Redacts the specified Chime room Message
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_RedactRoomMessage.html
     */
    toRedactRoomMessage() {
        this.to('chime:RedactRoomMessage');
        return this;
    }
    /**
     * Grants permission to regenerate the security token for the specified bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_RegenerateSecurityToken.html
     */
    toRegenerateSecurityToken() {
        this.to('chime:RegenerateSecurityToken');
        return this;
    }
    /**
     * Grants permission to modify the account name for your Amazon Chime Enterprise or Team account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/rename-account.html
     */
    toRenameAccount() {
        this.to('chime:RenameAccount');
        return this;
    }
    /**
     * Grants permission to renew the delegation request associated with an Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toRenewDelegate() {
        this.to('chime:RenewDelegate');
        return this;
    }
    /**
     * Grants permission to reset the account resource in your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toResetAccountResource() {
        this.to('chime:ResetAccountResource');
        return this;
    }
    /**
     * Grants permission to reset the personal meeting PIN for the specified user on an Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ResetPersonalPIN.html
     */
    toResetPersonalPIN() {
        this.to('chime:ResetPersonalPIN');
        return this;
    }
    /**
     * Grants permission to restore the specified phone number from the deltion queue back to the phone number inventory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_RestorePhoneNumber.html
     */
    toRestorePhoneNumber() {
        this.to('chime:RestorePhoneNumber');
        return this;
    }
    /**
     * Grants permission to download the file containing links to all user attachments returned as part of the "Request attachments" action
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html
     */
    toRetrieveDataExports() {
        this.to('chime:RetrieveDataExports');
        return this;
    }
    /**
     * Grants permission to search phone numbers that can be ordered from the carrier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_SearchAvailablePhoneNumbers.html
     */
    toSearchAvailablePhoneNumbers() {
        this.to('chime:SearchAvailablePhoneNumbers');
        return this;
    }
    /**
     * Grants permission to submit the "Request attachments" request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html
     */
    toStartDataExport() {
        this.to('chime:StartDataExport');
        return this;
    }
    /**
     * Grants permission to submit a customer service support request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/chime-getting-admin-support.html
     */
    toSubmitSupportRequest() {
        this.to('chime:SubmitSupportRequest');
        return this;
    }
    /**
     * Grants permission to suspend users from an Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    toSuspendUsers() {
        this.to('chime:SuspendUsers');
        return this;
    }
    /**
     * Grants permission to apply the specified tags to the specified Amazon Chime SDK attendee
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_TagAttendee.html
     */
    toTagAttendee() {
        this.to('chime:TagAttendee');
        return this;
    }
    /**
     * Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_TagMeeting.html
     */
    toTagMeeting() {
        this.to('chime:TagMeeting');
        return this;
    }
    /**
     * Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('chime:TagResource');
        return this;
    }
    /**
     * Grants permission to unauthorize an Active Directory from your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toUnauthorizeDirectory() {
        this.to('chime:UnauthorizeDirectory');
        return this;
    }
    /**
     * Grants permission to untag the specified tags from the specified Amazon Chime SDK attendee.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagAttendee.html
     */
    toUntagAttendee() {
        this.to('chime:UntagAttendee');
        return this;
    }
    /**
     * Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagMeeting.html
     */
    toUntagMeeting() {
        this.to('chime:UntagMeeting');
        return this;
    }
    /**
     * Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('chime:UntagResource');
        return this;
    }
    /**
     * Grants permission to update account details for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccount.html
     */
    toUpdateAccount() {
        this.to('chime:UpdateAccount');
        return this;
    }
    /**
     * Grants permission to update the OpenIdConfig attributes for your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toUpdateAccountOpenIdConfig() {
        this.to('chime:UpdateAccountOpenIdConfig');
        return this;
    }
    /**
     * Grants permission to update the account resource in your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toUpdateAccountResource() {
        this.to('chime:UpdateAccountResource');
        return this;
    }
    /**
     * Grants permission to update the settings for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccountSettings.html
     */
    toUpdateAccountSettings() {
        this.to('chime:UpdateAccountSettings');
        return this;
    }
    /**
     * Grants permission to update the status of the specified bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateBot.html
     */
    toUpdateBot() {
        this.to('chime:UpdateBot');
        return this;
    }
    /**
     * Grants permission to update your Call Detail Record S3 bucket
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:CreateBucket
     * - s3:DeleteBucket
     * - s3:ListAllMyBuckets
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toUpdateCDRSettings() {
        this.to('chime:UpdateCDRSettings');
        return this;
    }
    /**
     * Grants permission to update the global settings related to Amazon Chime for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateGlobalSettings.html
     */
    toUpdateGlobalSettings() {
        this.to('chime:UpdateGlobalSettings');
        return this;
    }
    /**
     * Grants permission to update phone number details for the specified phone number
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumber.html
     */
    toUpdatePhoneNumber() {
        this.to('chime:UpdatePhoneNumber');
        return this;
    }
    /**
     * Grants permission to update phone number settings related to Amazon Chime for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumberSettings.html
     */
    toUpdatePhoneNumberSettings() {
        this.to('chime:UpdatePhoneNumberSettings');
        return this;
    }
    /**
     * Grants permission to update a proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateProxySession.html
     */
    toUpdateProxySession() {
        this.to('chime:UpdateProxySession');
        return this;
    }
    /**
     * Grants permission to update a room
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateRoom.html
     */
    toUpdateRoom() {
        this.to('chime:UpdateRoom');
        return this;
    }
    /**
     * Grants permission to update room membership role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateRoomMembership.html
     */
    toUpdateRoomMembership() {
        this.to('chime:UpdateRoomMembership');
        return this;
    }
    /**
     * Grants permission to update properties of Amazon Chime SIP media application under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateSipMediaApplication.html
     */
    toUpdateSipMediaApplication() {
        this.to('chime:UpdateSipMediaApplication');
        return this;
    }
    /**
     * Grants permission to update properties of Amazon Chime SIP rule under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateSipRule.html
     */
    toUpdateSipRule() {
        this.to('chime:UpdateSipRule');
        return this;
    }
    /**
     * Grants permission to update the supported license tiers available for users in your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    toUpdateSupportedLicenses() {
        this.to('chime:UpdateSupportedLicenses');
        return this;
    }
    /**
     * Grants permission to update user details for a specified user ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUser.html
     */
    toUpdateUser() {
        this.to('chime:UpdateUser');
        return this;
    }
    /**
     * Grants permission to update the licenses for your Amazon Chime users
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    toUpdateUserLicenses() {
        this.to('chime:UpdateUserLicenses');
        return this;
    }
    /**
     * Grants permission to update user settings related to the specified Amazon Chime user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUserSettings.html
     */
    toUpdateUserSettings() {
        this.to('chime:UpdateUserSettings');
        return this;
    }
    /**
     * Grants permission to update Amazon Chime Voice Connector details for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnector.html
     */
    toUpdateVoiceConnector() {
        this.to('chime:UpdateVoiceConnector');
        return this;
    }
    /**
     * Grants permission to update Amazon Chime Voice Connector Group details for the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnectorGroup.html
     */
    toUpdateVoiceConnectorGroup() {
        this.to('chime:UpdateVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to validate the account resource in your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toValidateAccountResource() {
        this.to('chime:ValidateAccountResource');
        return this;
    }
    /**
     * Adds a resource of type meeting to the statement
     *
     * @param meetingId - Identifier for the meetingId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMeeting(meetingId, accountId, partition) {
        var arn = 'arn:${Partition}:chime::${AccountId}:meeting/${MeetingId}';
        arn = arn.replace('${MeetingId}', meetingId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Chime = Chime;
//# sourceMappingURL=data:application/json;base64,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