"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudhsm = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cloudhsm](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudhsm.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudhsm extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cloudhsm](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudhsm.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cloudhsm';
        this.accessLevelList = {
            "Tagging": [
                "AddTagsToResource",
                "RemoveTagsFromResource",
                "TagResource",
                "UntagResource"
            ],
            "Write": [
                "CopyBackupToRegion",
                "CreateCluster",
                "CreateHapg",
                "CreateHsm",
                "CreateLunaClient",
                "DeleteBackup",
                "DeleteCluster",
                "DeleteHapg",
                "DeleteHsm",
                "DeleteLunaClient",
                "InitializeCluster",
                "ModifyHapg",
                "ModifyHsm",
                "ModifyLunaClient",
                "RestoreBackup"
            ],
            "Read": [
                "DescribeBackups",
                "DescribeClusters",
                "DescribeHapg",
                "DescribeHsm",
                "DescribeLunaClient",
                "GetConfig",
                "ListTags",
                "ListTagsForResource"
            ],
            "List": [
                "ListAvailableZones",
                "ListHapgs",
                "ListHsms",
                "ListLunaClients"
            ]
        };
    }
    /**
     * Adds or overwrites one or more tags for the specified AWS CloudHSM resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        this.to('cloudhsm:AddTagsToResource');
        return this;
    }
    /**
     * Creates a copy of a backup in the specified region
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_CopyBackupToRegion.html
     */
    toCopyBackupToRegion() {
        this.to('cloudhsm:CopyBackupToRegion');
        return this;
    }
    /**
     * Creates a new AWS CloudHSM cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_CreateCluster.html
     */
    toCreateCluster() {
        this.to('cloudhsm:CreateCluster');
        return this;
    }
    /**
     * Creates a high-availability partition group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_CreateHapg.html
     */
    toCreateHapg() {
        this.to('cloudhsm:CreateHapg');
        return this;
    }
    /**
     * Creates a new hardware security module (HSM) in the specified AWS CloudHSM cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_CreateHsm.html
     */
    toCreateHsm() {
        this.to('cloudhsm:CreateHsm');
        return this;
    }
    /**
     * Creates an HSM client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_CreateLunaClient.html
     */
    toCreateLunaClient() {
        this.to('cloudhsm:CreateLunaClient');
        return this;
    }
    /**
     * Deletes the specified CloudHSM backup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DeleteBackup.html
     */
    toDeleteBackup() {
        this.to('cloudhsm:DeleteBackup');
        return this;
    }
    /**
     * Deletes the specified AWS CloudHSM cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DeleteCluster.html
     */
    toDeleteCluster() {
        this.to('cloudhsm:DeleteCluster');
        return this;
    }
    /**
     * Deletes a high-availability partition group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_DeleteHapg.html
     */
    toDeleteHapg() {
        this.to('cloudhsm:DeleteHapg');
        return this;
    }
    /**
     * Deletes the specified HSM
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DeleteHsm.html
     */
    toDeleteHsm() {
        this.to('cloudhsm:DeleteHsm');
        return this;
    }
    /**
     * Deletes a client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_DeleteLunaClient.html
     */
    toDeleteLunaClient() {
        this.to('cloudhsm:DeleteLunaClient');
        return this;
    }
    /**
     * Gets information about backups of AWS CloudHSM clusters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DescribeBackups.html
     */
    toDescribeBackups() {
        this.to('cloudhsm:DescribeBackups');
        return this;
    }
    /**
     * Gets information about AWS CloudHSM clusters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DescribeClusters.html
     */
    toDescribeClusters() {
        this.to('cloudhsm:DescribeClusters');
        return this;
    }
    /**
     * Retrieves information about a high-availability partition group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_DescribeHapg.html
     */
    toDescribeHapg() {
        this.to('cloudhsm:DescribeHapg');
        return this;
    }
    /**
     * Retrieves information about an HSM. You can identify the HSM by its ARN or its serial number
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_DescribeHsm.html
     */
    toDescribeHsm() {
        this.to('cloudhsm:DescribeHsm');
        return this;
    }
    /**
     * Retrieves information about an HSM client
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_DescribeLunaClient.html
     */
    toDescribeLunaClient() {
        this.to('cloudhsm:DescribeLunaClient');
        return this;
    }
    /**
     * Gets the configuration files necessary to connect to all high availability partition groups the client is associated with
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_GetConfig.html
     */
    toGetConfig() {
        this.to('cloudhsm:GetConfig');
        return this;
    }
    /**
     * Claims an AWS CloudHSM cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_InitializeCluster.html
     */
    toInitializeCluster() {
        this.to('cloudhsm:InitializeCluster');
        return this;
    }
    /**
     * Lists the Availability Zones that have available AWS CloudHSM capacity
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ListAvailableZones.html
     */
    toListAvailableZones() {
        this.to('cloudhsm:ListAvailableZones');
        return this;
    }
    /**
     * Lists the high-availability partition groups for the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ListHapgs.html
     */
    toListHapgs() {
        this.to('cloudhsm:ListHapgs');
        return this;
    }
    /**
     * Retrieves the identifiers of all of the HSMs provisioned for the current customer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ListHsms.html
     */
    toListHsms() {
        this.to('cloudhsm:ListHsms');
        return this;
    }
    /**
     * Lists all of the clients
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ListLunaClients.html
     */
    toListLunaClients() {
        this.to('cloudhsm:ListLunaClients');
        return this;
    }
    /**
     * Gets a list of tags for the specified AWS CloudHSM cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_ListTags.html
     */
    toListTags() {
        this.to('cloudhsm:ListTags');
        return this;
    }
    /**
     * Returns a list of all tags for the specified AWS CloudHSM resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('cloudhsm:ListTagsForResource');
        return this;
    }
    /**
     * Modifies an existing high-availability partition group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ModifyHapg.html
     */
    toModifyHapg() {
        this.to('cloudhsm:ModifyHapg');
        return this;
    }
    /**
     * Modifies an HSM
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ModifyHsm.html
     */
    toModifyHsm() {
        this.to('cloudhsm:ModifyHsm');
        return this;
    }
    /**
     * Modifies the certificate used by the client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ModifyLunaClient.html
     */
    toModifyLunaClient() {
        this.to('cloudhsm:ModifyLunaClient');
        return this;
    }
    /**
     * Removes one or more tags from the specified AWS CloudHSM resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        this.to('cloudhsm:RemoveTagsFromResource');
        return this;
    }
    /**
     * Restores the specified CloudHSM backup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_RestoreBackup.html
     */
    toRestoreBackup() {
        this.to('cloudhsm:RestoreBackup');
        return this;
    }
    /**
     * Adds or overwrites one or more tags for the specified AWS CloudHSM cluster
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('cloudhsm:TagResource');
        return this;
    }
    /**
     * Removes the specified tag or tags from the specified AWS CloudHSM cluster
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('cloudhsm:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type backup to the statement
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/userguide/backups.html
     *
     * @param cloudHsmBackupInstanceName - Identifier for the cloudHsmBackupInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBackup(cloudHsmBackupInstanceName, account, region, partition) {
        var arn = 'arn:${Partition}:cloudhsm:${Region}:${Account}:backup/${CloudHsmBackupInstanceName}';
        arn = arn.replace('${CloudHsmBackupInstanceName}', cloudHsmBackupInstanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/userguide/clusters.html
     *
     * @param cloudHsmClusterInstanceName - Identifier for the cloudHsmClusterInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(cloudHsmClusterInstanceName, account, region, partition) {
        var arn = 'arn:${Partition}:cloudhsm:${Region}:${Account}:cluster/${CloudHsmClusterInstanceName}';
        arn = arn.replace('${CloudHsmClusterInstanceName}', cloudHsmClusterInstanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Cloudhsm = Cloudhsm;
//# sourceMappingURL=data:application/json;base64,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