"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Synthetics = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [synthetics](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudwatchsynthetics.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Synthetics extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [synthetics](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudwatchsynthetics.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'synthetics';
        this.accessLevelList = {
            "Write": [
                "CreateCanary",
                "DeleteCanary",
                "StartCanary",
                "StopCanary",
                "TagResource",
                "UntagResource",
                "UpdateCanary"
            ],
            "Read": [
                "DescribeCanaries",
                "DescribeCanariesLastRun",
                "GetCanaryRuns",
                "ListTagsForResource"
            ]
        };
    }
    /**
     * Create a canary.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_CreateCanary.html
     */
    toCreateCanary() {
        this.to('synthetics:CreateCanary');
        return this;
    }
    /**
     * Deletes a canary. Amazon Synthetics deletes all the resources except for the Lambda function and the CloudWatch Alarms if you created one.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_DeleteCanary.html
     */
    toDeleteCanary() {
        this.to('synthetics:DeleteCanary');
        return this;
    }
    /**
     * Returns information of all canaries.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_DescribeCanaries.html
     */
    toDescribeCanaries() {
        this.to('synthetics:DescribeCanaries');
        return this;
    }
    /**
     * Returns information about the last test run associated with all canaries.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_DescribeCanariesLastRun.html
     */
    toDescribeCanariesLastRun() {
        this.to('synthetics:DescribeCanariesLastRun');
        return this;
    }
    /**
     * Returns information about all the test runs associated with a canary.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_GetCanaryRuns.html
     */
    toGetCanaryRuns() {
        this.to('synthetics:GetCanaryRuns');
        return this;
    }
    /**
     * Returns a list of all tags and values associated with a canary.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('synthetics:ListTagsForResource');
        return this;
    }
    /**
     * Starts a canary, so that Amazon Synthetics starts monitoring a website.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_StartCanary.html
     */
    toStartCanary() {
        this.to('synthetics:StartCanary');
        return this;
    }
    /**
     * Stops a canary.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_StopCanary.html
     */
    toStopCanary() {
        this.to('synthetics:StopCanary');
        return this;
    }
    /**
     * Adds one or more tags to a canary.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('synthetics:TagResource');
        return this;
    }
    /**
     * Removes one or more tags from a canary.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('synthetics:UntagResource');
        return this;
    }
    /**
     * Updates a canary.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_UpdateCanary.html
     */
    toUpdateCanary() {
        this.to('synthetics:UpdateCanary');
        return this;
    }
    /**
     * Adds a resource of type canary to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries.html
     *
     * @param canaryName - Identifier for the canaryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCanary(canaryName, account, partition) {
        var arn = 'arn:${Partition}:synthetics::${Account}:canary:${CanaryName}';
        arn = arn.replace('${CanaryName}', canaryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Synthetics = Synthetics;
//# sourceMappingURL=data:application/json;base64,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