"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeguruProfiler = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codeguru-profiler](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncodeguruprofiler.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CodeguruProfiler extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codeguru-profiler](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncodeguruprofiler.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codeguru-profiler';
        this.accessLevelList = {
            "Write": [
                "AddNotificationChannels",
                "ConfigureAgent",
                "CreateProfilingGroup",
                "DeleteProfilingGroup",
                "PostAgentProfile",
                "RemoveNotificationChannel",
                "SubmitFeedback",
                "UpdateProfilingGroup"
            ],
            "List": [
                "BatchGetFrameMetricData",
                "ListFindingsReports",
                "ListProfileTimes",
                "ListProfilingGroups"
            ],
            "Read": [
                "DescribeProfilingGroup",
                "GetFindingsReportAccountSummary",
                "GetNotificationConfiguration",
                "GetPolicy",
                "GetProfile",
                "GetRecommendations"
            ],
            "Tagging": [
                "ListTagsForResource",
                "TagResource",
                "UntagResource"
            ],
            "Permissions management": [
                "PutPermission",
                "RemovePermission"
            ]
        };
    }
    /**
     * Grants permission to add up to 2 topic ARNs of existing AWS SNS topics to publish notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_AddNotificationChannels.html
     */
    toAddNotificationChannels() {
        this.to('codeguru-profiler:AddNotificationChannels');
        return this;
    }
    /**
     * Grants permission to get the frame metric data for a Profiling Group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_BatchGetFrameMetricData.html
     */
    toBatchGetFrameMetricData() {
        this.to('codeguru-profiler:BatchGetFrameMetricData');
        return this;
    }
    /**
     * Grants permission for an agent to register with the orchestration service and retrieve profiling configuration information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ConfigureAgent.html
     */
    toConfigureAgent() {
        this.to('codeguru-profiler:ConfigureAgent');
        return this;
    }
    /**
     * Grants permission to create a profiling group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_CreateProfilingGroup.html
     */
    toCreateProfilingGroup() {
        this.to('codeguru-profiler:CreateProfilingGroup');
        return this;
    }
    /**
     * Grants permission to delete a profiling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_DeleteProfilingGroup.html
     */
    toDeleteProfilingGroup() {
        this.to('codeguru-profiler:DeleteProfilingGroup');
        return this;
    }
    /**
     * Grants permission to describe a profiling group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_DescribeProfilingGroup.html
     */
    toDescribeProfilingGroup() {
        this.to('codeguru-profiler:DescribeProfilingGroup');
        return this;
    }
    /**
     * Grants permission to get a summary of recent recommendations for each profiling group in the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetFindingsReportAccountSummary.html
     */
    toGetFindingsReportAccountSummary() {
        this.to('codeguru-profiler:GetFindingsReportAccountSummary');
        return this;
    }
    /**
     * Grants permission to get the notification configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetNotificationConfiguration.html
     */
    toGetNotificationConfiguration() {
        this.to('codeguru-profiler:GetNotificationConfiguration');
        return this;
    }
    /**
     * Grants permission to get the resource policy associated with the specified Profiling Group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetPolicy.html
     */
    toGetPolicy() {
        this.to('codeguru-profiler:GetPolicy');
        return this;
    }
    /**
     * Grants permission to get aggregated profiles for a specific profiling group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetProfile.html
     */
    toGetProfile() {
        this.to('codeguru-profiler:GetProfile');
        return this;
    }
    /**
     * Grants permission to get recommendations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetRecommendations.html
     */
    toGetRecommendations() {
        this.to('codeguru-profiler:GetRecommendations');
        return this;
    }
    /**
     * Grants permission to list the available recommendations reports for a specific profiling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListFindingsReports.html
     */
    toListFindingsReports() {
        this.to('codeguru-profiler:ListFindingsReports');
        return this;
    }
    /**
     * Grants permission to list the start times of the available aggregated profiles for a specific profiling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListProfileTimes.html
     */
    toListProfileTimes() {
        this.to('codeguru-profiler:ListProfileTimes');
        return this;
    }
    /**
     * Grants permission to list profiling groups in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListProfilingGroups.html
     */
    toListProfilingGroups() {
        this.to('codeguru-profiler:ListProfilingGroups');
        return this;
    }
    /**
     * Grants permission to list tags for a Profiling Group
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('codeguru-profiler:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to submit a profile collected by an agent belonging to a specific profiling group for aggregation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_PostAgentProfile.html
     */
    toPostAgentProfile() {
        this.to('codeguru-profiler:PostAgentProfile');
        return this;
    }
    /**
     * Grants permission to update the list of principals allowed for an action group in the resource policy associated with the specified Profiling Group.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_PutPermission.html
     */
    toPutPermission() {
        this.to('codeguru-profiler:PutPermission');
        return this;
    }
    /**
     * Grants permission to delete an already configured SNStopic arn from the notification configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_RemoveNotificationChannel.html
     */
    toRemoveNotificationChannel() {
        this.to('codeguru-profiler:RemoveNotificationChannel');
        return this;
    }
    /**
     * Grants permission to remove the permission of specified Action Group from the resource policy associated with the specified Profiling Group.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_RemovePermission.html
     */
    toRemovePermission() {
        this.to('codeguru-profiler:RemovePermission');
        return this;
    }
    /**
     * Grants permission to submit user feedback for useful or non useful anomaly
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_SubmitFeedback.html
     */
    toSubmitFeedback() {
        this.to('codeguru-profiler:SubmitFeedback');
        return this;
    }
    /**
     * Grants permission to add or overwrite tags to a Profiling Group
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_TagResource.html
     */
    toTagResource() {
        this.to('codeguru-profiler:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a Profiling Group
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_UntagResource.html
     */
    toUntagResource() {
        this.to('codeguru-profiler:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a specific profiling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_UpdateProfilingGroup.html
     */
    toUpdateProfilingGroup() {
        this.to('codeguru-profiler:UpdateProfilingGroup');
        return this;
    }
    /**
     * Adds a resource of type ProfilingGroup to the statement
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-ug/working-with-profiling-groups.html
     *
     * @param profilingGroupName - Identifier for the profilingGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProfilingGroup(profilingGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:codeguru-profiler:${Region}:${Account}:profilingGroup/${ProfilingGroupName}';
        arn = arn.replace('${ProfilingGroupName}', profilingGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.CodeguruProfiler = CodeguruProfiler;
//# sourceMappingURL=data:application/json;base64,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