"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Comprehend = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [comprehend](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncomprehend.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Comprehend extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [comprehend](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncomprehend.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'comprehend';
        this.accessLevelList = {
            "Read": [
                "BatchDetectDominantLanguage",
                "BatchDetectEntities",
                "BatchDetectKeyPhrases",
                "BatchDetectSentiment",
                "BatchDetectSyntax",
                "ClassifyDocument",
                "DescribeDocumentClassificationJob",
                "DescribeDocumentClassifier",
                "DescribeDominantLanguageDetectionJob",
                "DescribeEndpoint",
                "DescribeEntitiesDetectionJob",
                "DescribeEntityRecognizer",
                "DescribeKeyPhrasesDetectionJob",
                "DescribePiiEntitiesDetectionJob",
                "DescribeSentimentDetectionJob",
                "DescribeTopicsDetectionJob",
                "DetectDominantLanguage",
                "DetectEntities",
                "DetectKeyPhrases",
                "DetectPiiEntities",
                "DetectSentiment",
                "DetectSyntax"
            ],
            "Write": [
                "CreateDocumentClassifier",
                "CreateEndpoint",
                "CreateEntityRecognizer",
                "DeleteDocumentClassifier",
                "DeleteEndpoint",
                "DeleteEntityRecognizer",
                "StartDocumentClassificationJob",
                "StartDominantLanguageDetectionJob",
                "StartEntitiesDetectionJob",
                "StartKeyPhrasesDetectionJob",
                "StartPiiEntitiesDetectionJob",
                "StartSentimentDetectionJob",
                "StartTopicsDetectionJob",
                "StopDominantLanguageDetectionJob",
                "StopEntitiesDetectionJob",
                "StopKeyPhrasesDetectionJob",
                "StopPiiEntitiesDetectionJob",
                "StopSentimentDetectionJob",
                "StopTrainingDocumentClassifier",
                "StopTrainingEntityRecognizer",
                "UpdateEndpoint"
            ],
            "List": [
                "ListDocumentClassificationJobs",
                "ListDocumentClassifiers",
                "ListDominantLanguageDetectionJobs",
                "ListEndpoints",
                "ListEntitiesDetectionJobs",
                "ListEntityRecognizers",
                "ListKeyPhrasesDetectionJobs",
                "ListPiiEntitiesDetectionJobs",
                "ListSentimentDetectionJobs",
                "ListTagsForResource",
                "ListTopicsDetectionJobs"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to detect the language or languages present in the list of text documents
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectDominantLanguage.html
     */
    toBatchDetectDominantLanguage() {
        this.to('comprehend:BatchDetectDominantLanguage');
        return this;
    }
    /**
     * Grants permission to detect the named entities ("People", "Places", "Locations", etc) within the given list of text documents
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectEntities.html
     */
    toBatchDetectEntities() {
        this.to('comprehend:BatchDetectEntities');
        return this;
    }
    /**
     * Grants permission to detect the phrases in the list of text documents that are most indicative of the content
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectKeyPhrases.html
     */
    toBatchDetectKeyPhrases() {
        this.to('comprehend:BatchDetectKeyPhrases');
        return this;
    }
    /**
     * Grants permission to detect the sentiment of a text in the list of documents (Positive, Negative, Neutral, or Mixed)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectSentiment.html
     */
    toBatchDetectSentiment() {
        this.to('comprehend:BatchDetectSentiment');
        return this;
    }
    /**
     * Grants permission to detect syntactic information (like Part of Speech, Tokens) in a list of text documents
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectSyntax.html
     */
    toBatchDetectSyntax() {
        this.to('comprehend:BatchDetectSyntax');
        return this;
    }
    /**
     * Grants permission to create a new document classification request to analyze a single document in real-time, using a previously created and trained custom model and an endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ClassifyDocument.html
     */
    toClassifyDocument() {
        this.to('comprehend:ClassifyDocument');
        return this;
    }
    /**
     * Grants permission to create a new document classifier that you can use to categorize documents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVolumeKmsKey()
     * - .ifOutputKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateDocumentClassifier.html
     */
    toCreateDocumentClassifier() {
        this.to('comprehend:CreateDocumentClassifier');
        return this;
    }
    /**
     * Grants permission to create a model-specific endpoint for synchronous inference for a previously trained custom model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateEndpoint.html
     */
    toCreateEndpoint() {
        this.to('comprehend:CreateEndpoint');
        return this;
    }
    /**
     * Grants permission to create an entity recognizer using submitted files
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateEntityRecognizer.html
     */
    toCreateEntityRecognizer() {
        this.to('comprehend:CreateEntityRecognizer');
        return this;
    }
    /**
     * Grants permission to delete a previously created document classifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteDocumentClassifier.html
     */
    toDeleteDocumentClassifier() {
        this.to('comprehend:DeleteDocumentClassifier');
        return this;
    }
    /**
     * Grants permission to delete a model-specific endpoint for a previously-trained custom model. All endpoints must be deleted in order for the model to be deleted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteEndpoint.html
     */
    toDeleteEndpoint() {
        this.to('comprehend:DeleteEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a submitted entity recognizer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteEntityRecognizer.html
     */
    toDeleteEntityRecognizer() {
        this.to('comprehend:DeleteEntityRecognizer');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a document classification job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDocumentClassificationJob.html
     */
    toDescribeDocumentClassificationJob() {
        this.to('comprehend:DescribeDocumentClassificationJob');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a document classifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDocumentClassifier.html
     */
    toDescribeDocumentClassifier() {
        this.to('comprehend:DescribeDocumentClassifier');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a dominant language detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDominantLanguageDetectionJob.html
     */
    toDescribeDominantLanguageDetectionJob() {
        this.to('comprehend:DescribeDominantLanguageDetectionJob');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a specific endpoint. Use this operation to get the status of an endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEndpoint.html
     */
    toDescribeEndpoint() {
        this.to('comprehend:DescribeEndpoint');
        return this;
    }
    /**
     * Grants permission to get the properties associated with an entities detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEntitiesDetectionJob.html
     */
    toDescribeEntitiesDetectionJob() {
        this.to('comprehend:DescribeEntitiesDetectionJob');
        return this;
    }
    /**
     * Grants permission to provide details about an entity recognizer including status, S3 buckets containing training data, recognizer metadata, metrics, and so on
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEntityRecognizer.html
     */
    toDescribeEntityRecognizer() {
        this.to('comprehend:DescribeEntityRecognizer');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a key phrases detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeKeyPhrasesDetectionJob.html
     */
    toDescribeKeyPhrasesDetectionJob() {
        this.to('comprehend:DescribeKeyPhrasesDetectionJob');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a PII entities detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribePiiEntitiesDetectionJob.html
     */
    toDescribePiiEntitiesDetectionJob() {
        this.to('comprehend:DescribePiiEntitiesDetectionJob');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a sentiment detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeSentimentDetectionJob.html
     */
    toDescribeSentimentDetectionJob() {
        this.to('comprehend:DescribeSentimentDetectionJob');
        return this;
    }
    /**
     * Grants permission to get the properties associated with a topic detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeTopicsDetectionJob.html
     */
    toDescribeTopicsDetectionJob() {
        this.to('comprehend:DescribeTopicsDetectionJob');
        return this;
    }
    /**
     * Grants permission to detect the language or languages present in the text
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectDominantLanguage.html
     */
    toDetectDominantLanguage() {
        this.to('comprehend:DetectDominantLanguage');
        return this;
    }
    /**
     * Grants permission to detect the named entities ("People", "Places", "Locations", etc) within the given text document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectEntities.html
     */
    toDetectEntities() {
        this.to('comprehend:DetectEntities');
        return this;
    }
    /**
     * Grants permission to detect the phrases in the text that are most indicative of the content
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectKeyPhrases.html
     */
    toDetectKeyPhrases() {
        this.to('comprehend:DetectKeyPhrases');
        return this;
    }
    /**
     * Grants permission to detect the personally identifiable information entities ("Name", "SSN", "PIN", etc) within the given text document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectPiiEntities.html
     */
    toDetectPiiEntities() {
        this.to('comprehend:DetectPiiEntities');
        return this;
    }
    /**
     * Grants permission to detect the sentiment of a text in a document (Positive, Negative, Neutral, or Mixed)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectSentiment.html
     */
    toDetectSentiment() {
        this.to('comprehend:DetectSentiment');
        return this;
    }
    /**
     * Grants permission to detect syntactic information (like Part of Speech, Tokens) in a text document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectSyntax.html
     */
    toDetectSyntax() {
        this.to('comprehend:DetectSyntax');
        return this;
    }
    /**
     * Grants permission to get a list of the document classification jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDocumentClassificationJobs.html
     */
    toListDocumentClassificationJobs() {
        this.to('comprehend:ListDocumentClassificationJobs');
        return this;
    }
    /**
     * Grants permission to get a list of the document classifiers that you have created
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDocumentClassifiers.html
     */
    toListDocumentClassifiers() {
        this.to('comprehend:ListDocumentClassifiers');
        return this;
    }
    /**
     * Grants permission to get a list of the dominant language detection jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDominantLanguageDetectionJobs.html
     */
    toListDominantLanguageDetectionJobs() {
        this.to('comprehend:ListDominantLanguageDetectionJobs');
        return this;
    }
    /**
     * Grants permission to get a list of all existing endpoints that you've created
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEndpoints.html
     */
    toListEndpoints() {
        this.to('comprehend:ListEndpoints');
        return this;
    }
    /**
     * Grants permission to get a list of the entity detection jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEntitiesDetectionJobs.html
     */
    toListEntitiesDetectionJobs() {
        this.to('comprehend:ListEntitiesDetectionJobs');
        return this;
    }
    /**
     * Grants permission to get a list of the properties of all entity recognizers that you created, including recognizers currently in training
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEntityRecognizers.html
     */
    toListEntityRecognizers() {
        this.to('comprehend:ListEntityRecognizers');
        return this;
    }
    /**
     * Grants permission to get a list of key phrase detection jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListKeyPhrasesDetectionJobs.html
     */
    toListKeyPhrasesDetectionJobs() {
        this.to('comprehend:ListKeyPhrasesDetectionJobs');
        return this;
    }
    /**
     * Grants permission to get a list of PII entities detection jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListPiiEntitiesDetectionJobs.html
     */
    toListPiiEntitiesDetectionJobs() {
        this.to('comprehend:ListPiiEntitiesDetectionJobs');
        return this;
    }
    /**
     * Grants permission to get a list of sentiment detection jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListSentimentDetectionJobs.html
     */
    toListSentimentDetectionJobs() {
        this.to('comprehend:ListSentimentDetectionJobs');
        return this;
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('comprehend:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to get a list of the topic detection jobs that you have submitted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListTopicsDetectionJobs.html
     */
    toListTopicsDetectionJobs() {
        this.to('comprehend:ListTopicsDetectionJobs');
        return this;
    }
    /**
     * Grants permission to start an asynchronous document classification job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVolumeKmsKey()
     * - .ifOutputKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartDocumentClassificationJob.html
     */
    toStartDocumentClassificationJob() {
        this.to('comprehend:StartDocumentClassificationJob');
        return this;
    }
    /**
     * Grants permission to start an asynchronous dominant language detection job for a collection of documents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVolumeKmsKey()
     * - .ifOutputKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartDominantLanguageDetectionJob.html
     */
    toStartDominantLanguageDetectionJob() {
        this.to('comprehend:StartDominantLanguageDetectionJob');
        return this;
    }
    /**
     * Grants permission to start an asynchronous entity detection job for a collection of documents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVolumeKmsKey()
     * - .ifOutputKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartEntitiesDetectionJob.html
     */
    toStartEntitiesDetectionJob() {
        this.to('comprehend:StartEntitiesDetectionJob');
        return this;
    }
    /**
     * Grants permission to start an asynchronous key phrase detection job for a collection of documents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVolumeKmsKey()
     * - .ifOutputKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartKeyPhrasesDetectionJob.html
     */
    toStartKeyPhrasesDetectionJob() {
        this.to('comprehend:StartKeyPhrasesDetectionJob');
        return this;
    }
    /**
     * Grants permission to start an asynchronous PII entities detection job for a collection of documents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifOutputKmsKey()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartPiiEntitiesDetectionJob.html
     */
    toStartPiiEntitiesDetectionJob() {
        this.to('comprehend:StartPiiEntitiesDetectionJob');
        return this;
    }
    /**
     * Grants permission to start an asynchronous sentiment detection job for a collection of documents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVolumeKmsKey()
     * - .ifOutputKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartSentimentDetectionJob.html
     */
    toStartSentimentDetectionJob() {
        this.to('comprehend:StartSentimentDetectionJob');
        return this;
    }
    /**
     * Grants permission to start an asynchronous job to detect the most common topics in the collection of documents and the phrases associated with each topic
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVolumeKmsKey()
     * - .ifOutputKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartTopicsDetectionJob.html
     */
    toStartTopicsDetectionJob() {
        this.to('comprehend:StartTopicsDetectionJob');
        return this;
    }
    /**
     * Grants permission to stop a dominant language detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopDominantLanguageDetectionJob.html
     */
    toStopDominantLanguageDetectionJob() {
        this.to('comprehend:StopDominantLanguageDetectionJob');
        return this;
    }
    /**
     * Grants permission to stop an entity detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopEntitiesDetectionJob.html
     */
    toStopEntitiesDetectionJob() {
        this.to('comprehend:StopEntitiesDetectionJob');
        return this;
    }
    /**
     * Grants permission to stop a key phrase detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopKeyPhrasesDetectionJob.html
     */
    toStopKeyPhrasesDetectionJob() {
        this.to('comprehend:StopKeyPhrasesDetectionJob');
        return this;
    }
    /**
     * Grants permission to stop a PII entities detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopPiiEntitiesDetectionJob.html
     */
    toStopPiiEntitiesDetectionJob() {
        this.to('comprehend:StopPiiEntitiesDetectionJob');
        return this;
    }
    /**
     * Grants permission to stop a sentiment detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopSentimentDetectionJob.html
     */
    toStopSentimentDetectionJob() {
        this.to('comprehend:StopSentimentDetectionJob');
        return this;
    }
    /**
     * Grants permission to stop a previously created document classifier training job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopTrainingDocumentClassifier.html
     */
    toStopTrainingDocumentClassifier() {
        this.to('comprehend:StopTrainingDocumentClassifier');
        return this;
    }
    /**
     * Grants permission to stop a previously created entity recognizer training job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopTrainingEntityRecognizer.html
     */
    toStopTrainingEntityRecognizer() {
        this.to('comprehend:StopTrainingEntityRecognizer');
        return this;
    }
    /**
     * Grants permission to tag a resource with given key value pairs
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_TagResource.html
     */
    toTagResource() {
        this.to('comprehend:TagResource');
        return this;
    }
    /**
     * Grants permission to untag a resource with given key
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        this.to('comprehend:UntagResource');
        return this;
    }
    /**
     * Grants permission to update information about the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_UpdateEndpoint.html
     */
    toUpdateEndpoint() {
        this.to('comprehend:UpdateEndpoint');
        return this;
    }
    /**
     * Adds a resource of type document-classifier to the statement
     *
     * @param documentClassifierName - Identifier for the documentClassifierName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentClassifier(documentClassifierName, account, region, partition) {
        var arn = 'arn:${Partition}:comprehend:${Region}:${Account}:document-classifier/${DocumentClassifierName}';
        arn = arn.replace('${DocumentClassifierName}', documentClassifierName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type document-classifier-endpoint to the statement
     *
     * @param documentClassifierEndpointName - Identifier for the documentClassifierEndpointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentClassifierEndpoint(documentClassifierEndpointName, account, region, partition) {
        var arn = 'arn:${Partition}:comprehend:${Region}:${Account}:document-classifier-endpoint/${DocumentClassifierEndpointName}';
        arn = arn.replace('${DocumentClassifierEndpointName}', documentClassifierEndpointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type entity-recognizer to the statement
     *
     * @param entityRecognizerName - Identifier for the entityRecognizerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEntityRecognizer(entityRecognizerName, account, region, partition) {
        var arn = 'arn:${Partition}:comprehend:${Region}:${Account}:entity-recognizer/${EntityRecognizerName}';
        arn = arn.replace('${EntityRecognizerName}', entityRecognizerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type entity-recognizer-endpoint to the statement
     *
     * @param entityRecognizerEndpointName - Identifier for the entityRecognizerEndpointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEntityRecognizerEndpoint(entityRecognizerEndpointName, account, region, partition) {
        var arn = 'arn:${Partition}:comprehend:${Region}:${Account}:entity-recognizer-endpoint/${EntityRecognizerEndpointName}';
        arn = arn.replace('${EntityRecognizerEndpointName}', entityRecognizerEndpointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The output KMS key associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html#amazoncomprehend-policy-keys
     *
     * Applies to actions:
     * - .toCreateDocumentClassifier()
     * - .toStartDocumentClassificationJob()
     * - .toStartDominantLanguageDetectionJob()
     * - .toStartEntitiesDetectionJob()
     * - .toStartKeyPhrasesDetectionJob()
     * - .toStartPiiEntitiesDetectionJob()
     * - .toStartSentimentDetectionJob()
     * - .toStartTopicsDetectionJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifOutputKmsKey(value, operator) {
        return this.if(`comprehend:OutputKmsKey`, value, operator || 'ArnLike');
    }
    /**
     * The volume KMS key associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html#amazoncomprehend-policy-keys
     *
     * Applies to actions:
     * - .toCreateDocumentClassifier()
     * - .toCreateEntityRecognizer()
     * - .toStartDocumentClassificationJob()
     * - .toStartDominantLanguageDetectionJob()
     * - .toStartEntitiesDetectionJob()
     * - .toStartKeyPhrasesDetectionJob()
     * - .toStartSentimentDetectionJob()
     * - .toStartTopicsDetectionJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifVolumeKmsKey(value, operator) {
        return this.if(`comprehend:VolumeKmsKey`, value, operator || 'ArnLike');
    }
    /**
     * The list of all VPC security group ids associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html#amazoncomprehend-policy-keys
     *
     * Applies to actions:
     * - .toCreateDocumentClassifier()
     * - .toCreateEntityRecognizer()
     * - .toStartDocumentClassificationJob()
     * - .toStartDominantLanguageDetectionJob()
     * - .toStartEntitiesDetectionJob()
     * - .toStartKeyPhrasesDetectionJob()
     * - .toStartSentimentDetectionJob()
     * - .toStartTopicsDetectionJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcSecurityGroupIds(value, operator) {
        return this.if(`comprehend:VpcSecurityGroupIds`, value, operator || 'StringLike');
    }
    /**
     * The list of all VPC subnets associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html#amazoncomprehend-policy-keys
     *
     * Applies to actions:
     * - .toCreateDocumentClassifier()
     * - .toCreateEntityRecognizer()
     * - .toStartDocumentClassificationJob()
     * - .toStartDominantLanguageDetectionJob()
     * - .toStartEntitiesDetectionJob()
     * - .toStartKeyPhrasesDetectionJob()
     * - .toStartSentimentDetectionJob()
     * - .toStartTopicsDetectionJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcSubnets(value, operator) {
        return this.if(`comprehend:VpcSubnets`, value, operator || 'StringLike');
    }
}
exports.Comprehend = Comprehend;
//# sourceMappingURL=data:application/json;base64,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