"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Datasync = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [datasync](https://docs.aws.amazon.com/service-authorization/latest/reference/list_datasync.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Datasync extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [datasync](https://docs.aws.amazon.com/service-authorization/latest/reference/list_datasync.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'datasync';
        this.accessLevelList = {
            "Write": [
                "CancelTaskExecution",
                "CreateAgent",
                "CreateLocationEfs",
                "CreateLocationFsxWindows",
                "CreateLocationNfs",
                "CreateLocationObjectStorage",
                "CreateLocationS3",
                "CreateLocationSmb",
                "CreateTask",
                "DeleteAgent",
                "DeleteLocation",
                "DeleteTask",
                "StartTaskExecution",
                "TagResource",
                "UpdateAgent",
                "UpdateTask"
            ],
            "Read": [
                "DescribeAgent",
                "DescribeLocationEfs",
                "DescribeLocationFsxWindows",
                "DescribeLocationNfs",
                "DescribeLocationObjectStorage",
                "DescribeLocationS3",
                "DescribeLocationSmb",
                "DescribeTask",
                "DescribeTaskExecution",
                "ListTagsForResource"
            ],
            "List": [
                "ListAgents",
                "ListLocations",
                "ListTaskExecutions",
                "ListTasks"
            ],
            "Tagging": [
                "UntagResource"
            ]
        };
    }
    /**
     * Cancels execution of a sync task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CancelTaskExecution.html
     */
    toCancelTaskExecution() {
        this.to('datasync:CancelTaskExecution');
        return this;
    }
    /**
     * Activates an agent that you have deployed on your host.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateAgent.html
     */
    toCreateAgent() {
        this.to('datasync:CreateAgent');
        return this;
    }
    /**
     * Creates an endpoint for an Amazon EFS file system.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationEfs.html
     */
    toCreateLocationEfs() {
        this.to('datasync:CreateLocationEfs');
        return this;
    }
    /**
     * Creates an endpoint for an Amazon FSx Windows File Server file system.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationFsxWindows.html
     */
    toCreateLocationFsxWindows() {
        this.to('datasync:CreateLocationFsxWindows');
        return this;
    }
    /**
     * Creates an endpoint for a NFS file system.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationNfs.html
     */
    toCreateLocationNfs() {
        this.to('datasync:CreateLocationNfs');
        return this;
    }
    /**
     * Creates an endpoint for a self-managed object storage bucket.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationObjectStorage.html
     */
    toCreateLocationObjectStorage() {
        this.to('datasync:CreateLocationObjectStorage');
        return this;
    }
    /**
     * Creates an endpoint for an Amazon S3 bucket.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationS3.html
     */
    toCreateLocationS3() {
        this.to('datasync:CreateLocationS3');
        return this;
    }
    /**
     * Creates an endpoint for an SMB file system.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationSmb.html
     */
    toCreateLocationSmb() {
        this.to('datasync:CreateLocationSmb');
        return this;
    }
    /**
     * Creates a sync task.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateTask.html
     */
    toCreateTask() {
        this.to('datasync:CreateTask');
        return this;
    }
    /**
     * Deletes an agent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DeleteAgent.html
     */
    toDeleteAgent() {
        this.to('datasync:DeleteAgent');
        return this;
    }
    /**
     * Deletes the configuration of a location used by AWS DataSync.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DeleteLocation.html
     */
    toDeleteLocation() {
        this.to('datasync:DeleteLocation');
        return this;
    }
    /**
     * Deletes a sync task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DeleteTask.html
     */
    toDeleteTask() {
        this.to('datasync:DeleteTask');
        return this;
    }
    /**
     * Returns metadata such as name, network interfaces, and the status (that is, whether the agent is running or not) about a sync agent.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeAgent.html
     */
    toDescribeAgent() {
        this.to('datasync:DescribeAgent');
        return this;
    }
    /**
     * Returns metadata, such as the path information about an Amazon EFS sync location.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationEfs.html
     */
    toDescribeLocationEfs() {
        this.to('datasync:DescribeLocationEfs');
        return this;
    }
    /**
     * Returns metadata, such as the path information about an Amazon FSx Windows sync location.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationFsxWindows.html
     */
    toDescribeLocationFsxWindows() {
        this.to('datasync:DescribeLocationFsxWindows');
        return this;
    }
    /**
     * Returns metadata, such as the path information, about a NFS sync location.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationNfs.html
     */
    toDescribeLocationNfs() {
        this.to('datasync:DescribeLocationNfs');
        return this;
    }
    /**
     * Returns metadata about a self-managed object storage server location.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationObjectStorage.html
     */
    toDescribeLocationObjectStorage() {
        this.to('datasync:DescribeLocationObjectStorage');
        return this;
    }
    /**
     * Returns metadata, such as bucket name, about an Amazon S3 bucket sync location.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationS3.html
     */
    toDescribeLocationS3() {
        this.to('datasync:DescribeLocationS3');
        return this;
    }
    /**
     * Returns metadata, such as the path information, about an SMB sync location.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationSmb.html
     */
    toDescribeLocationSmb() {
        this.to('datasync:DescribeLocationSmb');
        return this;
    }
    /**
     * Returns metadata about a sync task.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeTask.html
     */
    toDescribeTask() {
        this.to('datasync:DescribeTask');
        return this;
    }
    /**
     * Returns detailed metadata about a sync task that is being executed.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeTaskExecution.html
     */
    toDescribeTaskExecution() {
        this.to('datasync:DescribeTaskExecution');
        return this;
    }
    /**
     * Returns a list of agents owned by an AWS account in a region specified in the request.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_ListAgents.html
     */
    toListAgents() {
        this.to('datasync:ListAgents');
        return this;
    }
    /**
     * Returns a lists of source and destination sync locations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_ListLocations.html
     */
    toListLocations() {
        this.to('datasync:ListLocations');
        return this;
    }
    /**
     * This operation lists the tags that have been added to the specified resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('datasync:ListTagsForResource');
        return this;
    }
    /**
     * Returns a list of executed sync tasks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_ListTaskExecutions.html
     */
    toListTaskExecutions() {
        this.to('datasync:ListTaskExecutions');
        return this;
    }
    /**
     * Returns a list of all the sync tasks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_ListTasks.html
     */
    toListTasks() {
        this.to('datasync:ListTasks');
        return this;
    }
    /**
     * Starts a specific invocation of a sync task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_StartTaskExecution.html
     */
    toStartTaskExecution() {
        this.to('datasync:StartTaskExecution');
        return this;
    }
    /**
     * Applies a key-value pair to an AWS resource.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_TagResource.html
     */
    toTagResource() {
        this.to('datasync:TagResource');
        return this;
    }
    /**
     * This operation removes one or more tags from the specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UntagResource.html
     */
    toUntagResource() {
        this.to('datasync:UntagResource');
        return this;
    }
    /**
     * Updates the name of an agent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateAgent.html
     */
    toUpdateAgent() {
        this.to('datasync:UpdateAgent');
        return this;
    }
    /**
     * Updates the metadata associated with a sync task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateTask.html
     */
    toUpdateTask() {
        this.to('datasync:UpdateTask');
        return this;
    }
    /**
     * Adds a resource of type agent to the statement
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/agent.html
     *
     * @param agentId - Identifier for the agentId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAgent(agentId, accountId, region, partition) {
        var arn = 'arn:${Partition}:datasync:${Region}:${AccountId}:agent/${AgentId}';
        arn = arn.replace('${AgentId}', agentId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type location to the statement
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/location.html
     *
     * @param locationId - Identifier for the locationId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLocation(locationId, accountId, region, partition) {
        var arn = 'arn:${Partition}:datasync:${Region}:${AccountId}:location/${LocationId}';
        arn = arn.replace('${LocationId}', locationId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type task to the statement
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/task.html
     *
     * @param taskId - Identifier for the taskId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTask(taskId, accountId, region, partition) {
        var arn = 'arn:${Partition}:datasync:${Region}:${AccountId}:task/${TaskId}';
        arn = arn.replace('${TaskId}', taskId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type taskexecution to the statement
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/taskexecution.html
     *
     * @param taskId - Identifier for the taskId.
     * @param executionId - Identifier for the executionId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTaskexecution(taskId, executionId, accountId, region, partition) {
        var arn = 'arn:${Partition}:datasync:${Region}:${AccountId}:task/${TaskId}/execution/${ExecutionId}';
        arn = arn.replace('${TaskId}', taskId);
        arn = arn.replace('${ExecutionId}', executionId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Datasync = Datasync;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGF0YXN5bmMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJkYXRhc3luYy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSxzQ0FBNEM7QUFFNUM7Ozs7R0FJRztBQUNILE1BQWEsUUFBUyxTQUFRLHdCQUFlO0lBRzNDOzs7O09BSUc7SUFDSCxZQUFhLEdBQVk7UUFDdkIsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBUk4sa0JBQWEsR0FBRyxVQUFVLENBQUM7UUFzYXhCLG9CQUFlLEdBQW9CO1lBQzNDLE9BQU8sRUFBRTtnQkFDUCxxQkFBcUI7Z0JBQ3JCLGFBQWE7Z0JBQ2IsbUJBQW1CO2dCQUNuQiwwQkFBMEI7Z0JBQzFCLG1CQUFtQjtnQkFDbkIsNkJBQTZCO2dCQUM3QixrQkFBa0I7Z0JBQ2xCLG1CQUFtQjtnQkFDbkIsWUFBWTtnQkFDWixhQUFhO2dCQUNiLGdCQUFnQjtnQkFDaEIsWUFBWTtnQkFDWixvQkFBb0I7Z0JBQ3BCLGFBQWE7Z0JBQ2IsYUFBYTtnQkFDYixZQUFZO2FBQ2I7WUFDRCxNQUFNLEVBQUU7Z0JBQ04sZUFBZTtnQkFDZixxQkFBcUI7Z0JBQ3JCLDRCQUE0QjtnQkFDNUIscUJBQXFCO2dCQUNyQiwrQkFBK0I7Z0JBQy9CLG9CQUFvQjtnQkFDcEIscUJBQXFCO2dCQUNyQixjQUFjO2dCQUNkLHVCQUF1QjtnQkFDdkIscUJBQXFCO2FBQ3RCO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLFlBQVk7Z0JBQ1osZUFBZTtnQkFDZixvQkFBb0I7Z0JBQ3BCLFdBQVc7YUFDWjtZQUNELFNBQVMsRUFBRTtnQkFDVCxlQUFlO2FBQ2hCO1NBQ0YsQ0FBQztJQXJjRixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLDBCQUEwQjtRQUMvQixJQUFJLENBQUMsRUFBRSxDQUFDLG1DQUFtQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLDZCQUE2QjtRQUNsQyxJQUFJLENBQUMsRUFBRSxDQUFDLHNDQUFzQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1FBQy9DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLElBQUksQ0FBQyxFQUFFLENBQUMsd0NBQXdDLENBQUMsQ0FBQztRQUNsRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxFQUFFLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUMvQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDOUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBNENEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLE9BQU8sQ0FBQyxPQUFlLEVBQUUsU0FBa0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckYsSUFBSSxHQUFHLEdBQUcsbUVBQW1FLENBQUM7UUFDOUUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ3pDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksR0FBRyxDQUFDLENBQUM7UUFDcEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksVUFBVSxDQUFDLFVBQWtCLEVBQUUsU0FBa0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDM0YsSUFBSSxHQUFHLEdBQUcseUVBQXlFLENBQUM7UUFDcEYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksR0FBRyxDQUFDLENBQUM7UUFDcEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksTUFBTSxDQUFDLE1BQWMsRUFBRSxTQUFrQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuRixJQUFJLEdBQUcsR0FBRyxpRUFBaUUsQ0FBQztRQUM1RSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDdkMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNwRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksZUFBZSxDQUFDLE1BQWMsRUFBRSxXQUFtQixFQUFFLFNBQWtCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2pILElBQUksR0FBRyxHQUFHLDBGQUEwRixDQUFDO1FBQ3JHLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUN2QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsRUFBRSxXQUFXLENBQUMsQ0FBQztRQUNqRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ3BELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztDQUNGO0FBdmlCRCw0QkF1aUJDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSBcIi4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWxcIjtcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gXCIuLi9zaGFyZWRcIjtcblxuLyoqXG4gKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2RhdGFzeW5jXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9kYXRhc3luYy5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIERhdGFzeW5jIGV4dGVuZHMgUG9saWN5U3RhdGVtZW50IHtcbiAgcHVibGljIHNlcnZpY2VQcmVmaXggPSAnZGF0YXN5bmMnO1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2RhdGFzeW5jXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9kYXRhc3luYy5odG1sKS5cbiAgICpcbiAgICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gICAqL1xuICBjb25zdHJ1Y3RvciAoc2lkPzogc3RyaW5nKSB7XG4gICAgc3VwZXIoc2lkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDYW5jZWxzIGV4ZWN1dGlvbiBvZiBhIHN5bmMgdGFzay5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGF0YXN5bmMvbGF0ZXN0L3VzZXJndWlkZS9BUElfQ2FuY2VsVGFza0V4ZWN1dGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DYW5jZWxUYXNrRXhlY3V0aW9uKCkge1xuICAgIHRoaXMudG8oJ2RhdGFzeW5jOkNhbmNlbFRhc2tFeGVjdXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBY3RpdmF0ZXMgYW4gYWdlbnQgdGhhdCB5b3UgaGF2ZSBkZXBsb3llZCBvbiB5b3VyIGhvc3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RhdGFzeW5jL2xhdGVzdC91c2VyZ3VpZGUvQVBJX0NyZWF0ZUFnZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUFnZW50KCkge1xuICAgIHRoaXMudG8oJ2RhdGFzeW5jOkNyZWF0ZUFnZW50Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhbiBlbmRwb2ludCBmb3IgYW4gQW1hem9uIEVGUyBmaWxlIHN5c3RlbS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGF0YXN5bmMvbGF0ZXN0L3VzZXJndWlkZS9BUElfQ3JlYXRlTG9jYXRpb25FZnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlTG9jYXRpb25FZnMoKSB7XG4gICAgdGhpcy50bygnZGF0YXN5bmM6Q3JlYXRlTG9jYXRpb25FZnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGFuIGVuZHBvaW50IGZvciBhbiBBbWF6b24gRlN4IFdpbmRvd3MgRmlsZSBTZXJ2ZXIgZmlsZSBzeXN0ZW0uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RhdGFzeW5jL2xhdGVzdC91c2VyZ3VpZGUvQVBJX0NyZWF0ZUxvY2F0aW9uRnN4V2luZG93cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVMb2NhdGlvbkZzeFdpbmRvd3MoKSB7XG4gICAgdGhpcy50bygnZGF0YXN5bmM6Q3JlYXRlTG9jYXRpb25Gc3hXaW5kb3dzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhbiBlbmRwb2ludCBmb3IgYSBORlMgZmlsZSBzeXN0ZW0uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RhdGFzeW5jL2xhdGVzdC91c2VyZ3VpZGUvQVBJX0NyZWF0ZUxvY2F0aW9uTmZzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUxvY2F0aW9uTmZzKCkge1xuICAgIHRoaXMudG8oJ2RhdGFzeW5jOkNyZWF0ZUxvY2F0aW9uTmZzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhbiBlbmRwb2ludCBmb3IgYSBzZWxmLW1hbmFnZWQgb2JqZWN0IHN0b3JhZ2UgYnVja2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kYXRhc3luYy9sYXRlc3QvdXNlcmd1aWRlL0FQSV9DcmVhdGVMb2NhdGlvbk9iamVjdFN0b3JhZ2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlTG9jYXRpb25PYmplY3RTdG9yYWdlKCkge1xuICAgIHRoaXMudG8oJ2RhdGFzeW5jOkNyZWF0ZUxvY2F0aW9uT2JqZWN0U3RvcmFnZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYW4gZW5kcG9pbnQgZm9yIGFuIEFtYXpvbiBTMyBidWNrZXQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RhdGFzeW5jL2xhdGVzdC91c2VyZ3VpZGUvQVBJX0NyZWF0ZUxvY2F0aW9uUzMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlTG9jYXRpb25TMygpIHtcbiAgICB0aGlzLnRvKCdkYXRhc3luYzpDcmVhdGVMb2NhdGlvblMzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhbiBlbmRwb2ludCBmb3IgYW4gU01CIGZpbGUgc3lzdGVtLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kYXRhc3luYy9sYXRlc3QvdXNlcmd1aWRlL0FQSV9DcmVhdGVMb2NhdGlvblNtYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVMb2NhdGlvblNtYigpIHtcbiAgICB0aGlzLnRvKCdkYXRhc3luYzpDcmVhdGVMb2NhdGlvblNtYicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBzeW5jIHRhc2suXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RhdGFzeW5jL2xhdGVzdC91c2VyZ3VpZGUvQVBJX0NyZWF0ZVRhc2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlVGFzaygpIHtcbiAgICB0aGlzLnRvKCdkYXRhc3luYzpDcmVhdGVUYXNrJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhbiBhZ2VudC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGF0YXN5bmMvbGF0ZXN0L3VzZXJndWlkZS9BUElfRGVsZXRlQWdlbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQWdlbnQoKSB7XG4gICAgdGhpcy50bygnZGF0YXN5bmM6RGVsZXRlQWdlbnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIHRoZSBjb25maWd1cmF0aW9uIG9mIGEgbG9jYXRpb24gdXNlZCBieSBBV1MgRGF0YVN5bmMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RhdGFzeW5jL2xhdGVzdC91c2VyZ3VpZGUvQVBJX0RlbGV0ZUxvY2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUxvY2F0aW9uKCkge1xuICAgIHRoaXMudG8oJ2RhdGFzeW5jOkRlbGV0ZUxvY2F0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIHN5bmMgdGFzay5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGF0YXN5bmMvbGF0ZXN0L3VzZXJndWlkZS9BUElfRGVsZXRlVGFzay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVUYXNrKCkge1xuICAgIHRoaXMudG8oJ2RhdGFzeW5jOkRlbGV0ZVRhc2snKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIG1ldGFkYXRhIHN1Y2ggYXMgbmFtZSwgbmV0d29yayBpbnRlcmZhY2VzLCBhbmQgdGhlIHN0YXR1cyAodGhhdCBpcywgd2hldGhlciB0aGUgYWdlbnQgaXMgcnVubmluZyBvciBub3QpIGFib3V0IGEgc3luYyBhZ2VudC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kYXRhc3luYy9sYXRlc3QvdXNlcmd1aWRlL0FQSV9EZXNjcmliZUFnZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQWdlbnQoKSB7XG4gICAgdGhpcy50bygnZGF0YXN5bmM6RGVzY3JpYmVBZ2VudCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgbWV0YWRhdGEsIHN1Y2ggYXMgdGhlIHBhdGggaW5mb3JtYXRpb24gYWJvdXQgYW4gQW1hem9uIEVGUyBzeW5jIGxvY2F0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RhdGFzeW5jL2xhdGVzdC91c2VyZ3VpZGUvQVBJX0Rlc2NyaWJlTG9jYXRpb25FZnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVMb2NhdGlvbkVmcygpIHtcbiAgICB0aGlzLnRvKCdkYXRhc3luYzpEZXNjcmliZUxvY2F0aW9uRWZzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBtZXRhZGF0YSwgc3VjaCBhcyB0aGUgcGF0aCBpbmZvcm1hdGlvbiBhYm91dCBhbiBBbWF6b24gRlN4IFdpbmRvd3Mgc3luYyBsb2NhdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kYXRhc3luYy9sYXRlc3QvdXNlcmd1aWRlL0FQSV9EZXNjcmliZUxvY2F0aW9uRnN4V2luZG93cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUxvY2F0aW9uRnN4V2luZG93cygpIHtcbiAgICB0aGlzLnRvKCdkYXRhc3luYzpEZXNjcmliZUxvY2F0aW9uRnN4V2luZG93cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgbWV0YWRhdGEsIHN1Y2ggYXMgdGhlIHBhdGggaW5mb3JtYXRpb24sIGFib3V0IGEgTkZTIHN5bmMgbG9jYXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGF0YXN5bmMvbGF0ZXN0L3VzZXJndWlkZS9BUElfRGVzY3JpYmVMb2NhdGlvbk5mcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUxvY2F0aW9uTmZzKCkge1xuICAgIHRoaXMudG8oJ2RhdGFzeW5jOkRlc2NyaWJlTG9jYXRpb25OZnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIG1ldGFkYXRhIGFib3V0IGEgc2VsZi1tYW5hZ2VkIG9iamVjdCBzdG9yYWdlIHNlcnZlciBsb2NhdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kYXRhc3luYy9sYXRlc3QvdXNlcmd1aWRlL0FQSV9EZXNjcmliZUxvY2F0aW9uT2JqZWN0U3RvcmFnZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUxvY2F0aW9uT2JqZWN0U3RvcmFnZSgpIHtcbiAgICB0aGlzLnRvKCdkYXRhc3luYzpEZXNjcmliZUxvY2F0aW9uT2JqZWN0U3RvcmFnZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgbWV0YWRhdGEsIHN1Y2ggYXMgYnVja2V0IG5hbWUsIGFib3V0IGFuIEFtYXpvbiBTMyBidWNrZXQgc3luYyBsb2NhdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kYXRhc3luYy9sYXRlc3QvdXNlcmd1aWRlL0FQSV9EZXNjcmliZUxvY2F0aW9uUzMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVMb2NhdGlvblMzKCkge1xuICAgIHRoaXMudG8oJ2RhdGFzeW5jOkRlc2NyaWJlTG9jYXRpb25TMycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgbWV0YWRhdGEsIHN1Y2ggYXMgdGhlIHBhdGggaW5mb3JtYXRpb24sIGFib3V0IGFuIFNNQiBzeW5jIGxvY2F0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RhdGFzeW5jL2xhdGVzdC91c2VyZ3VpZGUvQVBJX0Rlc2NyaWJlTG9jYXRpb25TbWIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVMb2NhdGlvblNtYigpIHtcbiAgICB0aGlzLnRvKCdkYXRhc3luYzpEZXNjcmliZUxvY2F0aW9uU21iJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBtZXRhZGF0YSBhYm91dCBhIHN5bmMgdGFzay5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kYXRhc3luYy9sYXRlc3QvdXNlcmd1aWRlL0FQSV9EZXNjcmliZVRhc2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVUYXNrKCkge1xuICAgIHRoaXMudG8oJ2RhdGFzeW5jOkRlc2NyaWJlVGFzaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgZGV0YWlsZWQgbWV0YWRhdGEgYWJvdXQgYSBzeW5jIHRhc2sgdGhhdCBpcyBiZWluZyBleGVjdXRlZC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kYXRhc3luYy9sYXRlc3QvdXNlcmd1aWRlL0FQSV9EZXNjcmliZVRhc2tFeGVjdXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVUYXNrRXhlY3V0aW9uKCkge1xuICAgIHRoaXMudG8oJ2RhdGFzeW5jOkRlc2NyaWJlVGFza0V4ZWN1dGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYSBsaXN0IG9mIGFnZW50cyBvd25lZCBieSBhbiBBV1MgYWNjb3VudCBpbiBhIHJlZ2lvbiBzcGVjaWZpZWQgaW4gdGhlIHJlcXVlc3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGF0YXN5bmMvbGF0ZXN0L3VzZXJndWlkZS9BUElfTGlzdEFnZW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QWdlbnRzKCkge1xuICAgIHRoaXMudG8oJ2RhdGFzeW5jOkxpc3RBZ2VudHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGEgbGlzdHMgb2Ygc291cmNlIGFuZCBkZXN0aW5hdGlvbiBzeW5jIGxvY2F0aW9ucy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kYXRhc3luYy9sYXRlc3QvdXNlcmd1aWRlL0FQSV9MaXN0TG9jYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RMb2NhdGlvbnMoKSB7XG4gICAgdGhpcy50bygnZGF0YXN5bmM6TGlzdExvY2F0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoaXMgb3BlcmF0aW9uIGxpc3RzIHRoZSB0YWdzIHRoYXQgaGF2ZSBiZWVuIGFkZGVkIHRvIHRoZSBzcGVjaWZpZWQgcmVzb3VyY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGF0YXN5bmMvbGF0ZXN0L3VzZXJndWlkZS9BUElfTGlzdFRhZ3NGb3JSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGFnc0ZvclJlc291cmNlKCkge1xuICAgIHRoaXMudG8oJ2RhdGFzeW5jOkxpc3RUYWdzRm9yUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGEgbGlzdCBvZiBleGVjdXRlZCBzeW5jIHRhc2tzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RhdGFzeW5jL2xhdGVzdC91c2VyZ3VpZGUvQVBJX0xpc3RUYXNrRXhlY3V0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGFza0V4ZWN1dGlvbnMoKSB7XG4gICAgdGhpcy50bygnZGF0YXN5bmM6TGlzdFRhc2tFeGVjdXRpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhIGxpc3Qgb2YgYWxsIHRoZSBzeW5jIHRhc2tzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RhdGFzeW5jL2xhdGVzdC91c2VyZ3VpZGUvQVBJX0xpc3RUYXNrcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGFza3MoKSB7XG4gICAgdGhpcy50bygnZGF0YXN5bmM6TGlzdFRhc2tzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU3RhcnRzIGEgc3BlY2lmaWMgaW52b2NhdGlvbiBvZiBhIHN5bmMgdGFzay5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGF0YXN5bmMvbGF0ZXN0L3VzZXJndWlkZS9BUElfU3RhcnRUYXNrRXhlY3V0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0VGFza0V4ZWN1dGlvbigpIHtcbiAgICB0aGlzLnRvKCdkYXRhc3luYzpTdGFydFRhc2tFeGVjdXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBcHBsaWVzIGEga2V5LXZhbHVlIHBhaXIgdG8gYW4gQVdTIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kYXRhc3luYy9sYXRlc3QvdXNlcmd1aWRlL0FQSV9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdSZXNvdXJjZSgpIHtcbiAgICB0aGlzLnRvKCdkYXRhc3luYzpUYWdSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoaXMgb3BlcmF0aW9uIHJlbW92ZXMgb25lIG9yIG1vcmUgdGFncyBmcm9tIHRoZSBzcGVjaWZpZWQgcmVzb3VyY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGF0YXN5bmMvbGF0ZXN0L3VzZXJndWlkZS9BUElfVW50YWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VbnRhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMudG8oJ2RhdGFzeW5jOlVudGFnUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIHRoZSBuYW1lIG9mIGFuIGFnZW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kYXRhc3luYy9sYXRlc3QvdXNlcmd1aWRlL0FQSV9VcGRhdGVBZ2VudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVBZ2VudCgpIHtcbiAgICB0aGlzLnRvKCdkYXRhc3luYzpVcGRhdGVBZ2VudCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgdGhlIG1ldGFkYXRhIGFzc29jaWF0ZWQgd2l0aCBhIHN5bmMgdGFzay5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGF0YXN5bmMvbGF0ZXN0L3VzZXJndWlkZS9BUElfVXBkYXRlVGFzay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVUYXNrKCkge1xuICAgIHRoaXMudG8oJ2RhdGFzeW5jOlVwZGF0ZVRhc2snKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIHByb3RlY3RlZCBhY2Nlc3NMZXZlbExpc3Q6IEFjY2Vzc0xldmVsTGlzdCA9IHtcbiAgICBcIldyaXRlXCI6IFtcbiAgICAgIFwiQ2FuY2VsVGFza0V4ZWN1dGlvblwiLFxuICAgICAgXCJDcmVhdGVBZ2VudFwiLFxuICAgICAgXCJDcmVhdGVMb2NhdGlvbkVmc1wiLFxuICAgICAgXCJDcmVhdGVMb2NhdGlvbkZzeFdpbmRvd3NcIixcbiAgICAgIFwiQ3JlYXRlTG9jYXRpb25OZnNcIixcbiAgICAgIFwiQ3JlYXRlTG9jYXRpb25PYmplY3RTdG9yYWdlXCIsXG4gICAgICBcIkNyZWF0ZUxvY2F0aW9uUzNcIixcbiAgICAgIFwiQ3JlYXRlTG9jYXRpb25TbWJcIixcbiAgICAgIFwiQ3JlYXRlVGFza1wiLFxuICAgICAgXCJEZWxldGVBZ2VudFwiLFxuICAgICAgXCJEZWxldGVMb2NhdGlvblwiLFxuICAgICAgXCJEZWxldGVUYXNrXCIsXG4gICAgICBcIlN0YXJ0VGFza0V4ZWN1dGlvblwiLFxuICAgICAgXCJUYWdSZXNvdXJjZVwiLFxuICAgICAgXCJVcGRhdGVBZ2VudFwiLFxuICAgICAgXCJVcGRhdGVUYXNrXCJcbiAgICBdLFxuICAgIFwiUmVhZFwiOiBbXG4gICAgICBcIkRlc2NyaWJlQWdlbnRcIixcbiAgICAgIFwiRGVzY3JpYmVMb2NhdGlvbkVmc1wiLFxuICAgICAgXCJEZXNjcmliZUxvY2F0aW9uRnN4V2luZG93c1wiLFxuICAgICAgXCJEZXNjcmliZUxvY2F0aW9uTmZzXCIsXG4gICAgICBcIkRlc2NyaWJlTG9jYXRpb25PYmplY3RTdG9yYWdlXCIsXG4gICAgICBcIkRlc2NyaWJlTG9jYXRpb25TM1wiLFxuICAgICAgXCJEZXNjcmliZUxvY2F0aW9uU21iXCIsXG4gICAgICBcIkRlc2NyaWJlVGFza1wiLFxuICAgICAgXCJEZXNjcmliZVRhc2tFeGVjdXRpb25cIixcbiAgICAgIFwiTGlzdFRhZ3NGb3JSZXNvdXJjZVwiXG4gICAgXSxcbiAgICBcIkxpc3RcIjogW1xuICAgICAgXCJMaXN0QWdlbnRzXCIsXG4gICAgICBcIkxpc3RMb2NhdGlvbnNcIixcbiAgICAgIFwiTGlzdFRhc2tFeGVjdXRpb25zXCIsXG4gICAgICBcIkxpc3RUYXNrc1wiXG4gICAgXSxcbiAgICBcIlRhZ2dpbmdcIjogW1xuICAgICAgXCJVbnRhZ1Jlc291cmNlXCJcbiAgICBdXG4gIH07XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGFnZW50IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RhdGFzeW5jL2xhdGVzdC91c2VyZ3VpZGUvYWdlbnQuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gYWdlbnRJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBhZ2VudElkLlxuICAgKiBAcGFyYW0gYWNjb3VudElkIC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQWdlbnQoYWdlbnRJZDogc3RyaW5nLCBhY2NvdW50SWQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmRhdGFzeW5jOiR7UmVnaW9ufToke0FjY291bnRJZH06YWdlbnQvJHtBZ2VudElkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWdlbnRJZH0nLCBhZ2VudElkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50SWR9JywgYWNjb3VudElkIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgbG9jYXRpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGF0YXN5bmMvbGF0ZXN0L3VzZXJndWlkZS9sb2NhdGlvbi5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBsb2NhdGlvbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGxvY2F0aW9uSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50SWQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Mb2NhdGlvbihsb2NhdGlvbklkOiBzdHJpbmcsIGFjY291bnRJZD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06ZGF0YXN5bmM6JHtSZWdpb259OiR7QWNjb3VudElkfTpsb2NhdGlvbi8ke0xvY2F0aW9uSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtMb2NhdGlvbklkfScsIGxvY2F0aW9uSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnRJZH0nLCBhY2NvdW50SWQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSB0YXNrIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RhdGFzeW5jL2xhdGVzdC91c2VyZ3VpZGUvdGFzay5odG1sXG4gICAqXG4gICAqIEBwYXJhbSB0YXNrSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgdGFza0lkLlxuICAgKiBAcGFyYW0gYWNjb3VudElkIC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uVGFzayh0YXNrSWQ6IHN0cmluZywgYWNjb3VudElkPzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpkYXRhc3luYzoke1JlZ2lvbn06JHtBY2NvdW50SWR9OnRhc2svJHtUYXNrSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtUYXNrSWR9JywgdGFza0lkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50SWR9JywgYWNjb3VudElkIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgdGFza2V4ZWN1dGlvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kYXRhc3luYy9sYXRlc3QvdXNlcmd1aWRlL3Rhc2tleGVjdXRpb24uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gdGFza0lkIC0gSWRlbnRpZmllciBmb3IgdGhlIHRhc2tJZC5cbiAgICogQHBhcmFtIGV4ZWN1dGlvbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGV4ZWN1dGlvbklkLlxuICAgKiBAcGFyYW0gYWNjb3VudElkIC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25UYXNrZXhlY3V0aW9uKHRhc2tJZDogc3RyaW5nLCBleGVjdXRpb25JZDogc3RyaW5nLCBhY2NvdW50SWQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmRhdGFzeW5jOiR7UmVnaW9ufToke0FjY291bnRJZH06dGFzay8ke1Rhc2tJZH0vZXhlY3V0aW9uLyR7RXhlY3V0aW9uSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtUYXNrSWR9JywgdGFza0lkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtFeGVjdXRpb25JZH0nLCBleGVjdXRpb25JZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudElkfScsIGFjY291bnRJZCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG59XG4iXX0=