"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Detective = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [detective](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondetective.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Detective extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [detective](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondetective.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'detective';
        this.accessLevelList = {
            "Write": [
                "AcceptInvitation",
                "CreateGraph",
                "CreateMembers",
                "DeleteGraph",
                "DeleteMembers",
                "DisassociateMembership",
                "RejectInvitation",
                "StartMonitoringMember"
            ],
            "Read": [
                "GetFreeTrialEligibility",
                "GetGraphIngestState",
                "GetMembers",
                "GetPricingInformation",
                "GetUsageInformation",
                "SearchGraph"
            ],
            "List": [
                "ListGraphs",
                "ListInvitations",
                "ListMembers"
            ]
        };
    }
    /**
     * Grants permission to accept an invitation to become a member of a behavior graph
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_AcceptInvitation.html
     */
    toAcceptInvitation() {
        this.to('detective:AcceptInvitation');
        return this;
    }
    /**
     * Grants permission to create a behavior graph and begin to aggregate security information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_CreateGraph.html
     */
    toCreateGraph() {
        this.to('detective:CreateGraph');
        return this;
    }
    /**
     * Grants permission to request the membership of one or more accounts in a behavior graph managed by this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_CreateMembers.html
     */
    toCreateMembers() {
        this.to('detective:CreateMembers');
        return this;
    }
    /**
     * Grants permission to delete a behavior graph and stop aggregating security information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_DeleteGraph.html
     */
    toDeleteGraph() {
        this.to('detective:DeleteGraph');
        return this;
    }
    /**
     * Grants permission to remove member accounts from a behavior graph managed by this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_DeleteMembers.html
     */
    toDeleteMembers() {
        this.to('detective:DeleteMembers');
        return this;
    }
    /**
     * Grants permission to remove the association of this account with a behavior graph
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_DisassociateMembership.html
     */
    toDisassociateMembership() {
        this.to('detective:DisassociateMembership');
        return this;
    }
    /**
     * Grants permission to retrieve a behavior graph's eligibility for a free trial period
     *
     * Access Level: Read
     */
    toGetFreeTrialEligibility() {
        this.to('detective:GetFreeTrialEligibility');
        return this;
    }
    /**
     * Grants permission to retrieve the data ingestion state of a behavior graph
     *
     * Access Level: Read
     */
    toGetGraphIngestState() {
        this.to('detective:GetGraphIngestState');
        return this;
    }
    /**
     * Grants permission to retrieve details on specified members of a behavior graph
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_GetMembers.html
     */
    toGetMembers() {
        this.to('detective:GetMembers');
        return this;
    }
    /**
     * Grants permission to retrieve information about Amazon Detective's pricing
     *
     * Access Level: Read
     */
    toGetPricingInformation() {
        this.to('detective:GetPricingInformation');
        return this;
    }
    /**
     * Grants permission to list usage information of a behavior graph
     *
     * Access Level: Read
     */
    toGetUsageInformation() {
        this.to('detective:GetUsageInformation');
        return this;
    }
    /**
     * Grants permission to list behavior graphs managed by this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListGraphs.html
     */
    toListGraphs() {
        this.to('detective:ListGraphs');
        return this;
    }
    /**
     * Grants permission to retrieve details on the behavior graphs to which this account has been invited to join
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListInvitations.html
     */
    toListInvitations() {
        this.to('detective:ListInvitations');
        return this;
    }
    /**
     * Grants permission to retrieve details on all members of a behavior graph
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListMembers.html
     */
    toListMembers() {
        this.to('detective:ListMembers');
        return this;
    }
    /**
     * Grants permission to reject an invitation to become a member of a behavior graph
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_RejectInvitation.html
     */
    toRejectInvitation() {
        this.to('detective:RejectInvitation');
        return this;
    }
    /**
     * Grants permission to search the data stored in a behavior graph
     *
     * Access Level: Read
     */
    toSearchGraph() {
        this.to('detective:SearchGraph');
        return this;
    }
    /**
     * Grants permission to start data ingest for a member account that has a status of ACCEPTED_BUT_DISABLED.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_StartMonitoringMember.html
     */
    toStartMonitoringMember() {
        this.to('detective:StartMonitoringMember');
        return this;
    }
    /**
     * Adds a resource of type Graph to the statement
     *
     * https://docs.aws.amazon.com/detective/latest/adminguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGraph(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:detective:${Region}:${Account}:graph:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Detective = Detective;
//# sourceMappingURL=data:application/json;base64,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