"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Directconnect = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [directconnect](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdirectconnect.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Directconnect extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [directconnect](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdirectconnect.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'directconnect';
        this.accessLevelList = {
            "Write": [
                "AcceptDirectConnectGatewayAssociationProposal",
                "AllocateConnectionOnInterconnect",
                "AllocateHostedConnection",
                "AllocatePrivateVirtualInterface",
                "AllocatePublicVirtualInterface",
                "AllocateTransitVirtualInterface",
                "AssociateConnectionWithLag",
                "AssociateHostedConnection",
                "AssociateVirtualInterface",
                "ConfirmConnection",
                "ConfirmPrivateVirtualInterface",
                "ConfirmPublicVirtualInterface",
                "ConfirmTransitVirtualInterface",
                "CreateBGPPeer",
                "CreateConnection",
                "CreateDirectConnectGateway",
                "CreateDirectConnectGatewayAssociation",
                "CreateDirectConnectGatewayAssociationProposal",
                "CreateInterconnect",
                "CreateLag",
                "CreatePrivateVirtualInterface",
                "CreatePublicVirtualInterface",
                "CreateTransitVirtualInterface",
                "DeleteBGPPeer",
                "DeleteConnection",
                "DeleteDirectConnectGateway",
                "DeleteDirectConnectGatewayAssociation",
                "DeleteDirectConnectGatewayAssociationProposal",
                "DeleteInterconnect",
                "DeleteLag",
                "DeleteVirtualInterface",
                "DisassociateConnectionFromLag",
                "StartBgpFailoverTest",
                "StopBgpFailoverTest",
                "UpdateDirectConnectGatewayAssociation",
                "UpdateLag",
                "UpdateVirtualInterfaceAttributes"
            ],
            "Read": [
                "DescribeConnectionLoa",
                "DescribeConnections",
                "DescribeConnectionsOnInterconnect",
                "DescribeDirectConnectGatewayAssociationProposals",
                "DescribeDirectConnectGatewayAssociations",
                "DescribeDirectConnectGatewayAttachments",
                "DescribeDirectConnectGateways",
                "DescribeHostedConnections",
                "DescribeInterconnectLoa",
                "DescribeInterconnects",
                "DescribeLags",
                "DescribeLoa",
                "DescribeTags",
                "DescribeVirtualGateways",
                "DescribeVirtualInterfaces"
            ],
            "List": [
                "DescribeLocations",
                "ListVirtualInterfaceTestHistory"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Accepts a proposal request to attach a virtual private gateway to a Direct Connect gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AcceptDirectConnectGatewayAssociationProposal.html
     */
    toAcceptDirectConnectGatewayAssociationProposal() {
        this.to('directconnect:AcceptDirectConnectGatewayAssociationProposal');
        return this;
    }
    /**
     * Creates a hosted connection on an interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateConnectionOnInterconnect.html
     */
    toAllocateConnectionOnInterconnect() {
        this.to('directconnect:AllocateConnectionOnInterconnect');
        return this;
    }
    /**
     * Creates a new hosted connection between a AWS Direct Connect partner's network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateHostedConnection.html
     */
    toAllocateHostedConnection() {
        this.to('directconnect:AllocateHostedConnection');
        return this;
    }
    /**
     * Provisions a private virtual interface to be owned by a different customer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePrivateVirtualInterface.html
     */
    toAllocatePrivateVirtualInterface() {
        this.to('directconnect:AllocatePrivateVirtualInterface');
        return this;
    }
    /**
     * Provisions a public virtual interface to be owned by a different customer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePublicVirtualInterface.html
     */
    toAllocatePublicVirtualInterface() {
        this.to('directconnect:AllocatePublicVirtualInterface');
        return this;
    }
    /**
     * Provisions a transit virtual interface to be owned by a different customer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateTransitVirtualInterface.html
     */
    toAllocateTransitVirtualInterface() {
        this.to('directconnect:AllocateTransitVirtualInterface');
        return this;
    }
    /**
     * Associates a connection with a LAG.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateConnectionWithLag.html
     */
    toAssociateConnectionWithLag() {
        this.to('directconnect:AssociateConnectionWithLag');
        return this;
    }
    /**
     * Associates a hosted connection and its virtual interfaces with a link aggregation group (LAG) or interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateHostedConnection.html
     */
    toAssociateHostedConnection() {
        this.to('directconnect:AssociateHostedConnection');
        return this;
    }
    /**
     * Associates a virtual interface with a specified link aggregation group (LAG) or connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateVirtualInterface.html
     */
    toAssociateVirtualInterface() {
        this.to('directconnect:AssociateVirtualInterface');
        return this;
    }
    /**
     * Confirm the creation of a hosted connection on an interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmConnection.html
     */
    toConfirmConnection() {
        this.to('directconnect:ConfirmConnection');
        return this;
    }
    /**
     * Accept ownership of a private virtual interface created by another customer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPrivateVirtualInterface.html
     */
    toConfirmPrivateVirtualInterface() {
        this.to('directconnect:ConfirmPrivateVirtualInterface');
        return this;
    }
    /**
     * Accept ownership of a public virtual interface created by another customer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPublicVirtualInterface.html
     */
    toConfirmPublicVirtualInterface() {
        this.to('directconnect:ConfirmPublicVirtualInterface');
        return this;
    }
    /**
     * Accept ownership of a transit virtual interface created by another customer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmTransitVirtualInterface.html
     */
    toConfirmTransitVirtualInterface() {
        this.to('directconnect:ConfirmTransitVirtualInterface');
        return this;
    }
    /**
     * Creates a BGP peer on the specified virtual interface.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateBGPPeer.html
     */
    toCreateBGPPeer() {
        this.to('directconnect:CreateBGPPeer');
        return this;
    }
    /**
     * Creates a new connection between the customer network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateConnection.html
     */
    toCreateConnection() {
        this.to('directconnect:CreateConnection');
        return this;
    }
    /**
     * Creates a Direct Connect gateway, which is an intermediate object that enables you to connect a set of virtual interfaces and virtual private gateways.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGateway.html
     */
    toCreateDirectConnectGateway() {
        this.to('directconnect:CreateDirectConnectGateway');
        return this;
    }
    /**
     * Creates an association between a Direct Connect gateway and a virtual private gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociation.html
     */
    toCreateDirectConnectGatewayAssociation() {
        this.to('directconnect:CreateDirectConnectGatewayAssociation');
        return this;
    }
    /**
     * Creates a proposal to associate the specified virtual private gateway with the specified Direct Connect gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociationProposal.html
     */
    toCreateDirectConnectGatewayAssociationProposal() {
        this.to('directconnect:CreateDirectConnectGatewayAssociationProposal');
        return this;
    }
    /**
     * Creates a new interconnect between a AWS Direct Connect partner's network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateInterconnect.html
     */
    toCreateInterconnect() {
        this.to('directconnect:CreateInterconnect');
        return this;
    }
    /**
     * Creates a link aggregation group (LAG) with the specified number of bundled physical connections between the customer network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateLag.html
     */
    toCreateLag() {
        this.to('directconnect:CreateLag');
        return this;
    }
    /**
     * Creates a new private virtual interface.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePrivateVirtualInterface.html
     */
    toCreatePrivateVirtualInterface() {
        this.to('directconnect:CreatePrivateVirtualInterface');
        return this;
    }
    /**
     * Creates a new public virtual interface.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePublicVirtualInterface.html
     */
    toCreatePublicVirtualInterface() {
        this.to('directconnect:CreatePublicVirtualInterface');
        return this;
    }
    /**
     * Creates a new transit virtual interface.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateTransitVirtualInterface.html
     */
    toCreateTransitVirtualInterface() {
        this.to('directconnect:CreateTransitVirtualInterface');
        return this;
    }
    /**
     * Deletes the specified BGP peer on the specified virtual interface with the specified customer address and ASN.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteBGPPeer.html
     */
    toDeleteBGPPeer() {
        this.to('directconnect:DeleteBGPPeer');
        return this;
    }
    /**
     * Deletes the connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteConnection.html
     */
    toDeleteConnection() {
        this.to('directconnect:DeleteConnection');
        return this;
    }
    /**
     * Deletes the specified Direct Connect gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGateway.html
     */
    toDeleteDirectConnectGateway() {
        this.to('directconnect:DeleteDirectConnectGateway');
        return this;
    }
    /**
     * Deletes the association between the specified Direct Connect gateway and virtual private gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociation.html
     */
    toDeleteDirectConnectGatewayAssociation() {
        this.to('directconnect:DeleteDirectConnectGatewayAssociation');
        return this;
    }
    /**
     * Deletes the association proposal request between the specified Direct Connect gateway and virtual private gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociationProposal.html
     */
    toDeleteDirectConnectGatewayAssociationProposal() {
        this.to('directconnect:DeleteDirectConnectGatewayAssociationProposal');
        return this;
    }
    /**
     * Deletes the specified interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteInterconnect.html
     */
    toDeleteInterconnect() {
        this.to('directconnect:DeleteInterconnect');
        return this;
    }
    /**
     * Deletes the specified link aggregation group (LAG).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteLag.html
     */
    toDeleteLag() {
        this.to('directconnect:DeleteLag');
        return this;
    }
    /**
     * Deletes a virtual interface.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteVirtualInterface.html
     */
    toDeleteVirtualInterface() {
        this.to('directconnect:DeleteVirtualInterface');
        return this;
    }
    /**
     * Returns the LOA-CFA for a Connection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionLoa.html
     */
    toDescribeConnectionLoa() {
        this.to('directconnect:DescribeConnectionLoa');
        return this;
    }
    /**
     * Displays all connections in this region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnections.html
     */
    toDescribeConnections() {
        this.to('directconnect:DescribeConnections');
        return this;
    }
    /**
     * Return a list of connections that have been provisioned on the given interconnect.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionsOnInterconnect.html
     */
    toDescribeConnectionsOnInterconnect() {
        this.to('directconnect:DescribeConnectionsOnInterconnect');
        return this;
    }
    /**
     * Describes one or more association proposals for connection between a virtual private gateway and a Direct Connect gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociationProposals.html
     */
    toDescribeDirectConnectGatewayAssociationProposals() {
        this.to('directconnect:DescribeDirectConnectGatewayAssociationProposals');
        return this;
    }
    /**
     * Lists the associations between your Direct Connect gateways and virtual private gateways.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociations.html
     */
    toDescribeDirectConnectGatewayAssociations() {
        this.to('directconnect:DescribeDirectConnectGatewayAssociations');
        return this;
    }
    /**
     * Lists the attachments between your Direct Connect gateways and virtual interfaces.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAttachments.html
     */
    toDescribeDirectConnectGatewayAttachments() {
        this.to('directconnect:DescribeDirectConnectGatewayAttachments');
        return this;
    }
    /**
     * Lists all your Direct Connect gateways or only the specified Direct Connect gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGateways.html
     */
    toDescribeDirectConnectGateways() {
        this.to('directconnect:DescribeDirectConnectGateways');
        return this;
    }
    /**
     * Lists the hosted connections that have been provisioned on the specified interconnect or link aggregation group (LAG).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeHostedConnections.html
     */
    toDescribeHostedConnections() {
        this.to('directconnect:DescribeHostedConnections');
        return this;
    }
    /**
     * Returns the LOA-CFA for an Interconnect.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnectLoa.html
     */
    toDescribeInterconnectLoa() {
        this.to('directconnect:DescribeInterconnectLoa');
        return this;
    }
    /**
     * Returns a list of interconnects owned by the AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnects.html
     */
    toDescribeInterconnects() {
        this.to('directconnect:DescribeInterconnects');
        return this;
    }
    /**
     * Describes all your link aggregation groups (LAG) or the specified LAG.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLags.html
     */
    toDescribeLags() {
        this.to('directconnect:DescribeLags');
        return this;
    }
    /**
     * Gets the LOA-CFA for a connection, interconnect, or link aggregation group (LAG).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLoa.html
     */
    toDescribeLoa() {
        this.to('directconnect:DescribeLoa');
        return this;
    }
    /**
     * Returns the list of AWS Direct Connect locations in the current AWS region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLocations.html
     */
    toDescribeLocations() {
        this.to('directconnect:DescribeLocations');
        return this;
    }
    /**
     * Describes the tags associated with the specified AWS Direct Connect resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeTags.html
     */
    toDescribeTags() {
        this.to('directconnect:DescribeTags');
        return this;
    }
    /**
     * Returns a list of virtual private gateways owned by the AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualGateways.html
     */
    toDescribeVirtualGateways() {
        this.to('directconnect:DescribeVirtualGateways');
        return this;
    }
    /**
     * Displays all virtual interfaces for an AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualInterfaces.html
     */
    toDescribeVirtualInterfaces() {
        this.to('directconnect:DescribeVirtualInterfaces');
        return this;
    }
    /**
     * Disassociates a connection from a link aggregation group (LAG).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DisassociateConnectionFromLag.html
     */
    toDisassociateConnectionFromLag() {
        this.to('directconnect:DisassociateConnectionFromLag');
        return this;
    }
    /**
     * Lists the virtual interface failover test history.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ListVirtualInterfaceTestHistory.html
     */
    toListVirtualInterfaceTestHistory() {
        this.to('directconnect:ListVirtualInterfaceTestHistory');
        return this;
    }
    /**
     * Starts the virtual interface failover test that verifies your configuration meets your resiliency requirements by placing the BGP peering session in the DOWN state. You can then send traffic to verify that there are no outages.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StartBgpFailoverTest.html
     */
    toStartBgpFailoverTest() {
        this.to('directconnect:StartBgpFailoverTest');
        return this;
    }
    /**
     * Stops the virtual interface failover test.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StopBgpFailoverTest.html
     */
    toStopBgpFailoverTest() {
        this.to('directconnect:StopBgpFailoverTest');
        return this;
    }
    /**
     * Adds the specified tags to the specified AWS Direct Connect resource. Each resource can have a maximum of 50 tags.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('directconnect:TagResource');
        return this;
    }
    /**
     * Removes one or more tags from the specified AWS Direct Connect resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('directconnect:UntagResource');
        return this;
    }
    /**
     * Updates the specified attributes of the Direct Connect gateway association.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateDirectConnectGatewayAssociation.html
     */
    toUpdateDirectConnectGatewayAssociation() {
        this.to('directconnect:UpdateDirectConnectGatewayAssociation');
        return this;
    }
    /**
     * Updates the attributes of the specified link aggregation group (LAG).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateLag.html
     */
    toUpdateLag() {
        this.to('directconnect:UpdateLag');
        return this;
    }
    /**
     * Updates the specified attributes of the specified virtual private interface.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateVirtualInterfaceAttributes.html
     */
    toUpdateVirtualInterfaceAttributes() {
        this.to('directconnect:UpdateVirtualInterfaceAttributes');
        return this;
    }
    /**
     * Adds a resource of type dxcon to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Connection.html
     *
     * @param connectionId - Identifier for the connectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDxcon(connectionId, account, region, partition) {
        var arn = 'arn:${Partition}:directconnect:${Region}:${Account}:dxcon/${ConnectionId}';
        arn = arn.replace('${ConnectionId}', connectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dxlag to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Lag.html
     *
     * @param lagId - Identifier for the lagId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDxlag(lagId, account, region, partition) {
        var arn = 'arn:${Partition}:directconnect:${Region}:${Account}:dxlag/${LagId}';
        arn = arn.replace('${LagId}', lagId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dxvif to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_VirtualInterface.html
     *
     * @param virtualInterfaceId - Identifier for the virtualInterfaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDxvif(virtualInterfaceId, account, region, partition) {
        var arn = 'arn:${Partition}:directconnect:${Region}:${Account}:dxvif/${VirtualInterfaceId}';
        arn = arn.replace('${VirtualInterfaceId}', virtualInterfaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dx-gateway to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DirectConnectGateway.html
     *
     * @param directConnectGatewayId - Identifier for the directConnectGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDxGateway(directConnectGatewayId, account, partition) {
        var arn = 'arn:${Partition}:directconnect::${Account}:dx-gateway/${DirectConnectGatewayId}';
        arn = arn.replace('${DirectConnectGatewayId}', directConnectGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Directconnect = Directconnect;
//# sourceMappingURL=data:application/json;base64,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