"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dynamodb = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [dynamodb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondynamodb.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Dynamodb extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [dynamodb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondynamodb.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'dynamodb';
        this.accessLevelList = {
            "Read": [
                "BatchGetItem",
                "ConditionCheckItem",
                "DescribeBackup",
                "DescribeContinuousBackups",
                "DescribeContributorInsights",
                "DescribeExport",
                "DescribeGlobalTable",
                "DescribeGlobalTableSettings",
                "DescribeLimits",
                "DescribeReservedCapacity",
                "DescribeReservedCapacityOfferings",
                "DescribeStream",
                "DescribeTable",
                "DescribeTableReplicaAutoScaling",
                "DescribeTimeToLive",
                "GetItem",
                "GetRecords",
                "GetShardIterator",
                "ListStreams",
                "ListTagsOfResource",
                "Query",
                "Scan"
            ],
            "Write": [
                "BatchWriteItem",
                "CreateBackup",
                "CreateGlobalTable",
                "CreateTable",
                "CreateTableReplica",
                "DeleteBackup",
                "DeleteItem",
                "DeleteTable",
                "DeleteTableReplica",
                "ExportTableToPointInTime",
                "PurchaseReservedCapacityOfferings",
                "PutItem",
                "RestoreTableFromBackup",
                "RestoreTableToPointInTime",
                "UpdateContinuousBackups",
                "UpdateContributorInsights",
                "UpdateGlobalTable",
                "UpdateGlobalTableSettings",
                "UpdateItem",
                "UpdateTable",
                "UpdateTableReplicaAutoScaling",
                "UpdateTimeToLive"
            ],
            "List": [
                "ListBackups",
                "ListContributorInsights",
                "ListExports",
                "ListGlobalTables",
                "ListTables"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Returns the attributes of one or more items from one or more tables
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchGetItem.html
     */
    toBatchGetItem() {
        this.to('dynamodb:BatchGetItem');
        return this;
    }
    /**
     * Puts or deletes multiple items in one or more tables
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchWriteItem.html
     */
    toBatchWriteItem() {
        this.to('dynamodb:BatchWriteItem');
        return this;
    }
    /**
     * The ConditionCheckItem operation checks the existence of a set of attributes for the item with the given primary key
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ConditionCheckItem.html
     */
    toConditionCheckItem() {
        this.to('dynamodb:ConditionCheckItem');
        return this;
    }
    /**
     * Creates a backup for an existing table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateBackup.html
     */
    toCreateBackup() {
        this.to('dynamodb:CreateBackup');
        return this;
    }
    /**
     * Enables the user to create a global table from an existing table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateGlobalTable.html
     */
    toCreateGlobalTable() {
        this.to('dynamodb:CreateGlobalTable');
        return this;
    }
    /**
     * The CreateTable operation adds a new table to your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateTable.html
     */
    toCreateTable() {
        this.to('dynamodb:CreateTable');
        return this;
    }
    /**
     * Adds a new replica table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/V2gt_IAM.html
     */
    toCreateTableReplica() {
        this.to('dynamodb:CreateTableReplica');
        return this;
    }
    /**
     * Deletes an existing backup of a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteBackup.html
     */
    toDeleteBackup() {
        this.to('dynamodb:DeleteBackup');
        return this;
    }
    /**
     * Deletes a single item in a table by primary key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html
     */
    toDeleteItem() {
        this.to('dynamodb:DeleteItem');
        return this;
    }
    /**
     * The DeleteTable operation deletes a table and all of its items
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteTable.html
     */
    toDeleteTable() {
        this.to('dynamodb:DeleteTable');
        return this;
    }
    /**
     * Deletes a replica table and all of its items
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/V2gt_IAM.html
     */
    toDeleteTableReplica() {
        this.to('dynamodb:DeleteTableReplica');
        return this;
    }
    /**
     * Describes an existing backup of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeBackup.html
     */
    toDescribeBackup() {
        this.to('dynamodb:DescribeBackup');
        return this;
    }
    /**
     * Checks the status of the backup restore settings on the specified table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeContinuousBackups.html
     */
    toDescribeContinuousBackups() {
        this.to('dynamodb:DescribeContinuousBackups');
        return this;
    }
    /**
     * Describes the contributor insights status and related details for a given table or global secondary index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeContributorInsights.html
     */
    toDescribeContributorInsights() {
        this.to('dynamodb:DescribeContributorInsights');
        return this;
    }
    /**
     * Describes an existing Export of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeExport.html
     */
    toDescribeExport() {
        this.to('dynamodb:DescribeExport');
        return this;
    }
    /**
     * Returns information about the specified global table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeGlobalTable.html
     */
    toDescribeGlobalTable() {
        this.to('dynamodb:DescribeGlobalTable');
        return this;
    }
    /**
     * Returns settings information about the specified global table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeGlobalTableSettings.html
     */
    toDescribeGlobalTableSettings() {
        this.to('dynamodb:DescribeGlobalTableSettings');
        return this;
    }
    /**
     * Returns the current provisioned-capacity limits for your AWS account in a region, both for the region as a whole and for any one DynamoDB table that you create there
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeLimits.html
     */
    toDescribeLimits() {
        this.to('dynamodb:DescribeLimits');
        return this;
    }
    /**
     * Describes one or more of the Reserved Capacity purchased
     *
     * Access Level: Read
     */
    toDescribeReservedCapacity() {
        this.to('dynamodb:DescribeReservedCapacity');
        return this;
    }
    /**
     * Describes Reserved Capacity offerings that are available for purchase
     *
     * Access Level: Read
     */
    toDescribeReservedCapacityOfferings() {
        this.to('dynamodb:DescribeReservedCapacityOfferings');
        return this;
    }
    /**
     * Returns information about a stream, including the current status of the stream, its Amazon Resource Name (ARN), the composition of its shards, and its corresponding DynamoDB table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeStream.html
     */
    toDescribeStream() {
        this.to('dynamodb:DescribeStream');
        return this;
    }
    /**
     * Returns information about the table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTable.html
     */
    toDescribeTable() {
        this.to('dynamodb:DescribeTable');
        return this;
    }
    /**
     * Describes the auto scaling settings across all replicas of the global table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTableReplicaAutoScaling.html
     */
    toDescribeTableReplicaAutoScaling() {
        this.to('dynamodb:DescribeTableReplicaAutoScaling');
        return this;
    }
    /**
     * Gives a description of the Time to Live (TTL) status on the specified table.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTimeToLive.html
     */
    toDescribeTimeToLive() {
        this.to('dynamodb:DescribeTimeToLive');
        return this;
    }
    /**
     * Initiates an Export of a DynamoDB table to S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ExportTableToPointInTime.html
     */
    toExportTableToPointInTime() {
        this.to('dynamodb:ExportTableToPointInTime');
        return this;
    }
    /**
     * The GetItem operation returns a set of attributes for the item with the given primary key
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html
     */
    toGetItem() {
        this.to('dynamodb:GetItem');
        return this;
    }
    /**
     * Retrieves the stream records from a given shard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetRecords.html
     */
    toGetRecords() {
        this.to('dynamodb:GetRecords');
        return this;
    }
    /**
     * Returns a shard iterator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetShardIterator.html
     */
    toGetShardIterator() {
        this.to('dynamodb:GetShardIterator');
        return this;
    }
    /**
     * List backups associated with the account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListBackups.html
     */
    toListBackups() {
        this.to('dynamodb:ListBackups');
        return this;
    }
    /**
     * Lists the ContributorInsightsSummary for all tables and global secondary indexes associated with the current account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListContributorInsights.html
     */
    toListContributorInsights() {
        this.to('dynamodb:ListContributorInsights');
        return this;
    }
    /**
     * List exports associated with the account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListExports.html
     */
    toListExports() {
        this.to('dynamodb:ListExports');
        return this;
    }
    /**
     * Lists all global tables that have a replica in the specified region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListGlobalTables.html
     */
    toListGlobalTables() {
        this.to('dynamodb:ListGlobalTables');
        return this;
    }
    /**
     * Returns an array of stream ARNs associated with the current account and endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListStreams.html
     */
    toListStreams() {
        this.to('dynamodb:ListStreams');
        return this;
    }
    /**
     * Returns an array of table names associated with the current account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListTables.html
     */
    toListTables() {
        this.to('dynamodb:ListTables');
        return this;
    }
    /**
     * List all tags on an Amazon DynamoDB resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListTagsOfResource.html
     */
    toListTagsOfResource() {
        this.to('dynamodb:ListTagsOfResource');
        return this;
    }
    /**
     * Purchases Reserved Capacity for use with your account
     *
     * Access Level: Write
     */
    toPurchaseReservedCapacityOfferings() {
        this.to('dynamodb:PurchaseReservedCapacityOfferings');
        return this;
    }
    /**
     * Creates a new item, or replaces an old item with a new item
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html
     */
    toPutItem() {
        this.to('dynamodb:PutItem');
        return this;
    }
    /**
     * Uses the primary key of a table or a secondary index to directly access items from that table or index
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Query.html
     */
    toQuery() {
        this.to('dynamodb:Query');
        return this;
    }
    /**
     * Creates a new table from an existing backup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_RestoreTableFromBackup.html
     */
    toRestoreTableFromBackup() {
        this.to('dynamodb:RestoreTableFromBackup');
        return this;
    }
    /**
     * Restores a table to a point in time
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_RestoreTableToPointInTime.html
     */
    toRestoreTableToPointInTime() {
        this.to('dynamodb:RestoreTableToPointInTime');
        return this;
    }
    /**
     * Returns one or more items and item attributes by accessing every item in a table or a secondary index
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Scan.html
     */
    toScan() {
        this.to('dynamodb:Scan');
        return this;
    }
    /**
     * Associate a set of tags with an Amazon DynamoDB resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('dynamodb:TagResource');
        return this;
    }
    /**
     * Removes the association of tags from an Amazon DynamoDB resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('dynamodb:UntagResource');
        return this;
    }
    /**
     * Enables or disables continuous backups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateContinuousBackups.html
     */
    toUpdateContinuousBackups() {
        this.to('dynamodb:UpdateContinuousBackups');
        return this;
    }
    /**
     * Updates the status for contributor insights for a specific table or global secondary index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateContributorInsights.html
     */
    toUpdateContributorInsights() {
        this.to('dynamodb:UpdateContributorInsights');
        return this;
    }
    /**
     * Enables the user to add or remove replicas in the specified global table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateGlobalTable.html
     */
    toUpdateGlobalTable() {
        this.to('dynamodb:UpdateGlobalTable');
        return this;
    }
    /**
     * Enables the user to update settings of the specified global table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateGlobalTableSettings.html
     */
    toUpdateGlobalTableSettings() {
        this.to('dynamodb:UpdateGlobalTableSettings');
        return this;
    }
    /**
     * Edits an existing item's attributes, or adds a new item to the table if it does not already exist
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html
     */
    toUpdateItem() {
        this.to('dynamodb:UpdateItem');
        return this;
    }
    /**
     * Modifies the provisioned throughput settings, global secondary indexes, or DynamoDB Streams settings for a given table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTable.html
     */
    toUpdateTable() {
        this.to('dynamodb:UpdateTable');
        return this;
    }
    /**
     * Updates auto scaling settings on your replica table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTableReplicaAutoScaling.html
     */
    toUpdateTableReplicaAutoScaling() {
        this.to('dynamodb:UpdateTableReplicaAutoScaling');
        return this;
    }
    /**
     * Enables or disables TTL for the specified table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTimeToLive.html
     */
    toUpdateTimeToLive() {
        this.to('dynamodb:UpdateTimeToLive');
        return this;
    }
    /**
     * Adds a resource of type index to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.htmlHowItWorks.CoreComponents.html#HowItWorks.CoreComponents.PrimaryKey
     *
     * @param tableName - Identifier for the tableName.
     * @param indexName - Identifier for the indexName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIndex(tableName, indexName, account, region, partition) {
        var arn = 'arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}/index/${IndexName}';
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${IndexName}', indexName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.htmlHowItWorks.CoreComponents.html#HowItWorks.CoreComponents.Streams
     *
     * @param tableName - Identifier for the tableName.
     * @param streamLabel - Identifier for the streamLabel.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onStream(tableName, streamLabel, account, region, partition) {
        var arn = 'arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}/stream/${StreamLabel}';
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${StreamLabel}', streamLabel);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type table to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.htmlHowItWorks.CoreComponents.html#HowItWorks.CoreComponents.TablesItemsAttributes
     *
     * @param tableName - Identifier for the tableName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTable(tableName, account, region, partition) {
        var arn = 'arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}';
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type backup to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/backuprestore_HowItWorks.html
     *
     * @param tableName - Identifier for the tableName.
     * @param backupName - Identifier for the backupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBackup(tableName, backupName, account, region, partition) {
        var arn = 'arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}/backup/${BackupName}';
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${BackupName}', backupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type export to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/backuprestore_HowItWorks.html
     *
     * @param tableName - Identifier for the tableName.
     * @param exportName - Identifier for the exportName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onExport(tableName, exportName, account, region, partition) {
        var arn = 'arn:${Partition}:dynamodb:${Region}:${Account}:table/${TableName}/export/${ExportName}';
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${ExportName}', exportName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type global-table to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables_HowItWorks.html
     *
     * @param globalTableName - Identifier for the globalTableName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGlobalTable(globalTableName, account, partition) {
        var arn = 'arn:${Partition}:dynamodb::${Account}:global-table/${GlobalTableName}';
        arn = arn.replace('${GlobalTableName}', globalTableName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filter based on the attribute (field or column) names of the table.
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toBatchWriteItem()
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPutItem()
     * - .toQuery()
     * - .toScan()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAttributes(value, operator) {
        return this.if(`dynamodb:Attributes`, value, operator || 'StringLike');
    }
    /**
     * Used to block Transactions APIs calls and allow the non-Transaction APIs calls and vice-versa.
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPutItem()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnclosingOperation(value, operator) {
        return this.if(`dynamodb:EnclosingOperation`, value, operator || 'StringLike');
    }
    /**
     * Filters based on the partition key of the table.
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toBatchWriteItem()
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPutItem()
     * - .toQuery()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLeadingKeys(value, operator) {
        return this.if(`dynamodb:LeadingKeys`, value, operator || 'StringLike');
    }
    /**
     * Filter based on the ReturnConsumedCapacity parameter of a request. Contains either "TOTAL" or "NONE".
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toBatchWriteItem()
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPutItem()
     * - .toQuery()
     * - .toScan()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReturnConsumedCapacity(value, operator) {
        return this.if(`dynamodb:ReturnConsumedCapacity`, value, operator || 'StringLike');
    }
    /**
     * Filter based on the ReturnValues parameter of request. Contains one of the following: "ALL_OLD", "UPDATED_OLD","ALL_NEW","UPDATED_NEW", or "NONE".
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toPutItem()
     * - .toQuery()
     * - .toScan()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReturnValues(value, operator) {
        return this.if(`dynamodb:ReturnValues`, value, operator || 'StringLike');
    }
    /**
     * Filter based on the Select parameter of a Query or Scan request.
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toGetItem()
     * - .toQuery()
     * - .toScan()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSelect(value, operator) {
        return this.if(`dynamodb:Select`, value, operator || 'StringLike');
    }
}
exports.Dynamodb = Dynamodb;
//# sourceMappingURL=data:application/json;base64,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