"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elasticache = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [elasticache](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticache.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elasticache extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [elasticache](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticache.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'elasticache';
        this.accessLevelList = {
            "Tagging": [
                "AddTagsToResource",
                "RemoveTagsFromResource"
            ],
            "Write": [
                "AuthorizeCacheSecurityGroupIngress",
                "BatchApplyUpdateAction",
                "BatchStopUpdateAction",
                "CompleteMigration",
                "CopySnapshot",
                "CreateCacheCluster",
                "CreateCacheParameterGroup",
                "CreateCacheSecurityGroup",
                "CreateCacheSubnetGroup",
                "CreateGlobalReplicationGroup",
                "CreateReplicationGroup",
                "CreateSnapshot",
                "CreateUser",
                "CreateUserGroup",
                "DecreaseNodeGroupsInGlobalReplicationGroup",
                "DecreaseReplicaCount",
                "DeleteCacheCluster",
                "DeleteCacheParameterGroup",
                "DeleteCacheSecurityGroup",
                "DeleteCacheSubnetGroup",
                "DeleteGlobalReplicationGroup",
                "DeleteReplicationGroup",
                "DeleteSnapshot",
                "DeleteUser",
                "DeleteUserGroup",
                "DisassociateGlobalReplicationGroup",
                "FailoverGlobalReplicationGroup",
                "IncreaseNodeGroupsInGlobalReplicationGroup",
                "IncreaseReplicaCount",
                "ModifyCacheCluster",
                "ModifyCacheParameterGroup",
                "ModifyCacheSubnetGroup",
                "ModifyGlobalReplicationGroup",
                "ModifyReplicationGroup",
                "ModifyReplicationGroupShardConfiguration",
                "ModifyUser",
                "ModifyUserGroup",
                "PurchaseReservedCacheNodesOffering",
                "RebalanceSlotsInGlobalReplicationGroup",
                "RebootCacheCluster",
                "ResetCacheParameterGroup",
                "RevokeCacheSecurityGroupIngress",
                "StartMigration",
                "TestFailover"
            ],
            "List": [
                "DescribeCacheClusters",
                "DescribeCacheEngineVersions",
                "DescribeCacheParameterGroups",
                "DescribeCacheParameters",
                "DescribeCacheSecurityGroups",
                "DescribeCacheSubnetGroups",
                "DescribeEngineDefaultParameters",
                "DescribeEvents",
                "DescribeGlobalReplicationGroups",
                "DescribeReplicationGroups",
                "DescribeReservedCacheNodes",
                "DescribeReservedCacheNodesOfferings",
                "DescribeServiceUpdates",
                "DescribeSnapshots",
                "DescribeUpdateActions",
                "DescribeUserGroups",
                "DescribeUsers",
                "ListAllowedNodeTypeModifications"
            ],
            "Read": [
                "ListTagsForResource"
            ]
        };
    }
    /**
     * The AddTagsToResource action adds up to 10 cost allocation tags to the named resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        this.to('elasticache:AddTagsToResource');
        return this;
    }
    /**
     * The AuthorizeCacheSecurityGroupIngress action allows network ingress to a cache security group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupIngress
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_AuthorizeCacheSecurityGroupIngress.html
     */
    toAuthorizeCacheSecurityGroupIngress() {
        this.to('elasticache:AuthorizeCacheSecurityGroupIngress');
        return this;
    }
    /**
     * Apply the service update.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_BatchApplyUpdateAction.html
     */
    toBatchApplyUpdateAction() {
        this.to('elasticache:BatchApplyUpdateAction');
        return this;
    }
    /**
     * Stop the service update.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_BatchStopUpdateAction.html
     */
    toBatchStopUpdateAction() {
        this.to('elasticache:BatchStopUpdateAction');
        return this;
    }
    /**
     * Stop the service update.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CompleteMigration.html
     */
    toCompleteMigration() {
        this.to('elasticache:CompleteMigration');
        return this;
    }
    /**
     * The CopySnapshot action makes a copy of an existing snapshot.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     * - s3:DeleteObject
     * - s3:GetBucketAcl
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CopySnapshot.html
     */
    toCopySnapshot() {
        this.to('elasticache:CopySnapshot');
        return this;
    }
    /**
     * The CreateCacheCluster action creates a cache cluster.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - elasticache:AddTagsToResource
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheCluster.html
     */
    toCreateCacheCluster() {
        this.to('elasticache:CreateCacheCluster');
        return this;
    }
    /**
     * The CreateCacheParameterGroup action creates a new cache parameter group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheParameterGroup.html
     */
    toCreateCacheParameterGroup() {
        this.to('elasticache:CreateCacheParameterGroup');
        return this;
    }
    /**
     * The CreateCacheSecurityGroup action creates a new cache security group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheSecurityGroup.html
     */
    toCreateCacheSecurityGroup() {
        this.to('elasticache:CreateCacheSecurityGroup');
        return this;
    }
    /**
     * The CreateCacheSubnetGroup action creates a new cache subnet group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheSubnetGroup.html
     */
    toCreateCacheSubnetGroup() {
        this.to('elasticache:CreateCacheSubnetGroup');
        return this;
    }
    /**
     * The CreateGlobalReplicationGroup action creates a global datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateGlobalReplicationGroup.html
     */
    toCreateGlobalReplicationGroup() {
        this.to('elasticache:CreateGlobalReplicationGroup');
        return this;
    }
    /**
     * The CreateReplicationGroup action creates a replication group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - elasticache:AddTagsToResource
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateReplicationGroup.html
     */
    toCreateReplicationGroup() {
        this.to('elasticache:CreateReplicationGroup');
        return this;
    }
    /**
     * The CreateSnapshot action creates a copy of an entire cache cluster at a specific moment in time.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     * - s3:DeleteObject
     * - s3:GetBucketAcl
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        this.to('elasticache:CreateSnapshot');
        return this;
    }
    /**
     * The CreateUser action creates a new user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        this.to('elasticache:CreateUser');
        return this;
    }
    /**
     * The CreateUserGroup action creates a new user group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateUserGroup.html
     */
    toCreateUserGroup() {
        this.to('elasticache:CreateUserGroup');
        return this;
    }
    /**
     * The DecreaseNodeGroupsInGlobalReplicationGroup action dec a global datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DecreaseNodeGroupsInGlobalReplicationGroup.html
     */
    toDecreaseNodeGroupsInGlobalReplicationGroup() {
        this.to('elasticache:DecreaseNodeGroupsInGlobalReplicationGroup');
        return this;
    }
    /**
     * The DecreaseReplicaCount action decreases the number of replicas in a Redis replication group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DecreaseReplicaCount.html
     */
    toDecreaseReplicaCount() {
        this.to('elasticache:DecreaseReplicaCount');
        return this;
    }
    /**
     * The DeleteCacheCluster action deletes a previously provisioned cache cluster.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheCluster.html
     */
    toDeleteCacheCluster() {
        this.to('elasticache:DeleteCacheCluster');
        return this;
    }
    /**
     * The DeleteCacheParameterGroup action deletes the specified cache parameter group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheParameterGroup.html
     */
    toDeleteCacheParameterGroup() {
        this.to('elasticache:DeleteCacheParameterGroup');
        return this;
    }
    /**
     * The DeleteCacheSecurityGroup action deletes a cache security group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheSecurityGroup.html
     */
    toDeleteCacheSecurityGroup() {
        this.to('elasticache:DeleteCacheSecurityGroup');
        return this;
    }
    /**
     * The DeleteCacheSubnetGroup action deletes a cache subnet group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheSubnetGroup.html
     */
    toDeleteCacheSubnetGroup() {
        this.to('elasticache:DeleteCacheSubnetGroup');
        return this;
    }
    /**
     * The DeleteGlobalReplicationGroup action deletes a global datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteGlobalReplicationGroup.html
     */
    toDeleteGlobalReplicationGroup() {
        this.to('elasticache:DeleteGlobalReplicationGroup');
        return this;
    }
    /**
     * The DeleteReplicationGroup action deletes an existing replication group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteReplicationGroup.html
     */
    toDeleteReplicationGroup() {
        this.to('elasticache:DeleteReplicationGroup');
        return this;
    }
    /**
     * The DeleteSnapshot action deletes an existing snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteSnapshot.html
     */
    toDeleteSnapshot() {
        this.to('elasticache:DeleteSnapshot');
        return this;
    }
    /**
     * The DeleteUser action deletes an existing user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        this.to('elasticache:DeleteUser');
        return this;
    }
    /**
     * The DeleteUserGroup action deletes an existing user group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteUserGroup.html
     */
    toDeleteUserGroup() {
        this.to('elasticache:DeleteUserGroup');
        return this;
    }
    /**
     * The DescribeCacheClusters action returns information about all provisioned cache clusters if no cache cluster identifier is specified, or about a specific cache cluster if a cache cluster identifier is supplied.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheClusters.html
     */
    toDescribeCacheClusters() {
        this.to('elasticache:DescribeCacheClusters');
        return this;
    }
    /**
     * The DescribeCacheEngineVersions action returns a list of the available cache engines and their versions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheEngineVersions.html
     */
    toDescribeCacheEngineVersions() {
        this.to('elasticache:DescribeCacheEngineVersions');
        return this;
    }
    /**
     * The DescribeCacheParameterGroups action returns information about parameter groups for this account, or a particular parameter group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheParameterGroups.html
     */
    toDescribeCacheParameterGroups() {
        this.to('elasticache:DescribeCacheParameterGroups');
        return this;
    }
    /**
     * The DescribeCacheParameters action returns the detailed parameter list for a particular cache parameter group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheParameters.html
     */
    toDescribeCacheParameters() {
        this.to('elasticache:DescribeCacheParameters');
        return this;
    }
    /**
     * The DescribeCacheSecurityGroups action returns a list of cache security group descriptions, or the description of the specified security group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheSecurityGroups.html
     */
    toDescribeCacheSecurityGroups() {
        this.to('elasticache:DescribeCacheSecurityGroups');
        return this;
    }
    /**
     * The DescribeCacheSubnetGroups action returns a list of cache subnet group descriptions, or the description of the specified subnet group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheSubnetGroups.html
     */
    toDescribeCacheSubnetGroups() {
        this.to('elasticache:DescribeCacheSubnetGroups');
        return this;
    }
    /**
     * The DescribeEngineDefaultParameters action returns the default engine and system parameter information for the specified cache engine.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeEngineDefaultParameters.html
     */
    toDescribeEngineDefaultParameters() {
        this.to('elasticache:DescribeEngineDefaultParameters');
        return this;
    }
    /**
     * The DescribeEvents action returns events related to cache clusters, cache security groups, and cache parameter groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents() {
        this.to('elasticache:DescribeEvents');
        return this;
    }
    /**
     * The DescribeGlobalReplicationGroups action returns information about global datastores for this account, or a particular global datastore.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeGlobalReplicationGroups.html
     */
    toDescribeGlobalReplicationGroups() {
        this.to('elasticache:DescribeGlobalReplicationGroups');
        return this;
    }
    /**
     * The DescribeReplicationGroups action returns information about replication groups for this account, or a particular replication group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeReplicationGroups.html
     */
    toDescribeReplicationGroups() {
        this.to('elasticache:DescribeReplicationGroups');
        return this;
    }
    /**
     * The DescribeReservedCacheNodes action returns information about reserved cache nodes for this account, or a particular reserved cache node.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeReservedCacheNodes.html
     */
    toDescribeReservedCacheNodes() {
        this.to('elasticache:DescribeReservedCacheNodes');
        return this;
    }
    /**
     * The DescribeReservedCacheNodesOfferings action lists available reserved cache node offerings.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeReservedCacheNodesOfferings.html
     */
    toDescribeReservedCacheNodesOfferings() {
        this.to('elasticache:DescribeReservedCacheNodesOfferings');
        return this;
    }
    /**
     * Returns details of the service updates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeServiceUpdates.html
     */
    toDescribeServiceUpdates() {
        this.to('elasticache:DescribeServiceUpdates');
        return this;
    }
    /**
     * The DescribeSnapshots action returns information about cache cluster snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeSnapshots.html
     */
    toDescribeSnapshots() {
        this.to('elasticache:DescribeSnapshots');
        return this;
    }
    /**
     * Returns details of the update actions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeUpdateActions.html
     */
    toDescribeUpdateActions() {
        this.to('elasticache:DescribeUpdateActions');
        return this;
    }
    /**
     * The DescribeUserGroups action returns information about all user groups for this account, or a particular user group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeUserGroups.html
     */
    toDescribeUserGroups() {
        this.to('elasticache:DescribeUserGroups');
        return this;
    }
    /**
     * The DescribeUsers action returns information about all users for this account, or a particular user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeUsers.html
     */
    toDescribeUsers() {
        this.to('elasticache:DescribeUsers');
        return this;
    }
    /**
     * The DisassociateGlobalReplicationGroup action removes a secondary Replication Group from the Global Datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DisassociateGlobalReplicationGroup.html
     */
    toDisassociateGlobalReplicationGroup() {
        this.to('elasticache:DisassociateGlobalReplicationGroup');
        return this;
    }
    /**
     * The FailoverGlobalReplicationGroup action removes a secondary Replication Group from the Global Datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_FailoverGlobalReplicationGroup.html
     */
    toFailoverGlobalReplicationGroup() {
        this.to('elasticache:FailoverGlobalReplicationGroup');
        return this;
    }
    /**
     * The IncreaseNodeGroupsInGlobalReplicationGroup action increases the number of node groups in the Global Datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_IncreaseNodeGroupsInGlobalReplicationGroup.html
     */
    toIncreaseNodeGroupsInGlobalReplicationGroup() {
        this.to('elasticache:IncreaseNodeGroupsInGlobalReplicationGroup');
        return this;
    }
    /**
     * The IncreaseReplicaCount action increases the number of replicas in a Redis replication group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_IncreaseReplicaCount.html
     */
    toIncreaseReplicaCount() {
        this.to('elasticache:IncreaseReplicaCount');
        return this;
    }
    /**
     * List Allowed Node Type Modifications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ListAllowedNodeTypeModifications.html
     */
    toListAllowedNodeTypeModifications() {
        this.to('elasticache:ListAllowedNodeTypeModifications');
        return this;
    }
    /**
     * The ListTagsForResource action lists all cost allocation tags currently on the named resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('elasticache:ListTagsForResource');
        return this;
    }
    /**
     * The ModifyCacheCluster action modifies the settings for a cache cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheCluster.html
     */
    toModifyCacheCluster() {
        this.to('elasticache:ModifyCacheCluster');
        return this;
    }
    /**
     * The ModifyCacheParameterGroup action modifies the parameters of a cache parameter group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheParameterGroup.html
     */
    toModifyCacheParameterGroup() {
        this.to('elasticache:ModifyCacheParameterGroup');
        return this;
    }
    /**
     * The ModifyCacheSubnetGroup action modifies an existing cache subnet group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheSubnetGroup.html
     */
    toModifyCacheSubnetGroup() {
        this.to('elasticache:ModifyCacheSubnetGroup');
        return this;
    }
    /**
     * The ModifyGlobalReplicationGroup action modifies the settings for a Global Datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyGlobalReplicationGroup.html
     */
    toModifyGlobalReplicationGroup() {
        this.to('elasticache:ModifyGlobalReplicationGroup');
        return this;
    }
    /**
     * The ModifyReplicationGroup action modifies the settings for a replication group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyReplicationGroup.html
     */
    toModifyReplicationGroup() {
        this.to('elasticache:ModifyReplicationGroup');
        return this;
    }
    /**
     * The ModifyReplicationGroupShardConfiguration action allows you to add shards, remove shards, or rebalance the keyspaces among exisiting shards.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyReplicationGroupShardConfiguration.html
     */
    toModifyReplicationGroupShardConfiguration() {
        this.to('elasticache:ModifyReplicationGroupShardConfiguration');
        return this;
    }
    /**
     * The ModifyUser action modifies an existing user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyUser.html
     */
    toModifyUser() {
        this.to('elasticache:ModifyUser');
        return this;
    }
    /**
     * The ModifyUserGroup action modifies an existing user group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyUserGroup.html
     */
    toModifyUserGroup() {
        this.to('elasticache:ModifyUserGroup');
        return this;
    }
    /**
     * The PurchaseReservedCacheNodesOffering action allows you to purchase a reserved cache node offering.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_PurchaseReservedCacheNodesOffering.html
     */
    toPurchaseReservedCacheNodesOffering() {
        this.to('elasticache:PurchaseReservedCacheNodesOffering');
        return this;
    }
    /**
     * The RebalanceSlotsInGlobalReplicationGroup action redistributes slots to ensure uniform distribution across existing shards in the cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RebalanceSlotsInGlobalReplicationGroup.html
     */
    toRebalanceSlotsInGlobalReplicationGroup() {
        this.to('elasticache:RebalanceSlotsInGlobalReplicationGroup');
        return this;
    }
    /**
     * The RebootCacheCluster action reboots some, or all, of the cache nodes within a provisioned cache cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RebootCacheCluster.html
     */
    toRebootCacheCluster() {
        this.to('elasticache:RebootCacheCluster');
        return this;
    }
    /**
     * The RemoveTagsFromResource action removes the tags identified by the TagKeys list from the named resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        this.to('elasticache:RemoveTagsFromResource');
        return this;
    }
    /**
     * The ResetCacheParameterGroup action modifies the parameters of a cache parameter group to the engine or system default value.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ResetCacheParameterGroup.html
     */
    toResetCacheParameterGroup() {
        this.to('elasticache:ResetCacheParameterGroup');
        return this;
    }
    /**
     * The RevokeCacheSecurityGroupIngress action revokes ingress from a cache security group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RevokeCacheSecurityGroupIngress.html
     */
    toRevokeCacheSecurityGroupIngress() {
        this.to('elasticache:RevokeCacheSecurityGroupIngress');
        return this;
    }
    /**
     * Start the migration of data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_StartMigration.html
     */
    toStartMigration() {
        this.to('elasticache:StartMigration');
        return this;
    }
    /**
     * The TestFailover action allows you to test automatic failover on a specified node group in a replication group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_TestFailover.html
     */
    toTestFailover() {
        this.to('elasticache:TestFailover');
        return this;
    }
    /**
     * Adds a resource of type parametergroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html
     *
     * @param cacheParameterGroupName - Identifier for the cacheParameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onParametergroup(cacheParameterGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:parametergroup:${CacheParameterGroupName}';
        arn = arn.replace('${CacheParameterGroupName}', cacheParameterGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type securitygroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html
     *
     * @param cacheSecurityGroupName - Identifier for the cacheSecurityGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSecuritygroup(cacheSecurityGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:securitygroup:${CacheSecurityGroupName}';
        arn = arn.replace('${CacheSecurityGroupName}', cacheSecurityGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subnetgroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html
     *
     * @param cacheSubnetGroupName - Identifier for the cacheSubnetGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSubnetgroup(cacheSubnetGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:subnetgroup:${CacheSubnetGroupName}';
        arn = arn.replace('${CacheSubnetGroupName}', cacheSubnetGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type replicationgroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Replication.html
     *
     * @param replicationGroupId - Identifier for the replicationGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReplicationgroup(replicationGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:replicationgroup:${ReplicationGroupId}';
        arn = arn.replace('${ReplicationGroupId}', replicationGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html
     *
     * @param cacheClusterId - Identifier for the cacheClusterId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCluster(cacheClusterId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:cluster:${CacheClusterId}';
        arn = arn.replace('${CacheClusterId}', cacheClusterId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type reserved-instance to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/reserved-nodes.html
     *
     * @param reservedCacheNodeId - Identifier for the reservedCacheNodeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReservedInstance(reservedCacheNodeId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:reserved-instance:${ReservedCacheNodeId}';
        arn = arn.replace('${ReservedCacheNodeId}', reservedCacheNodeId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/backups.html
     *
     * @param snapshotName - Identifier for the snapshotName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSnapshot(snapshotName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:snapshot:${SnapshotName}';
        arn = arn.replace('${SnapshotName}', snapshotName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type globalreplicationgroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Redis-Global-Datastore.html
     *
     * @param globalReplicationGroupId - Identifier for the globalReplicationGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGlobalreplicationgroup(globalReplicationGroupId, account, partition) {
        var arn = 'arn:${Partition}:elasticache::${Account}:globalreplicationgroup:${GlobalReplicationGroupId}';
        arn = arn.replace('${GlobalReplicationGroupId}', globalReplicationGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html
     *
     * @param userId - Identifier for the userId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onUser(userId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:user:${UserId}';
        arn = arn.replace('${UserId}', userId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type usergroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html
     *
     * @param userGroupId - Identifier for the userGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onUsergroup(userGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:usergroup:${UserGroupId}';
        arn = arn.replace('${UserGroupId}', userGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Elasticache = Elasticache;
//# sourceMappingURL=data:application/json;base64,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