"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mediapackage = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mediapackage](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediapackage.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mediapackage extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mediapackage](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediapackage.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mediapackage';
        this.accessLevelList = {
            "Write": [
                "CreateChannel",
                "CreateHarvestJob",
                "CreateOriginEndpoint",
                "DeleteChannel",
                "DeleteOriginEndpoint",
                "RotateIngestEndpointCredentials",
                "TagResource",
                "UntagResource",
                "UpdateChannel",
                "UpdateOriginEndpoint"
            ],
            "Read": [
                "DescribeChannel",
                "DescribeHarvestJob",
                "DescribeOriginEndpoint",
                "ListChannels",
                "ListHarvestJobs",
                "ListOriginEndpoints",
                "ListTagsForResource"
            ]
        };
    }
    /**
     * Grants permission to create a channel in AWS Elemental MediaPackage.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels.html#channelspost
     */
    toCreateChannel() {
        this.to('mediapackage:CreateChannel');
        return this;
    }
    /**
     * Grants permission to create a harvest job in AWS Elemental MediaPackage.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/harvest_jobs.html#harvest_jobspost
     */
    toCreateHarvestJob() {
        this.to('mediapackage:CreateHarvestJob');
        return this;
    }
    /**
     * Grants permission to create an endpoint in AWS Elemental MediaPackage.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints.html#origin_endpointspost
     */
    toCreateOriginEndpoint() {
        this.to('mediapackage:CreateOriginEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a channel in AWS Elemental MediaPackage.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id.html#channels-iddelete
     */
    toDeleteChannel() {
        this.to('mediapackage:DeleteChannel');
        return this;
    }
    /**
     * Grants permission to delete an endpoint in AWS Elemental MediaPackage.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints-id.html#origin_endpoints-iddelete
     */
    toDeleteOriginEndpoint() {
        this.to('mediapackage:DeleteOriginEndpoint');
        return this;
    }
    /**
     * Grants permission to view the details of a channel in AWS Elemental MediaPackage.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id.html#channels-idget
     */
    toDescribeChannel() {
        this.to('mediapackage:DescribeChannel');
        return this;
    }
    /**
     * Grants permission to view the details of a harvest job in AWS Elemental MediaPackage.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/harvest_jobs-id.html#harvest_jobs-idget
     */
    toDescribeHarvestJob() {
        this.to('mediapackage:DescribeHarvestJob');
        return this;
    }
    /**
     * Grants permission to view the details of an endpoint in AWS Elemental MediaPackage.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints-id.html#origin_endpoints-idget
     */
    toDescribeOriginEndpoint() {
        this.to('mediapackage:DescribeOriginEndpoint');
        return this;
    }
    /**
     * Grants permission to view a list of channels in AWS Elemental MediaPackage.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels.html#channelsget
     */
    toListChannels() {
        this.to('mediapackage:ListChannels');
        return this;
    }
    /**
     * Grants permission to view a list of harvest jobs in AWS Elemental MediaPackage.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/harvest_jobs.html#harvest_jobsget
     */
    toListHarvestJobs() {
        this.to('mediapackage:ListHarvestJobs');
        return this;
    }
    /**
     * Grants permission to view a list of endpoints in AWS Elemental MediaPackage.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints.html#origin_endpointsget
     */
    toListOriginEndpoints() {
        this.to('mediapackage:ListOriginEndpoints');
        return this;
    }
    /**
     * Grants permission to list the tags assigned to a Channel or OriginEndpoint.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/tags-resource-arn.html#tags-resource-arnget
     */
    toListTagsForResource() {
        this.to('mediapackage:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to rotate IngestEndpoint credentials for a Channel in AWS Elemental MediaPackage.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id-ingest_endpoints-ingest_endpoint_id-credentials.html#channels-id-ingest_endpoints-ingest_endpoint_id-credentialsput
     */
    toRotateIngestEndpointCredentials() {
        this.to('mediapackage:RotateIngestEndpointCredentials');
        return this;
    }
    /**
     *
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/hj-create.html
     */
    toTagResource() {
        this.to('mediapackage:TagResource');
        return this;
    }
    /**
     * Grants permission to delete tags to a Channel or OriginEndpoint.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/tags-resource-arn.html#tags-resource-arndelete
     */
    toUntagResource() {
        this.to('mediapackage:UntagResource');
        return this;
    }
    /**
     * Grants permission to make changes to a channel in AWS Elemental MediaPackage.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id.html#channels-idput
     */
    toUpdateChannel() {
        this.to('mediapackage:UpdateChannel');
        return this;
    }
    /**
     * Grants permission to make changes to an endpoint in AWS Elemental MediaPackage.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints-id.html#origin_endpoints-idput
     */
    toUpdateOriginEndpoint() {
        this.to('mediapackage:UpdateOriginEndpoint');
        return this;
    }
    /**
     * Adds a resource of type channels to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/channels.html
     *
     * @param channelIdentifier - Identifier for the channelIdentifier.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannels(channelIdentifier, account, region, partition) {
        var arn = 'arn:${Partition}:mediapackage:${Region}:${Account}:channels/${ChannelIdentifier}';
        arn = arn.replace('${ChannelIdentifier}', channelIdentifier);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type origin_endpoints to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/endpoints.html
     *
     * @param originEndpointIdentifier - Identifier for the originEndpointIdentifier.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOriginEndpoints(originEndpointIdentifier, account, region, partition) {
        var arn = 'arn:${Partition}:mediapackage:${Region}:${Account}:origin_endpoints/${OriginEndpointIdentifier}';
        arn = arn.replace('${OriginEndpointIdentifier}', originEndpointIdentifier);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type harvest_jobs to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/harvest-jobs.html
     *
     * @param harvestJobIdentifier - Identifier for the harvestJobIdentifier.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHarvestJobs(harvestJobIdentifier, account, region, partition) {
        var arn = 'arn:${Partition}:mediapackage:${Region}:${Account}:harvest_jobs/${HarvestJobIdentifier}';
        arn = arn.replace('${HarvestJobIdentifier}', harvestJobIdentifier);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mediapackage = Mediapackage;
//# sourceMappingURL=data:application/json;base64,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