"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mediastore = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mediastore](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediastore.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mediastore extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mediastore](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediastore.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mediastore';
        this.accessLevelList = {
            "Write": [
                "CreateContainer",
                "DeleteContainer",
                "DeleteCorsPolicy",
                "DeleteLifecyclePolicy",
                "DeleteMetricPolicy",
                "DeleteObject",
                "PutCorsPolicy",
                "PutLifecyclePolicy",
                "PutMetricPolicy",
                "PutObject",
                "StartAccessLogging",
                "StopAccessLogging"
            ],
            "Permissions management": [
                "DeleteContainerPolicy",
                "PutContainerPolicy"
            ],
            "List": [
                "DescribeContainer",
                "DescribeObject",
                "ListContainers",
                "ListItems"
            ],
            "Read": [
                "GetContainerPolicy",
                "GetCorsPolicy",
                "GetLifecyclePolicy",
                "GetMetricPolicy",
                "GetObject",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to create containers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_CreateContainer.html
     */
    toCreateContainer() {
        this.to('mediastore:CreateContainer');
        return this;
    }
    /**
     * Grants permission to delete any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainer.html
     */
    toDeleteContainer() {
        this.to('mediastore:DeleteContainer');
        return this;
    }
    /**
     * Grants permission to delete the access policy of any container in the current account.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainerPolicy.html
     */
    toDeleteContainerPolicy() {
        this.to('mediastore:DeleteContainerPolicy');
        return this;
    }
    /**
     * Grants permission to delete the CORS policy from any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteCorsPolicy.html
     */
    toDeleteCorsPolicy() {
        this.to('mediastore:DeleteCorsPolicy');
        return this;
    }
    /**
     * Grants permission to delete the lifecycle policy from any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteLifecyclePolicy.html
     */
    toDeleteLifecyclePolicy() {
        this.to('mediastore:DeleteLifecyclePolicy');
        return this;
    }
    /**
     * Grants permission to delete the metric policy from any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteMetricPolicy.html
     */
    toDeleteMetricPolicy() {
        this.to('mediastore:DeleteMetricPolicy');
        return this;
    }
    /**
     * Grants permission to delete objects.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DeleteObject.html
     */
    toDeleteObject() {
        this.to('mediastore:DeleteObject');
        return this;
    }
    /**
     * Grants permission to retrieve details on any container in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DescribeContainer.html
     */
    toDescribeContainer() {
        this.to('mediastore:DescribeContainer');
        return this;
    }
    /**
     * Grants permission to retrieve object metadata.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DescribeObject.html
     */
    toDescribeObject() {
        this.to('mediastore:DescribeObject');
        return this;
    }
    /**
     * Grants permission to retrieve the access policy of any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetContainerPolicy.html
     */
    toGetContainerPolicy() {
        this.to('mediastore:GetContainerPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the CORS policy of any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetCorsPolicy.html
     */
    toGetCorsPolicy() {
        this.to('mediastore:GetCorsPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the lifecycle policy that is assigned to any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetLifecyclePolicy.html
     */
    toGetLifecyclePolicy() {
        this.to('mediastore:GetLifecyclePolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the metric policy that is assigned to any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetMetricPolicy.html
     */
    toGetMetricPolicy() {
        this.to('mediastore:GetMetricPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve objects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_GetObject.html
     */
    toGetObject() {
        this.to('mediastore:GetObject');
        return this;
    }
    /**
     * Grants permission to retrieve a list of containers in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListContainers.html
     */
    toListContainers() {
        this.to('mediastore:ListContainers');
        return this;
    }
    /**
     * Grants permission to retrieve a list of objects and folders in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_ListItems.html
     */
    toListItems() {
        this.to('mediastore:ListItems');
        return this;
    }
    /**
     * Grants permission to list tags on any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('mediastore:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to create or replace the access policy of any container in the current account.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutContainerPolicy.html
     */
    toPutContainerPolicy() {
        this.to('mediastore:PutContainerPolicy');
        return this;
    }
    /**
     * Grants permission to add or modify the CORS policy of any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutCorsPolicy.html
     */
    toPutCorsPolicy() {
        this.to('mediastore:PutCorsPolicy');
        return this;
    }
    /**
     * Grants permission to add or modify the lifecycle policy that is assigned to any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutLifecyclePolicy.html
     */
    toPutLifecyclePolicy() {
        this.to('mediastore:PutLifecyclePolicy');
        return this;
    }
    /**
     * Grants permission to add or modify the metric policy that is assigned to any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutMetricPolicy.html
     */
    toPutMetricPolicy() {
        this.to('mediastore:PutMetricPolicy');
        return this;
    }
    /**
     * Grants permission to upload objects.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_PutObject.html
     */
    toPutObject() {
        this.to('mediastore:PutObject');
        return this;
    }
    /**
     * Grants permission to enable access logging on any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_StartAccessLogging.html
     */
    toStartAccessLogging() {
        this.to('mediastore:StartAccessLogging');
        return this;
    }
    /**
     * Grants permission to disable access logging on any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_StopAccessLogging.html
     */
    toStopAccessLogging() {
        this.to('mediastore:StopAccessLogging');
        return this;
    }
    /**
     * Grants permission to add tags to any container in the current account.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        this.to('mediastore:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from any container in the current account.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('mediastore:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type container to the statement
     *
     * https://docs.aws.amazon.com/mediastore/latest/ug/containers.html
     *
     * @param containerName - Identifier for the containerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onContainer(containerName, account, region, partition) {
        var arn = 'arn:${Partition}:mediastore:${Region}:${Account}:container/${ContainerName}';
        arn = arn.replace('${ContainerName}', containerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mediastore = Mediastore;
//# sourceMappingURL=data:application/json;base64,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