"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schemas = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [schemas](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoneventbridgeschemas.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Schemas extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [schemas](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoneventbridgeschemas.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'schemas';
        this.accessLevelList = {
            "Write": [
                "CreateDiscoverer",
                "CreateRegistry",
                "CreateSchema",
                "DeleteDiscoverer",
                "DeleteRegistry",
                "DeleteResourcePolicy",
                "DeleteSchema",
                "DeleteSchemaVersion",
                "PutCodeBinding",
                "PutResourcePolicy",
                "StartDiscoverer",
                "StopDiscoverer",
                "UpdateDiscoverer",
                "UpdateRegistry",
                "UpdateSchema"
            ],
            "Read": [
                "DescribeCodeBinding",
                "DescribeDiscoverer",
                "DescribeRegistry",
                "DescribeSchema",
                "ExportSchema",
                "GetCodeBindingSource",
                "GetDiscoveredSchema",
                "GetResourcePolicy"
            ],
            "List": [
                "ListDiscoverers",
                "ListRegistries",
                "ListSchemaVersions",
                "ListSchemas",
                "ListTagsForResource",
                "SearchSchemas"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Creates an event schema discoverer. Once created, your events will be automatically map into corresponding schema documents
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#CreateDiscoverer
     */
    toCreateDiscoverer() {
        this.to('schemas:CreateDiscoverer');
        return this;
    }
    /**
     * Create a new schema registry in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#CreateRegistry
     */
    toCreateRegistry() {
        this.to('schemas:CreateRegistry');
        return this;
    }
    /**
     * Create a new schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#CreateSchema
     */
    toCreateSchema() {
        this.to('schemas:CreateSchema');
        return this;
    }
    /**
     * Deletes discoverer in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#DeleteDiscoverer
     */
    toDeleteDiscoverer() {
        this.to('schemas:DeleteDiscoverer');
        return this;
    }
    /**
     * Deletes an existing registry in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#DeleteRegistry
     */
    toDeleteRegistry() {
        this.to('schemas:DeleteRegistry');
        return this;
    }
    /**
     * Delete the resource-based policy attached to a given registry.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#DeleteResourcePolicy
     */
    toDeleteResourcePolicy() {
        this.to('schemas:DeleteResourcePolicy');
        return this;
    }
    /**
     * Deletes an existing schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#DeleteSchema
     */
    toDeleteSchema() {
        this.to('schemas:DeleteSchema');
        return this;
    }
    /**
     * Deletes a specific version of schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-version-schemaversion.html#DeleteSchemaVersion
     */
    toDeleteSchemaVersion() {
        this.to('schemas:DeleteSchemaVersion');
        return this;
    }
    /**
     * Retrieves metadata for generated code for specific schema in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language.html#DescribeCodeBinding
     */
    toDescribeCodeBinding() {
        this.to('schemas:DescribeCodeBinding');
        return this;
    }
    /**
     * Retrieves discoverer metadata in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#DescribeDiscoverer
     */
    toDescribeDiscoverer() {
        this.to('schemas:DescribeDiscoverer');
        return this;
    }
    /**
     * Describes an existing registry metadata in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#DescribeRegistry
     */
    toDescribeRegistry() {
        this.to('schemas:DescribeRegistry');
        return this;
    }
    /**
     * Retrieves an existing schema in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#DescribeSchema
     */
    toDescribeSchema() {
        this.to('schemas:DescribeSchema');
        return this;
    }
    /**
     * Allows exporting AWS registry or discovered schemas in OpenAPI 3 format to JSONSchema format.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#ExportSchema
     */
    toExportSchema() {
        this.to('schemas:ExportSchema');
        return this;
    }
    /**
     * Retrieves metadata for generated code for specific schema in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language-source.html#GetCodeBindingSource
     */
    toGetCodeBindingSource() {
        this.to('schemas:GetCodeBindingSource');
        return this;
    }
    /**
     * Retrieves schema for the provided list of sample events.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discover.html#GetDiscoveredSchema
     */
    toGetDiscoveredSchema() {
        this.to('schemas:GetDiscoveredSchema');
        return this;
    }
    /**
     * Retrieves the resource-based policy attached to a given registry.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#GetResourcePolicy
     */
    toGetResourcePolicy() {
        this.to('schemas:GetResourcePolicy');
        return this;
    }
    /**
     * Lists all the discoverers in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers.html#ListDiscoverers
     */
    toListDiscoverers() {
        this.to('schemas:ListDiscoverers');
        return this;
    }
    /**
     * List all discoverers in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries.html#ListRegistries
     */
    toListRegistries() {
        this.to('schemas:ListRegistries');
        return this;
    }
    /**
     * List all versions of a schema.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-versions.html#ListSchemaVersions
     */
    toListSchemaVersions() {
        this.to('schemas:ListSchemaVersions');
        return this;
    }
    /**
     * List all schemas.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas.html#ListSchemas
     */
    toListSchemas() {
        this.to('schemas:ListSchemas');
        return this;
    }
    /**
     * This action lists tags for a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#ListTagsForResource
     */
    toListTagsForResource() {
        this.to('schemas:ListTagsForResource');
        return this;
    }
    /**
     * Generates code for specific schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language.html#PutCodeBinding
     */
    toPutCodeBinding() {
        this.to('schemas:PutCodeBinding');
        return this;
    }
    /**
     * Attach resource-based policy to the specific registry.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#PutResourcePolicy
     */
    toPutResourcePolicy() {
        this.to('schemas:PutResourcePolicy');
        return this;
    }
    /**
     * Searches schemas based on specified keywords in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-search.html#SearchSchemas
     */
    toSearchSchemas() {
        this.to('schemas:SearchSchemas');
        return this;
    }
    /**
     * Starts the specified discoverer. Once started the discoverer will automatically register schemas for published events to configured source in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#StartDiscoverer
     */
    toStartDiscoverer() {
        this.to('schemas:StartDiscoverer');
        return this;
    }
    /**
     * Starts the specified discoverer. Once started the discoverer will automatically register schemas for published events to configured source in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#StopDiscoverer
     */
    toStopDiscoverer() {
        this.to('schemas:StopDiscoverer');
        return this;
    }
    /**
     * This action tags an resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#TagResource
     */
    toTagResource() {
        this.to('schemas:TagResource');
        return this;
    }
    /**
     * This action removes a tag from on a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#UntagResource
     */
    toUntagResource() {
        this.to('schemas:UntagResource');
        return this;
    }
    /**
     * Updates an existing discoverer in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#UpdateDiscoverer
     */
    toUpdateDiscoverer() {
        this.to('schemas:UpdateDiscoverer');
        return this;
    }
    /**
     * Updates an existing registry metadata in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#UpdateRegistry
     */
    toUpdateRegistry() {
        this.to('schemas:UpdateRegistry');
        return this;
    }
    /**
     * Updates an existing schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#UpdateSchema
     */
    toUpdateSchema() {
        this.to('schemas:UpdateSchema');
        return this;
    }
    /**
     * Adds a resource of type discoverer to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html
     *
     * @param discovererId - Identifier for the discovererId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDiscoverer(discovererId, account, region, partition) {
        var arn = 'arn:${Partition}:schemas:${Region}:${Account}:discoverer/${DiscovererId}';
        arn = arn.replace('${DiscovererId}', discovererId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type registry to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html
     *
     * @param registryName - Identifier for the registryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRegistry(registryName, account, region, partition) {
        var arn = 'arn:${Partition}:schemas:${Region}:${Account}:registry/${RegistryName}';
        arn = arn.replace('${RegistryName}', registryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type schema to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html
     *
     * @param registryName - Identifier for the registryName.
     * @param schemaName - Identifier for the schemaName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSchema(registryName, schemaName, account, region, partition) {
        var arn = 'arn:${Partition}:schemas:${Region}:${Account}:schema/${RegistryName}/${SchemaName}';
        arn = arn.replace('${RegistryName}', registryName);
        arn = arn.replace('${SchemaName}', schemaName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Schemas = Schemas;
//# sourceMappingURL=data:application/json;base64,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