"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Fms = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [fms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsfirewallmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Fms extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [fms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsfirewallmanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'fms';
        this.accessLevelList = {
            "Write": [
                "AssociateAdminAccount",
                "DeleteAppsList",
                "DeleteNotificationChannel",
                "DeletePolicy",
                "DeleteProtocolsList",
                "DisassociateAdminAccount",
                "PutAppsList",
                "PutNotificationChannel",
                "PutPolicy",
                "PutProtocolsList"
            ],
            "Read": [
                "GetAdminAccount",
                "GetAppsList",
                "GetComplianceDetail",
                "GetNotificationChannel",
                "GetPolicy",
                "GetProtectionStatus",
                "GetProtocolsList",
                "ListTagsForResource"
            ],
            "List": [
                "ListAppsLists",
                "ListComplianceStatus",
                "ListMemberAccounts",
                "ListPolicies",
                "ListProtocolsLists"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Sets the AWS Firewall Manager administrator account and enables the service in all organization accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_AssociateAdminAccount.html
     */
    toAssociateAdminAccount() {
        this.to('fms:AssociateAdminAccount');
        return this;
    }
    /**
     * Grants permission to permanently deletes an AWS Firewall Manager applications list.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeleteAppsList.html
     */
    toDeleteAppsList() {
        this.to('fms:DeleteAppsList');
        return this;
    }
    /**
     * Deletes an AWS Firewall Manager association with the IAM role and the Amazon Simple Notification Service (SNS) topic that is used to notify the FM administrator about major FM events and errors across the organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeleteNotificationChannel.html
     */
    toDeleteNotificationChannel() {
        this.to('fms:DeleteNotificationChannel');
        return this;
    }
    /**
     * Permanently deletes an AWS Firewall Manager policy.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        this.to('fms:DeletePolicy');
        return this;
    }
    /**
     * Grants permission to permanently deletes an AWS Firewall Manager protocols list.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeleteProtocolsList.html
     */
    toDeleteProtocolsList() {
        this.to('fms:DeleteProtocolsList');
        return this;
    }
    /**
     * Disassociates the account that has been set as the AWS Firewall Manager administrator account and and disables the service in all organization accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DisassociateAdminAccount.html
     */
    toDisassociateAdminAccount() {
        this.to('fms:DisassociateAdminAccount');
        return this;
    }
    /**
     * Returns the AWS Organizations master account that is associated with AWS Firewall Manager as the AWS Firewall Manager administrator.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetAdminAccount.html
     */
    toGetAdminAccount() {
        this.to('fms:GetAdminAccount');
        return this;
    }
    /**
     * Grants permission to return information about the specified AWS Firewall Manager applications list.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetAppsList.html
     */
    toGetAppsList() {
        this.to('fms:GetAppsList');
        return this;
    }
    /**
     * Returns detailed compliance information about the specified member account. Details include resources that are in and out of compliance with the specified policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetComplianceDetail.html
     */
    toGetComplianceDetail() {
        this.to('fms:GetComplianceDetail');
        return this;
    }
    /**
     * Returns information about the Amazon Simple Notification Service (SNS) topic that is used to record AWS Firewall Manager SNS logs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetNotificationChannel.html
     */
    toGetNotificationChannel() {
        this.to('fms:GetNotificationChannel');
        return this;
    }
    /**
     * Returns information about the specified AWS Firewall Manager policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetPolicy.html
     */
    toGetPolicy() {
        this.to('fms:GetPolicy');
        return this;
    }
    /**
     * Returns policy-level attack summary information in the event of a potential DDoS attack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetProtectionStatus.html
     */
    toGetProtectionStatus() {
        this.to('fms:GetProtectionStatus');
        return this;
    }
    /**
     * Grants permission to return information about the specified AWS Firewall Manager protocols list.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetProtocolsList.html
     */
    toGetProtocolsList() {
        this.to('fms:GetProtocolsList');
        return this;
    }
    /**
     * Grants permission to return an array of AppsListDataSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListAppsLists.html
     */
    toListAppsLists() {
        this.to('fms:ListAppsLists');
        return this;
    }
    /**
     * Returns an array of PolicyComplianceStatus objects in the response. Use PolicyComplianceStatus to get a summary of which member accounts are protected by the specified policy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListComplianceStatus.html
     */
    toListComplianceStatus() {
        this.to('fms:ListComplianceStatus');
        return this;
    }
    /**
     * Returns an array of member account ids if the caller is FMS admin account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListMemberAccounts.html
     */
    toListMemberAccounts() {
        this.to('fms:ListMemberAccounts');
        return this;
    }
    /**
     * Returns an array of PolicySummary objects in the response.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListPolicies.html
     */
    toListPolicies() {
        this.to('fms:ListPolicies');
        return this;
    }
    /**
     * Grants permission to return an array of ProtocolsListDataSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListProtocolsLists.html
     */
    toListProtocolsLists() {
        this.to('fms:ListProtocolsLists');
        return this;
    }
    /**
     * Lists the Tags for a given resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('fms:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to create an AWS Firewall Manager applications list.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutAppsList.html
     */
    toPutAppsList() {
        this.to('fms:PutAppsList');
        return this;
    }
    /**
     * Designates the IAM role and Amazon Simple Notification Service (SNS) topic that AWS Firewall Manager (FM) could use to notify the FM administrator about major FM events and errors across the organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutNotificationChannel.html
     */
    toPutNotificationChannel() {
        this.to('fms:PutNotificationChannel');
        return this;
    }
    /**
     * Creates an AWS Firewall Manager policy.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutPolicy.html
     */
    toPutPolicy() {
        this.to('fms:PutPolicy');
        return this;
    }
    /**
     * Grants permission to creates an AWS Firewall Manager protocols list.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutProtocolsList.html
     */
    toPutProtocolsList() {
        this.to('fms:PutProtocolsList');
        return this;
    }
    /**
     * Adds a Tag to a given resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('fms:TagResource');
        return this;
    }
    /**
     * Removes a Tag from a given resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('fms:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type policy to the statement
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_Policy.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPolicy(id, account, region, partition) {
        var arn = 'arn:${Partition}:fms:${Region}:${Account}:policy/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type applications-list to the statement
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_AppsListData.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplicationsList(id, account, region, partition) {
        var arn = 'arn:${Partition}:fms:${Region}:${Account}:applications-list/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type protocols-list to the statement
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ProtocolsListData.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProtocolsList(id, account, region, partition) {
        var arn = 'arn:${Partition}:fms:${Region}:${Account}:protocols-list/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Fms = Fms;
//# sourceMappingURL=data:application/json;base64,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