"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Globalaccelerator = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [globalaccelerator](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsglobalaccelerator.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Globalaccelerator extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [globalaccelerator](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsglobalaccelerator.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'globalaccelerator';
        this.accessLevelList = {
            "Write": [
                "AdvertiseByoipCidr",
                "CreateAccelerator",
                "CreateEndpointGroup",
                "CreateListener",
                "DeleteAccelerator",
                "DeleteEndpointGroup",
                "DeleteListener",
                "DeprovisionByoipCidr",
                "ProvisionByoipCidr",
                "UpdateAccelerator",
                "UpdateAcceleratorAttributes",
                "UpdateEndpointGroup",
                "UpdateListener",
                "WithdrawByoipCidr"
            ],
            "Read": [
                "DescribeAccelerator",
                "DescribeAcceleratorAttributes",
                "DescribeEndpointGroup",
                "DescribeListener",
                "ListTagsForResource"
            ],
            "List": [
                "ListAccelerators",
                "ListByoipCidrs",
                "ListEndpointGroups",
                "ListListeners"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Advertises an IPv4 address range that is provisioned for use with your accelerator through bring your own IP addresses (BYOIP).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_AdvertiseByoipCidr.html
     */
    toAdvertiseByoipCidr() {
        this.to('globalaccelerator:AdvertiseByoipCidr');
        return this;
    }
    /**
     * Create an accelerator.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateAccelerator.html
     */
    toCreateAccelerator() {
        this.to('globalaccelerator:CreateAccelerator');
        return this;
    }
    /**
     * Add an endpoint group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateEndpointGroup.html
     */
    toCreateEndpointGroup() {
        this.to('globalaccelerator:CreateEndpointGroup');
        return this;
    }
    /**
     * Add a listener.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateListener.html
     */
    toCreateListener() {
        this.to('globalaccelerator:CreateListener');
        return this;
    }
    /**
     * Delete the accelerator.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteAccelerator.html
     */
    toDeleteAccelerator() {
        this.to('globalaccelerator:DeleteAccelerator');
        return this;
    }
    /**
     * Delete the endpoint group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteEndpointGroup.html
     */
    toDeleteEndpointGroup() {
        this.to('globalaccelerator:DeleteEndpointGroup');
        return this;
    }
    /**
     * Delete the listener.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteListener.html
     */
    toDeleteListener() {
        this.to('globalaccelerator:DeleteListener');
        return this;
    }
    /**
     * Releases the specified address range that you provisioned for use with your accelerator through bring your own IP addresses (BYOIP) and deletes the corresponding address pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeprovisionByoipCidr.html
     */
    toDeprovisionByoipCidr() {
        this.to('globalaccelerator:DeprovisionByoipCidr');
        return this;
    }
    /**
     * Describe the accelerator.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeAccelerator.html
     */
    toDescribeAccelerator() {
        this.to('globalaccelerator:DescribeAccelerator');
        return this;
    }
    /**
     * Describe the accelerator Attributes.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeAcceleratorAttributes.html
     */
    toDescribeAcceleratorAttributes() {
        this.to('globalaccelerator:DescribeAcceleratorAttributes');
        return this;
    }
    /**
     * Describe the endpoint group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeEndpointGroup.html
     */
    toDescribeEndpointGroup() {
        this.to('globalaccelerator:DescribeEndpointGroup');
        return this;
    }
    /**
     * Describe the listener.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeListener.html
     */
    toDescribeListener() {
        this.to('globalaccelerator:DescribeListener');
        return this;
    }
    /**
     * List the accelerators.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListAccelerators.html
     */
    toListAccelerators() {
        this.to('globalaccelerator:ListAccelerators');
        return this;
    }
    /**
     * List the byoip cidrs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListByoipCidrs.html
     */
    toListByoipCidrs() {
        this.to('globalaccelerator:ListByoipCidrs');
        return this;
    }
    /**
     * List the endpoint groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListEndpointGroups.html
     */
    toListEndpointGroups() {
        this.to('globalaccelerator:ListEndpointGroups');
        return this;
    }
    /**
     * List the listeners.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListListeners.html
     */
    toListListeners() {
        this.to('globalaccelerator:ListListeners');
        return this;
    }
    /**
     * List tags for a globalaccelerator resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('globalaccelerator:ListTagsForResource');
        return this;
    }
    /**
     * Provisions an address range for use with your accelerator through bring your own IP addresses (BYOIP) and creates a corresponding address pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ProvisionByoipCidr.html
     */
    toProvisionByoipCidr() {
        this.to('globalaccelerator:ProvisionByoipCidr');
        return this;
    }
    /**
     * Add tags to globalaccelerator resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_TagResource.html
     */
    toTagResource() {
        this.to('globalaccelerator:TagResource');
        return this;
    }
    /**
     * Remove tags from globalaccelerator resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        this.to('globalaccelerator:UntagResource');
        return this;
    }
    /**
     * Update the accelerator.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateAccelerator.html
     */
    toUpdateAccelerator() {
        this.to('globalaccelerator:UpdateAccelerator');
        return this;
    }
    /**
     * Update the accelerator attributes.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateAcceleratorAttributes.html
     */
    toUpdateAcceleratorAttributes() {
        this.to('globalaccelerator:UpdateAcceleratorAttributes');
        return this;
    }
    /**
     * Update the endpoint group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateEndpointGroup.html
     */
    toUpdateEndpointGroup() {
        this.to('globalaccelerator:UpdateEndpointGroup');
        return this;
    }
    /**
     * Update the listener.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateListener.html
     */
    toUpdateListener() {
        this.to('globalaccelerator:UpdateListener');
        return this;
    }
    /**
     * Stops advertising an IPv4 address range that is provisioned as an address pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_WithdrawByoipCidr.html
     */
    toWithdrawByoipCidr() {
        this.to('globalaccelerator:WithdrawByoipCidr');
        return this;
    }
    /**
     * Adds a resource of type accelerator to the statement
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_Accelerator.html
     *
     * @param acceleratorId - Identifier for the acceleratorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAccelerator(acceleratorId, account, partition) {
        var arn = 'arn:${Partition}:globalaccelerator::${Account}:accelerator/${AcceleratorId}';
        arn = arn.replace('${AcceleratorId}', acceleratorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type listener to the statement
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_Listener.html
     *
     * @param acceleratorId - Identifier for the acceleratorId.
     * @param listenerId - Identifier for the listenerId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onListener(acceleratorId, listenerId, account, partition) {
        var arn = 'arn:${Partition}:globalaccelerator::${Account}:accelerator/${AcceleratorId}/listener/${ListenerId}';
        arn = arn.replace('${AcceleratorId}', acceleratorId);
        arn = arn.replace('${ListenerId}', listenerId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type endpointgroup to the statement
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_EndpointGroup.html
     *
     * @param acceleratorId - Identifier for the acceleratorId.
     * @param listenerId - Identifier for the listenerId.
     * @param endpointGroupId - Identifier for the endpointGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEndpointgroup(acceleratorId, listenerId, endpointGroupId, account, partition) {
        var arn = 'arn:${Partition}:globalaccelerator::${Account}:accelerator/${AcceleratorId}/listener/${ListenerId}/endpoint-group/${EndpointGroupId}';
        arn = arn.replace('${AcceleratorId}', acceleratorId);
        arn = arn.replace('${ListenerId}', listenerId);
        arn = arn.replace('${EndpointGroupId}', endpointGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Globalaccelerator = Globalaccelerator;
//# sourceMappingURL=data:application/json;base64,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