"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Honeycode = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [honeycode](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonhoneycode.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Honeycode extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [honeycode](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonhoneycode.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'honeycode';
        this.accessLevelList = {
            "Write": [
                "ApproveTeamAssociation",
                "CreateTenant",
                "InvokeScreenAutomation",
                "RejectTeamAssociation"
            ],
            "Read": [
                "GetScreenData"
            ],
            "List": [
                "ListTeamAssociations",
                "ListTenants"
            ]
        };
    }
    /**
     * Grants permission to approve a team association request for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#approve-team-association
     */
    toApproveTeamAssociation() {
        this.to('honeycode:ApproveTeamAssociation');
        return this;
    }
    /**
     * Grants permission to create a new tenant within Amazon Honeycode for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/tenant.html#create-tenant
     */
    toCreateTenant() {
        this.to('honeycode:CreateTenant');
        return this;
    }
    /**
     * Grants permission to load the data from a screen
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_GetScreenData.html
     */
    toGetScreenData() {
        this.to('honeycode:GetScreenData');
        return this;
    }
    /**
     * Grants permission to invoke a screen automation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_InvokeScreenAutomation.html
     */
    toInvokeScreenAutomation() {
        this.to('honeycode:InvokeScreenAutomation');
        return this;
    }
    /**
     * Grants permission to list all pending and approved team associations with your AWS Account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#list-team-associations
     */
    toListTeamAssociations() {
        this.to('honeycode:ListTeamAssociations');
        return this;
    }
    /**
     * Grants permission to list all tenants of Amazon Honeycode for your AWS Account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/tenant.html#list-tenants
     */
    toListTenants() {
        this.to('honeycode:ListTenants');
        return this;
    }
    /**
     * Grants permission to reject a team association request for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#reject-team-association
     */
    toRejectTeamAssociation() {
        this.to('honeycode:RejectTeamAssociation');
        return this;
    }
    /**
     * Adds a resource of type screen to the statement
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-screen.html
     *
     * @param workbookId - Identifier for the workbookId.
     * @param appId - Identifier for the appId.
     * @param screenId - Identifier for the screenId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onScreen(workbookId, appId, screenId, account, region, partition) {
        var arn = 'arn:${Partition}:honeycode:${Region}:${Account}:screen:workbook/${WorkbookId}/app/${AppId}/screen/${ScreenId}';
        arn = arn.replace('${WorkbookId}', workbookId);
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${ScreenId}', screenId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type screen-automation to the statement
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-screen-automation.html
     *
     * @param workbookId - Identifier for the workbookId.
     * @param appId - Identifier for the appId.
     * @param screenId - Identifier for the screenId.
     * @param automationId - Identifier for the automationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onScreenAutomation(workbookId, appId, screenId, automationId, account, region, partition) {
        var arn = 'arn:${Partition}:honeycode:${Region}:${Account}:screen-automation:workbook/${WorkbookId}/app/${AppId}/screen/${ScreenId}/automation/${AutomationId}';
        arn = arn.replace('${WorkbookId}', workbookId);
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${ScreenId}', screenId);
        arn = arn.replace('${AutomationId}', automationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Honeycode = Honeycode;
//# sourceMappingURL=data:application/json;base64,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