"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Inspector = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [inspector](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoninspector.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Inspector extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [inspector](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoninspector.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'inspector';
        this.accessLevelList = {
            "Write": [
                "AddAttributesToFindings",
                "CreateAssessmentTarget",
                "CreateAssessmentTemplate",
                "CreateResourceGroup",
                "DeleteAssessmentRun",
                "DeleteAssessmentTarget",
                "DeleteAssessmentTemplate",
                "RegisterCrossAccountAccessRole",
                "RemoveAttributesFromFindings",
                "StartAssessmentRun",
                "StopAssessmentRun",
                "SubscribeToEvent",
                "UnsubscribeFromEvent",
                "UpdateAssessmentTarget"
            ],
            "Read": [
                "DescribeAssessmentRuns",
                "DescribeAssessmentTargets",
                "DescribeAssessmentTemplates",
                "DescribeCrossAccountAccessRole",
                "DescribeFindings",
                "DescribeResourceGroups",
                "DescribeRulesPackages",
                "GetTelemetryMetadata",
                "PreviewAgents"
            ],
            "List": [
                "ListAssessmentRunAgents",
                "ListAssessmentRuns",
                "ListAssessmentTargets",
                "ListAssessmentTemplates",
                "ListEventSubscriptions",
                "ListFindings",
                "ListRulesPackages",
                "ListTagsForResource"
            ],
            "Tagging": [
                "SetTagsForResource"
            ]
        };
    }
    /**
     * Assigns attributes (key and value pairs) to the findings that are specified by the ARNs of the findings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_AddAttributesToFindings.html
     */
    toAddAttributesToFindings() {
        this.to('inspector:AddAttributesToFindings');
        return this;
    }
    /**
     * Creates a new assessment target using the ARN of the resource group that is generated by CreateResourceGroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_CreateAssessmentTarget.html
     */
    toCreateAssessmentTarget() {
        this.to('inspector:CreateAssessmentTarget');
        return this;
    }
    /**
     * Creates an assessment template for the assessment target that is specified by the ARN of the assessment target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_CreateAssessmentTemplate.html
     */
    toCreateAssessmentTemplate() {
        this.to('inspector:CreateAssessmentTemplate');
        return this;
    }
    /**
     * Creates a resource group using the specified set of tags (key and value pairs) that are used to select the EC2 instances to be included in an Amazon Inspector assessment target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_CreateResourceGroup.html
     */
    toCreateResourceGroup() {
        this.to('inspector:CreateResourceGroup');
        return this;
    }
    /**
     * Deletes the assessment run that is specified by the ARN of the assessment run.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_DeleteAssessmentRun.html
     */
    toDeleteAssessmentRun() {
        this.to('inspector:DeleteAssessmentRun');
        return this;
    }
    /**
     * Deletes the assessment target that is specified by the ARN of the assessment target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_DeleteAssessmentTarget.html
     */
    toDeleteAssessmentTarget() {
        this.to('inspector:DeleteAssessmentTarget');
        return this;
    }
    /**
     * Deletes the assessment template that is specified by the ARN of the assessment template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_DeleteAssessmentTemplate.html
     */
    toDeleteAssessmentTemplate() {
        this.to('inspector:DeleteAssessmentTemplate');
        return this;
    }
    /**
     * Describes the assessment runs that are specified by the ARNs of the assessment runs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_DescribeAssessmentRuns.html
     */
    toDescribeAssessmentRuns() {
        this.to('inspector:DescribeAssessmentRuns');
        return this;
    }
    /**
     * Describes the assessment targets that are specified by the ARNs of the assessment targets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_DescribeAssessmentTargets.html
     */
    toDescribeAssessmentTargets() {
        this.to('inspector:DescribeAssessmentTargets');
        return this;
    }
    /**
     * Describes the assessment templates that are specified by the ARNs of the assessment templates.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_DescribeAssessmentTemplates.html
     */
    toDescribeAssessmentTemplates() {
        this.to('inspector:DescribeAssessmentTemplates');
        return this;
    }
    /**
     * Describes the IAM role that enables Amazon Inspector to access your AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_DescribeCrossAccountAccessRole.html
     */
    toDescribeCrossAccountAccessRole() {
        this.to('inspector:DescribeCrossAccountAccessRole');
        return this;
    }
    /**
     * Describes the findings that are specified by the ARNs of the findings.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_DescribeFindings.html
     */
    toDescribeFindings() {
        this.to('inspector:DescribeFindings');
        return this;
    }
    /**
     * Describes the resource groups that are specified by the ARNs of the resource groups.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_DescribeResourceGroups.html
     */
    toDescribeResourceGroups() {
        this.to('inspector:DescribeResourceGroups');
        return this;
    }
    /**
     * Describes the rules packages that are specified by the ARNs of the rules packages.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_DescribeRulesPackages.html
     */
    toDescribeRulesPackages() {
        this.to('inspector:DescribeRulesPackages');
        return this;
    }
    /**
     * Information about the data that is collected for the specified assessment run.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_GetTelemetryMetadata.html
     */
    toGetTelemetryMetadata() {
        this.to('inspector:GetTelemetryMetadata');
        return this;
    }
    /**
     * Lists the agents of the assessment runs that are specified by the ARNs of the assessment runs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_ListAssessmentRunAgents.html
     */
    toListAssessmentRunAgents() {
        this.to('inspector:ListAssessmentRunAgents');
        return this;
    }
    /**
     * Lists the assessment runs that correspond to the assessment templates that are specified by the ARNs of the assessment templates.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_ListAssessmentRuns.html
     */
    toListAssessmentRuns() {
        this.to('inspector:ListAssessmentRuns');
        return this;
    }
    /**
     * Lists the ARNs of the assessment targets within this AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_ListAssessmentTargets.html
     */
    toListAssessmentTargets() {
        this.to('inspector:ListAssessmentTargets');
        return this;
    }
    /**
     * Lists the assessment templates that correspond to the assessment targets that are specified by the ARNs of the assessment targets.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_ListAssessmentTemplates.html
     */
    toListAssessmentTemplates() {
        this.to('inspector:ListAssessmentTemplates');
        return this;
    }
    /**
     * Lists all the event subscriptions for the assessment template that is specified by the ARN of the assessment template.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_ListEventSubscriptions.html
     */
    toListEventSubscriptions() {
        this.to('inspector:ListEventSubscriptions');
        return this;
    }
    /**
     * Lists findings that are generated by the assessment runs that are specified by the ARNs of the assessment runs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_ListFindings.html
     */
    toListFindings() {
        this.to('inspector:ListFindings');
        return this;
    }
    /**
     * Lists all available Amazon Inspector rules packages.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_ListRulesPackages.html
     */
    toListRulesPackages() {
        this.to('inspector:ListRulesPackages');
        return this;
    }
    /**
     * Lists all tags associated with an assessment template.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('inspector:ListTagsForResource');
        return this;
    }
    /**
     * Previews the agents installed on the EC2 instances that are part of the specified assessment target.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_PreviewAgents.html
     */
    toPreviewAgents() {
        this.to('inspector:PreviewAgents');
        return this;
    }
    /**
     * Registers the IAM role that Amazon Inspector uses to list your EC2 instances at the start of the assessment run or when you call the PreviewAgents action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_RegisterCrossAccountAccessRole.html
     */
    toRegisterCrossAccountAccessRole() {
        this.to('inspector:RegisterCrossAccountAccessRole');
        return this;
    }
    /**
     * Removes entire attributes (key and value pairs) from the findings that are specified by the ARNs of the findings where an attribute with the specified key exists.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_RemoveAttributesFromFindings.html
     */
    toRemoveAttributesFromFindings() {
        this.to('inspector:RemoveAttributesFromFindings');
        return this;
    }
    /**
     * Sets tags (key and value pairs) to the assessment template that is specified by the ARN of the assessment template.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_SetTagsForResource.html
     */
    toSetTagsForResource() {
        this.to('inspector:SetTagsForResource');
        return this;
    }
    /**
     * Starts the assessment run specified by the ARN of the assessment template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_StartAssessmentRun.html
     */
    toStartAssessmentRun() {
        this.to('inspector:StartAssessmentRun');
        return this;
    }
    /**
     * Stops the assessment run that is specified by the ARN of the assessment run.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_StopAssessmentRun.html
     */
    toStopAssessmentRun() {
        this.to('inspector:StopAssessmentRun');
        return this;
    }
    /**
     * Enables the process of sending Amazon Simple Notification Service (SNS) notifications about a specified event to a specified SNS topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_SubscribeToEvent.html
     */
    toSubscribeToEvent() {
        this.to('inspector:SubscribeToEvent');
        return this;
    }
    /**
     * Disables the process of sending Amazon Simple Notification Service (SNS) notifications about a specified event to a specified SNS topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_UnsubscribeFromEvent.html
     */
    toUnsubscribeFromEvent() {
        this.to('inspector:UnsubscribeFromEvent');
        return this;
    }
    /**
     * Updates the assessment target that is specified by the ARN of the assessment target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/latest/APIReference/API_UpdateAssessmentTarget.html
     */
    toUpdateAssessmentTarget() {
        this.to('inspector:UpdateAssessmentTarget');
        return this;
    }
}
exports.Inspector = Inspector;
//# sourceMappingURL=data:application/json;base64,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