"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Greengrass = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [greengrass](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotgreengrass.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Greengrass extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [greengrass](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotgreengrass.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'greengrass';
        this.accessLevelList = {
            "Write": [
                "AssociateRoleToGroup",
                "CreateConnectorDefinition",
                "CreateConnectorDefinitionVersion",
                "CreateCoreDefinition",
                "CreateCoreDefinitionVersion",
                "CreateDeployment",
                "CreateDeviceDefinition",
                "CreateDeviceDefinitionVersion",
                "CreateFunctionDefinition",
                "CreateFunctionDefinitionVersion",
                "CreateGroup",
                "CreateGroupCertificateAuthority",
                "CreateGroupVersion",
                "CreateLoggerDefinition",
                "CreateLoggerDefinitionVersion",
                "CreateResourceDefinition",
                "CreateResourceDefinitionVersion",
                "CreateSoftwareUpdateJob",
                "CreateSubscriptionDefinition",
                "CreateSubscriptionDefinitionVersion",
                "DeleteConnectorDefinition",
                "DeleteCoreDefinition",
                "DeleteDeviceDefinition",
                "DeleteFunctionDefinition",
                "DeleteGroup",
                "DeleteLoggerDefinition",
                "DeleteResourceDefinition",
                "DeleteSubscriptionDefinition",
                "DisassociateRoleFromGroup",
                "DisassociateServiceRoleFromAccount",
                "ResetDeployments",
                "StartBulkDeployment",
                "StopBulkDeployment",
                "UpdateConnectivityInfo",
                "UpdateConnectorDefinition",
                "UpdateCoreDefinition",
                "UpdateDeviceDefinition",
                "UpdateFunctionDefinition",
                "UpdateGroup",
                "UpdateGroupCertificateConfiguration",
                "UpdateLoggerDefinition",
                "UpdateResourceDefinition",
                "UpdateSubscriptionDefinition"
            ],
            "Permissions management": [
                "AssociateServiceRoleToAccount"
            ],
            "Read": [
                "Discover",
                "GetAssociatedRole",
                "GetBulkDeploymentStatus",
                "GetConnectivityInfo",
                "GetConnectorDefinition",
                "GetConnectorDefinitionVersion",
                "GetCoreDefinition",
                "GetCoreDefinitionVersion",
                "GetDeploymentStatus",
                "GetDeviceDefinition",
                "GetDeviceDefinitionVersion",
                "GetFunctionDefinition",
                "GetFunctionDefinitionVersion",
                "GetGroup",
                "GetGroupCertificateAuthority",
                "GetGroupCertificateConfiguration",
                "GetGroupVersion",
                "GetLoggerDefinition",
                "GetLoggerDefinitionVersion",
                "GetResourceDefinition",
                "GetResourceDefinitionVersion",
                "GetServiceRoleForAccount",
                "GetSubscriptionDefinition",
                "GetSubscriptionDefinitionVersion"
            ],
            "List": [
                "ListBulkDeploymentDetailedReports",
                "ListBulkDeployments",
                "ListConnectorDefinitionVersions",
                "ListConnectorDefinitions",
                "ListCoreDefinitionVersions",
                "ListCoreDefinitions",
                "ListDeployments",
                "ListDeviceDefinitionVersions",
                "ListDeviceDefinitions",
                "ListFunctionDefinitionVersions",
                "ListFunctionDefinitions",
                "ListGroupCertificateAuthorities",
                "ListGroupVersions",
                "ListGroups",
                "ListLoggerDefinitionVersions",
                "ListLoggerDefinitions",
                "ListResourceDefinitionVersions",
                "ListResourceDefinitions",
                "ListSubscriptionDefinitionVersions",
                "ListSubscriptionDefinitions",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to associate a role with a group. The role's permissions must allow Greengrass core Lambda functions and connectors to perform actions in other AWS services.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/associateroletogroup-put.html
     */
    toAssociateRoleToGroup() {
        this.to('greengrass:AssociateRoleToGroup');
        return this;
    }
    /**
     * Grants permission to associate a role with your account. AWS IoT Greengrass uses this role to access your Lambda functions and AWS IoT resources.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/associateserviceroletoaccount-put.html
     */
    toAssociateServiceRoleToAccount() {
        this.to('greengrass:AssociateServiceRoleToAccount');
        return this;
    }
    /**
     * Grants permission to create a connector definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createconnectordefinition-post.html
     */
    toCreateConnectorDefinition() {
        this.to('greengrass:CreateConnectorDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing connector definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createconnectordefinitionversion-post.html
     */
    toCreateConnectorDefinitionVersion() {
        this.to('greengrass:CreateConnectorDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create a core definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createcoredefinition-post.html
     */
    toCreateCoreDefinition() {
        this.to('greengrass:CreateCoreDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing core definition. Greengrass groups must each contain exactly one Greengrass core.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createcoredefinitionversion-post.html
     */
    toCreateCoreDefinitionVersion() {
        this.to('greengrass:CreateCoreDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create a deployment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createdeployment-post.html
     */
    toCreateDeployment() {
        this.to('greengrass:CreateDeployment');
        return this;
    }
    /**
     * Grants permission to create a device definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createdevicedefinition-post.html
     */
    toCreateDeviceDefinition() {
        this.to('greengrass:CreateDeviceDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing device definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createdevicedefinitionversion-post.html
     */
    toCreateDeviceDefinitionVersion() {
        this.to('greengrass:CreateDeviceDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create a Lambda function definition to be used in a group that contains a list of Lambda functions and their configurations.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createfunctiondefinition-post.html
     */
    toCreateFunctionDefinition() {
        this.to('greengrass:CreateFunctionDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing Lambda function definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createfunctiondefinitionversion-post.html
     */
    toCreateFunctionDefinitionVersion() {
        this.to('greengrass:CreateFunctionDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create a group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/creategroup-post.html
     */
    toCreateGroup() {
        this.to('greengrass:CreateGroup');
        return this;
    }
    /**
     * Grants permission to create a CA for the group, or rotate the existing CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/creategroupcertificateauthority-post.html
     */
    toCreateGroupCertificateAuthority() {
        this.to('greengrass:CreateGroupCertificateAuthority');
        return this;
    }
    /**
     * Grants permission to create a version of a group that has already been defined.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/creategroupversion-post.html
     */
    toCreateGroupVersion() {
        this.to('greengrass:CreateGroupVersion');
        return this;
    }
    /**
     * Grants permission to create a logger definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createloggerdefinition-post.html
     */
    toCreateLoggerDefinition() {
        this.to('greengrass:CreateLoggerDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing logger definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createloggerdefinitionversion-post.html
     */
    toCreateLoggerDefinitionVersion() {
        this.to('greengrass:CreateLoggerDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create a resource definition that contains a list of resources to be used in a group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createresourcedefinition-post.html
     */
    toCreateResourceDefinition() {
        this.to('greengrass:CreateResourceDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing resource definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createresourcedefinitionversion-post.html
     */
    toCreateResourceDefinitionVersion() {
        this.to('greengrass:CreateResourceDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create an AWS IoT job that will trigger your Greengrass cores to update the software they are running.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createsoftwareupdatejob-post.html
     */
    toCreateSoftwareUpdateJob() {
        this.to('greengrass:CreateSoftwareUpdateJob');
        return this;
    }
    /**
     * Grants permission to create a subscription definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createsubscriptiondefinition-post.html
     */
    toCreateSubscriptionDefinition() {
        this.to('greengrass:CreateSubscriptionDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing subscription definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createsubscriptiondefinitionversion-post.html
     */
    toCreateSubscriptionDefinitionVersion() {
        this.to('greengrass:CreateSubscriptionDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to delete a connector definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deleteconnectordefinition-delete.html
     */
    toDeleteConnectorDefinition() {
        this.to('greengrass:DeleteConnectorDefinition');
        return this;
    }
    /**
     * Grants permission to delete a core definition. Deleting a definition that is currently in use in a deployment affects future deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deletecoredefinition-delete.html
     */
    toDeleteCoreDefinition() {
        this.to('greengrass:DeleteCoreDefinition');
        return this;
    }
    /**
     * Grants permission to delete a device definition. Deleting a definition that is currently in use in a deployment affects future deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deletedevicedefinition-delete.html
     */
    toDeleteDeviceDefinition() {
        this.to('greengrass:DeleteDeviceDefinition');
        return this;
    }
    /**
     * Grants permission to delete a Lambda function definition. Deleting a definition that is currently in use in a deployment affects future deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deletefunctiondefinition-delete.html
     */
    toDeleteFunctionDefinition() {
        this.to('greengrass:DeleteFunctionDefinition');
        return this;
    }
    /**
     * Grants permission to delete a group that is not currently in use in a deployment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deletegroup-delete.html
     */
    toDeleteGroup() {
        this.to('greengrass:DeleteGroup');
        return this;
    }
    /**
     * Grants permission to delete a logger definition. Deleting a definition that is currently in use in a deployment affects future deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deleteloggerdefinition-delete.html
     */
    toDeleteLoggerDefinition() {
        this.to('greengrass:DeleteLoggerDefinition');
        return this;
    }
    /**
     * Grants permission to delete a resource definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deleteresourcedefinition-delete.html
     */
    toDeleteResourceDefinition() {
        this.to('greengrass:DeleteResourceDefinition');
        return this;
    }
    /**
     * Grants permission to delete a subscription definition. Deleting a definition that is currently in use in a deployment affects future deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deletesubscriptiondefinition-delete.html
     */
    toDeleteSubscriptionDefinition() {
        this.to('greengrass:DeleteSubscriptionDefinition');
        return this;
    }
    /**
     * Grants permission to disassociate the role from a group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/disassociaterolefromgroup-delete.html
     */
    toDisassociateRoleFromGroup() {
        this.to('greengrass:DisassociateRoleFromGroup');
        return this;
    }
    /**
     * Grants permission to disassociate the service role from an account. Without a service role, deployments will not work.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/disassociateservicerolefromaccount-delete.html
     */
    toDisassociateServiceRoleFromAccount() {
        this.to('greengrass:DisassociateServiceRoleFromAccount');
        return this;
    }
    /**
     * Grants permission to retrieve information required to connect to a Greengrass core.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/developerguide/gg-discover-api.html
     */
    toDiscover() {
        this.to('greengrass:Discover');
        return this;
    }
    /**
     * Grants permission to retrieve the role associated with a group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getassociatedrole-get.html
     */
    toGetAssociatedRole() {
        this.to('greengrass:GetAssociatedRole');
        return this;
    }
    /**
     * Grants permission to return the status of a bulk deployment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getbulkdeploymentstatus-get.html
     */
    toGetBulkDeploymentStatus() {
        this.to('greengrass:GetBulkDeploymentStatus');
        return this;
    }
    /**
     * Grants permission to retrieve the connectivity information for a core.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getconnectivityinfo-get.html
     */
    toGetConnectivityInfo() {
        this.to('greengrass:GetConnectivityInfo');
        return this;
    }
    /**
     * Grants permission to retrieve information about a connector definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getconnectordefinition-get.html
     */
    toGetConnectorDefinition() {
        this.to('greengrass:GetConnectorDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a connector definition version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getconnectordefinitionversion-get.html
     */
    toGetConnectorDefinitionVersion() {
        this.to('greengrass:GetConnectorDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about a core definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getcoredefinition-get.html
     */
    toGetCoreDefinition() {
        this.to('greengrass:GetCoreDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a core definition version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getcoredefinitionversion-get.html
     */
    toGetCoreDefinitionVersion() {
        this.to('greengrass:GetCoreDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to return the status of a deployment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getdeploymentstatus-get.html
     */
    toGetDeploymentStatus() {
        this.to('greengrass:GetDeploymentStatus');
        return this;
    }
    /**
     * Grants permission to retrieve information about a device definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getdevicedefinition-get.html
     */
    toGetDeviceDefinition() {
        this.to('greengrass:GetDeviceDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a device definition version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getdevicedefinitionversion-get.html
     */
    toGetDeviceDefinitionVersion() {
        this.to('greengrass:GetDeviceDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about a Lambda function definition, such as its creation time and latest version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getfunctiondefinition-get.html
     */
    toGetFunctionDefinition() {
        this.to('greengrass:GetFunctionDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a Lambda function definition version, such as which Lambda functions are included in the version and their configurations.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getfunctiondefinitionversion-get.html
     */
    toGetFunctionDefinitionVersion() {
        this.to('greengrass:GetFunctionDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about a group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getgroup-get.html
     */
    toGetGroup() {
        this.to('greengrass:GetGroup');
        return this;
    }
    /**
     * Grants permission to return the public key of the CA associated with a group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getgroupcertificateauthority-get.html
     */
    toGetGroupCertificateAuthority() {
        this.to('greengrass:GetGroupCertificateAuthority');
        return this;
    }
    /**
     * Grants permission to retrieve the current configuration for the CA used by a group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getgroupcertificateconfiguration-get.html
     */
    toGetGroupCertificateConfiguration() {
        this.to('greengrass:GetGroupCertificateConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve information about a group version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getgroupversion-get.html
     */
    toGetGroupVersion() {
        this.to('greengrass:GetGroupVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about a logger definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getloggerdefinition-get.html
     */
    toGetLoggerDefinition() {
        this.to('greengrass:GetLoggerDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a logger definition version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getloggerdefinitionversion-get.html
     */
    toGetLoggerDefinitionVersion() {
        this.to('greengrass:GetLoggerDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about a resource definition, such as its creation time and latest version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getresourcedefinition-get.html
     */
    toGetResourceDefinition() {
        this.to('greengrass:GetResourceDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a resource definition version, such as which resources are included in the version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getresourcedefinitionversion-get.html
     */
    toGetResourceDefinitionVersion() {
        this.to('greengrass:GetResourceDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve the service role that is attached to an account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getserviceroleforaccount-get.html
     */
    toGetServiceRoleForAccount() {
        this.to('greengrass:GetServiceRoleForAccount');
        return this;
    }
    /**
     * Grants permission to retrieve information about a subscription definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getsubscriptiondefinition-get.html
     */
    toGetSubscriptionDefinition() {
        this.to('greengrass:GetSubscriptionDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a subscription definition version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getsubscriptiondefinitionversion-get.html
     */
    toGetSubscriptionDefinitionVersion() {
        this.to('greengrass:GetSubscriptionDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve a paginated list of the deployments that have been started in a bulk deployment operation and their current deployment status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listbulkdeploymentdetailedreports-get.html
     */
    toListBulkDeploymentDetailedReports() {
        this.to('greengrass:ListBulkDeploymentDetailedReports');
        return this;
    }
    /**
     * Grants permission to retrieve a list of bulk deployments.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listbulkdeployments-get.html
     */
    toListBulkDeployments() {
        this.to('greengrass:ListBulkDeployments');
        return this;
    }
    /**
     * Grants permission to list the versions of a connector definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listconnectordefinitionversions-get.html
     */
    toListConnectorDefinitionVersions() {
        this.to('greengrass:ListConnectorDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of connector definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listconnectordefinitions-get.html
     */
    toListConnectorDefinitions() {
        this.to('greengrass:ListConnectorDefinitions');
        return this;
    }
    /**
     * Grants permission to list the versions of a core definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listcoredefinitionversions-get.html
     */
    toListCoreDefinitionVersions() {
        this.to('greengrass:ListCoreDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of core definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listcoredefinitions-get.html
     */
    toListCoreDefinitions() {
        this.to('greengrass:ListCoreDefinitions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of all deployments for a group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listdeployments-get.html
     */
    toListDeployments() {
        this.to('greengrass:ListDeployments');
        return this;
    }
    /**
     * Grants permission to list the versions of a device definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listdevicedefinitionversions-get.html
     */
    toListDeviceDefinitionVersions() {
        this.to('greengrass:ListDeviceDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of device definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listdevicedefinitions-get.html
     */
    toListDeviceDefinitions() {
        this.to('greengrass:ListDeviceDefinitions');
        return this;
    }
    /**
     * Grants permission to list the versions of a Lambda function definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listfunctiondefinitionversions-get.html
     */
    toListFunctionDefinitionVersions() {
        this.to('greengrass:ListFunctionDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of Lambda function definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listfunctiondefinitions-get.html
     */
    toListFunctionDefinitions() {
        this.to('greengrass:ListFunctionDefinitions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of current CAs for a group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listgroupcertificateauthorities-get.html
     */
    toListGroupCertificateAuthorities() {
        this.to('greengrass:ListGroupCertificateAuthorities');
        return this;
    }
    /**
     * Grants permission to list the versions of a group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listgroupversions-get.html
     */
    toListGroupVersions() {
        this.to('greengrass:ListGroupVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listgroups-get.html
     */
    toListGroups() {
        this.to('greengrass:ListGroups');
        return this;
    }
    /**
     * Grants permission to list the versions of a logger definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listloggerdefinitionversions-get.html
     */
    toListLoggerDefinitionVersions() {
        this.to('greengrass:ListLoggerDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of logger definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listloggerdefinitions-get.html
     */
    toListLoggerDefinitions() {
        this.to('greengrass:ListLoggerDefinitions');
        return this;
    }
    /**
     * Grants permission to list the versions of a resource definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listresourcedefinitionversions-get.html
     */
    toListResourceDefinitionVersions() {
        this.to('greengrass:ListResourceDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of resource definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listresourcedefinitions-get.html
     */
    toListResourceDefinitions() {
        this.to('greengrass:ListResourceDefinitions');
        return this;
    }
    /**
     * Grants permission to list the versions of a subscription definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listsubscriptiondefinitionversions-get.html
     */
    toListSubscriptionDefinitionVersions() {
        this.to('greengrass:ListSubscriptionDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of subscription definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listsubscriptiondefinitions-get.html
     */
    toListSubscriptionDefinitions() {
        this.to('greengrass:ListSubscriptionDefinitions');
        return this;
    }
    /**
     * Grants permission to list the tags for a resource.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listtagsforresource-get.html
     */
    toListTagsForResource() {
        this.to('greengrass:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to reset a group's deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/resetdeployments-post.html
     */
    toResetDeployments() {
        this.to('greengrass:ResetDeployments');
        return this;
    }
    /**
     * Grants permission to deploy multiple groups in one operation.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/startbulkdeployment-post.html
     */
    toStartBulkDeployment() {
        this.to('greengrass:StartBulkDeployment');
        return this;
    }
    /**
     * Grants permission to stop the execution of a bulk deployment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/stopbulkdeployment-put.html
     */
    toStopBulkDeployment() {
        this.to('greengrass:StopBulkDeployment');
        return this;
    }
    /**
     * Grants permission to add tags to a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/tagresource-post.html
     */
    toTagResource() {
        this.to('greengrass:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/untagresource-delete.html
     */
    toUntagResource() {
        this.to('greengrass:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the connectivity information for a Greengrass core. Any devices that belong to the group that has this core will receive this information in order to find the location of the core and connect to it.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updateconnectivityinfo-put.html
     */
    toUpdateConnectivityInfo() {
        this.to('greengrass:UpdateConnectivityInfo');
        return this;
    }
    /**
     * Grants permission to update a connector definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updateconnectordefinition-put.html
     */
    toUpdateConnectorDefinition() {
        this.to('greengrass:UpdateConnectorDefinition');
        return this;
    }
    /**
     * Grants permission to update a core definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updatecoredefinition-put.html
     */
    toUpdateCoreDefinition() {
        this.to('greengrass:UpdateCoreDefinition');
        return this;
    }
    /**
     * Grants permission to update a device definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updatedevicedefinition-put.html
     */
    toUpdateDeviceDefinition() {
        this.to('greengrass:UpdateDeviceDefinition');
        return this;
    }
    /**
     * Grants permission to update a Lambda function definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updatefunctiondefinition-put.html
     */
    toUpdateFunctionDefinition() {
        this.to('greengrass:UpdateFunctionDefinition');
        return this;
    }
    /**
     * Grants permission to update a group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updategroup-put.html
     */
    toUpdateGroup() {
        this.to('greengrass:UpdateGroup');
        return this;
    }
    /**
     * Grants permission to update the certificate expiry time for a group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updategroupcertificateconfiguration-put.html
     */
    toUpdateGroupCertificateConfiguration() {
        this.to('greengrass:UpdateGroupCertificateConfiguration');
        return this;
    }
    /**
     * Grants permission to update a logger definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updateloggerdefinition-put.html
     */
    toUpdateLoggerDefinition() {
        this.to('greengrass:UpdateLoggerDefinition');
        return this;
    }
    /**
     * Grants permission to update a resource definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updateresourcedefinition-put.html
     */
    toUpdateResourceDefinition() {
        this.to('greengrass:UpdateResourceDefinition');
        return this;
    }
    /**
     * Grants permission to update a subscription definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updatesubscriptiondefinition-put.html
     */
    toUpdateSubscriptionDefinition() {
        this.to('greengrass:UpdateSubscriptionDefinition');
        return this;
    }
    /**
     * Adds a resource of type connectivityInfo to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-connectivityinfo.html
     *
     * @param thingName - Identifier for the thingName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConnectivityInfo(thingName, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/things/${ThingName}/connectivityInfo';
        arn = arn.replace('${ThingName}', thingName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type artifact to the statement
     *
     * @param groupId - Identifier for the groupId.
     * @param deploymentId - Identifier for the deploymentId.
     * @param artifactId - Identifier for the artifactId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onArtifact(groupId, deploymentId, artifactId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/deployments/${DeploymentId}/artifacts/lambda/${ArtifactId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${DeploymentId}', deploymentId);
        arn = arn.replace('${ArtifactId}', artifactId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type certificateAuthority to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/developerguide/gg-sec.html
     *
     * @param groupId - Identifier for the groupId.
     * @param certificateAuthorityId - Identifier for the certificateAuthorityId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCertificateAuthority(groupId, certificateAuthorityId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/certificateauthorities/${CertificateAuthorityId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${CertificateAuthorityId}', certificateAuthorityId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deployment to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-createdeploymentrequest.html
     *
     * @param groupId - Identifier for the groupId.
     * @param deploymentId - Identifier for the deploymentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDeployment(groupId, deploymentId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/deployments/${DeploymentId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${DeploymentId}', deploymentId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bulkDeployment to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/developerguide/bulk-deploy-cli.html
     *
     * @param bulkDeploymentId - Identifier for the bulkDeploymentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBulkDeployment(bulkDeploymentId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/bulk/deployments/${BulkDeploymentId}';
        arn = arn.replace('${BulkDeploymentId}', bulkDeploymentId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-groupinformation.html
     *
     * @param groupId - Identifier for the groupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGroup(groupId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type groupVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-groupversion.html
     *
     * @param groupId - Identifier for the groupId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGroupVersion(groupId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/versions/${VersionId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type coreDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-core.html
     *
     * @param coreDefinitionId - Identifier for the coreDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCoreDefinition(coreDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/cores/${CoreDefinitionId}';
        arn = arn.replace('${CoreDefinitionId}', coreDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type coreDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-coredefinitionversion.html
     *
     * @param coreDefinitionId - Identifier for the coreDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCoreDefinitionVersion(coreDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/cores/${CoreDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${CoreDefinitionId}', coreDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deviceDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-device.html
     *
     * @param deviceDefinitionId - Identifier for the deviceDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeviceDefinition(deviceDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/devices/${DeviceDefinitionId}';
        arn = arn.replace('${DeviceDefinitionId}', deviceDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deviceDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-devicedefinitionversion.html
     *
     * @param deviceDefinitionId - Identifier for the deviceDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDeviceDefinitionVersion(deviceDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/devices/${DeviceDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${DeviceDefinitionId}', deviceDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type functionDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-function.html
     *
     * @param functionDefinitionId - Identifier for the functionDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFunctionDefinition(functionDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/functions/${FunctionDefinitionId}';
        arn = arn.replace('${FunctionDefinitionId}', functionDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type functionDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-functiondefinitionversion.html
     *
     * @param functionDefinitionId - Identifier for the functionDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFunctionDefinitionVersion(functionDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/functions/${FunctionDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${FunctionDefinitionId}', functionDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subscriptionDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-subscription.html
     *
     * @param subscriptionDefinitionId - Identifier for the subscriptionDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSubscriptionDefinition(subscriptionDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/subscriptions/${SubscriptionDefinitionId}';
        arn = arn.replace('${SubscriptionDefinitionId}', subscriptionDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subscriptionDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-subscriptiondefinitionversion.html
     *
     * @param subscriptionDefinitionId - Identifier for the subscriptionDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSubscriptionDefinitionVersion(subscriptionDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/subscriptions/${SubscriptionDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${SubscriptionDefinitionId}', subscriptionDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type loggerDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-logger.html
     *
     * @param loggerDefinitionId - Identifier for the loggerDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLoggerDefinition(loggerDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/loggers/${LoggerDefinitionId}';
        arn = arn.replace('${LoggerDefinitionId}', loggerDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type loggerDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-loggerdefinitionversion.html
     *
     * @param loggerDefinitionId - Identifier for the loggerDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLoggerDefinitionVersion(loggerDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/loggers/${LoggerDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${LoggerDefinitionId}', loggerDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resourceDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-resource.html
     *
     * @param resourceDefinitionId - Identifier for the resourceDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResourceDefinition(resourceDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/resources/${ResourceDefinitionId}';
        arn = arn.replace('${ResourceDefinitionId}', resourceDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resourceDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-resourcedefinitionversion.html
     *
     * @param resourceDefinitionId - Identifier for the resourceDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onResourceDefinitionVersion(resourceDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/resources/${ResourceDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${ResourceDefinitionId}', resourceDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type connectorDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-connector.html
     *
     * @param connectorDefinitionId - Identifier for the connectorDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnectorDefinition(connectorDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/connectors/${ConnectorDefinitionId}';
        arn = arn.replace('${ConnectorDefinitionId}', connectorDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type connectorDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-connectordefinitionversion.html
     *
     * @param connectorDefinitionId - Identifier for the connectorDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConnectorDefinitionVersion(connectorDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/connectors/${ConnectorDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${ConnectorDefinitionId}', connectorDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type thing to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/thing-registry.html
     *
     * @param thingName - Identifier for the thingName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onThing(thingName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:thing/${ThingName}';
        arn = arn.replace('${ThingName}', thingName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Greengrass = Greengrass;
//# sourceMappingURL=data:application/json;base64,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