"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kinesisanalytics = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [kinesisanalytics](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkinesisanalytics.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kinesisanalytics extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [kinesisanalytics](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkinesisanalytics.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'kinesisanalytics';
        this.accessLevelList = {
            "Write": [
                "AddApplicationInput",
                "AddApplicationOutput",
                "AddApplicationReferenceDataSource",
                "CreateApplication",
                "DeleteApplication",
                "DeleteApplicationOutput",
                "DeleteApplicationReferenceDataSource",
                "StartApplication",
                "StopApplication",
                "UpdateApplication"
            ],
            "Read": [
                "DescribeApplication",
                "DiscoverInputSchema",
                "GetApplicationState",
                "ListTagsForResource"
            ],
            "List": [
                "ListApplications"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Adds input to the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_AddApplicationInput.html
     */
    toAddApplicationInput() {
        this.to('kinesisanalytics:AddApplicationInput');
        return this;
    }
    /**
     * Adds output to the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_AddApplicationOutput.html
     */
    toAddApplicationOutput() {
        this.to('kinesisanalytics:AddApplicationOutput');
        return this;
    }
    /**
     * Adds reference data source to the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_AddApplicationReferenceDataSource.html
     */
    toAddApplicationReferenceDataSource() {
        this.to('kinesisanalytics:AddApplicationReferenceDataSource');
        return this;
    }
    /**
     * Creates an application.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_CreateApplication.html
     */
    toCreateApplication() {
        this.to('kinesisanalytics:CreateApplication');
        return this;
    }
    /**
     * Deletes the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_DeleteApplication.html
     */
    toDeleteApplication() {
        this.to('kinesisanalytics:DeleteApplication');
        return this;
    }
    /**
     * Deletes the specified output of the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_DeleteApplicationOutput.html
     */
    toDeleteApplicationOutput() {
        this.to('kinesisanalytics:DeleteApplicationOutput');
        return this;
    }
    /**
     * Deletes the specified reference data source of the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_DeleteApplicationReferenceDataSource.html
     */
    toDeleteApplicationReferenceDataSource() {
        this.to('kinesisanalytics:DeleteApplicationReferenceDataSource');
        return this;
    }
    /**
     * Describes the specified application.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_DescribeApplication.html
     */
    toDescribeApplication() {
        this.to('kinesisanalytics:DescribeApplication');
        return this;
    }
    /**
     * Discovers the input schema for the application.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_DiscoverInputSchema.html
     */
    toDiscoverInputSchema() {
        this.to('kinesisanalytics:DiscoverInputSchema');
        return this;
    }
    /**
     * Grant permission to Kinesis Data Analytics console to display stream results for Kinesis Data Analytics SQL runtime applications.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/api-permissions-reference.html#api-permissions-reference-gas
     */
    toGetApplicationState() {
        this.to('kinesisanalytics:GetApplicationState');
        return this;
    }
    /**
     * List applications for the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_ListApplications.html
     */
    toListApplications() {
        this.to('kinesisanalytics:ListApplications');
        return this;
    }
    /**
     * Fetch the tags associated with the application.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('kinesisanalytics:ListTagsForResource');
        return this;
    }
    /**
     * Starts the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_StartsApplication.html
     */
    toStartApplication() {
        this.to('kinesisanalytics:StartApplication');
        return this;
    }
    /**
     * Stops the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_StopApplication.html
     */
    toStopApplication() {
        this.to('kinesisanalytics:StopApplication');
        return this;
    }
    /**
     * Add tags to the application.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_TagResource.html
     */
    toTagResource() {
        this.to('kinesisanalytics:TagResource');
        return this;
    }
    /**
     * Remove the specified tags from the application.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_UntagResource.html
     */
    toUntagResource() {
        this.to('kinesisanalytics:UntagResource');
        return this;
    }
    /**
     * Updates the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_UpdateApplication.html
     */
    toUpdateApplication() {
        this.to('kinesisanalytics:UpdateApplication');
        return this;
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplication(applicationName, account, region, partition) {
        var arn = 'arn:${Partition}:kinesisanalytics:${Region}:${Account}:application/${ApplicationName}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Kinesisanalytics = Kinesisanalytics;
//# sourceMappingURL=data:application/json;base64,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