"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lambda = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [lambda](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awslambda.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Lambda extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [lambda](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awslambda.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'lambda';
        this.accessLevelList = {
            "Permissions management": [
                "AddLayerVersionPermission",
                "AddPermission",
                "DisableReplication",
                "EnableReplication",
                "RemoveLayerVersionPermission",
                "RemovePermission"
            ],
            "Write": [
                "CreateAlias",
                "CreateEventSourceMapping",
                "CreateFunction",
                "DeleteAlias",
                "DeleteEventSourceMapping",
                "DeleteFunction",
                "DeleteFunctionConcurrency",
                "DeleteFunctionEventInvokeConfig",
                "DeleteLayerVersion",
                "DeleteProvisionedConcurrencyConfig",
                "InvokeAsync",
                "InvokeFunction",
                "PublishLayerVersion",
                "PublishVersion",
                "PutFunctionConcurrency",
                "PutFunctionEventInvokeConfig",
                "PutProvisionedConcurrencyConfig",
                "TagResource",
                "UntagResource",
                "UpdateAlias",
                "UpdateEventSourceMapping",
                "UpdateFunctionCode",
                "UpdateFunctionConfiguration",
                "UpdateFunctionEventInvokeConfig"
            ],
            "Read": [
                "GetAccountSettings",
                "GetAlias",
                "GetEventSourceMapping",
                "GetFunction",
                "GetFunctionConcurrency",
                "GetFunctionConfiguration",
                "GetFunctionEventInvokeConfig",
                "GetLayerVersion",
                "GetLayerVersionPolicy",
                "GetPolicy",
                "GetProvisionedConcurrencyConfig",
                "ListTags"
            ],
            "List": [
                "ListAliases",
                "ListEventSourceMappings",
                "ListFunctionEventInvokeConfigs",
                "ListFunctions",
                "ListLayerVersions",
                "ListLayers",
                "ListProvisionedConcurrencyConfigs",
                "ListVersionsByFunction"
            ]
        };
    }
    /**
     * Grants permission to add permissions to the resource-based policy of a version of an AWS Lambda layer
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_AddLayerVersionPermission.html
     */
    toAddLayerVersionPermission() {
        this.to('lambda:AddLayerVersionPermission');
        return this;
    }
    /**
     * Grants permission to give an AWS service or another account permission to use an AWS Lambda function
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPrincipal()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_AddPermission.html
     */
    toAddPermission() {
        this.to('lambda:AddPermission');
        return this;
    }
    /**
     * Grants permission to create an alias for a Lambda function version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_CreateAlias.html
     */
    toCreateAlias() {
        this.to('lambda:CreateAlias');
        return this;
    }
    /**
     * Grants permission to create a mapping between an event source and an AWS Lambda function
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFunctionArn()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_CreateEventSourceMapping.html
     */
    toCreateEventSourceMapping() {
        this.to('lambda:CreateEventSourceMapping');
        return this;
    }
    /**
     * Grants permission to create an AWS Lambda function
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifLayer()
     * - .ifVpcIds()
     * - .ifSubnetIds()
     * - .ifSecurityGroupIds()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_CreateFunction.html
     */
    toCreateFunction() {
        this.to('lambda:CreateFunction');
        return this;
    }
    /**
     * Grants permission to delete an AWS Lambda function alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteAlias.html
     */
    toDeleteAlias() {
        this.to('lambda:DeleteAlias');
        return this;
    }
    /**
     * Grants permission to delete an AWS Lambda event source mapping
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFunctionArn()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteEventSourceMapping.html
     */
    toDeleteEventSourceMapping() {
        this.to('lambda:DeleteEventSourceMapping');
        return this;
    }
    /**
     * Grants permission to delete an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteFunction.html
     */
    toDeleteFunction() {
        this.to('lambda:DeleteFunction');
        return this;
    }
    /**
     * Grants permission to remove a concurrent execution limit from an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteFunctionConcurrency.html
     */
    toDeleteFunctionConcurrency() {
        this.to('lambda:DeleteFunctionConcurrency');
        return this;
    }
    /**
     * Grants permission to delete the configuration for asynchronous invocation for an AWS Lambda function, version, or alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteFunctionEventInvokeConfig.html
     */
    toDeleteFunctionEventInvokeConfig() {
        this.to('lambda:DeleteFunctionEventInvokeConfig');
        return this;
    }
    /**
     * Grants permission to delete a version of an AWS Lambda layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteLayerVersion.html
     */
    toDeleteLayerVersion() {
        this.to('lambda:DeleteLayerVersion');
        return this;
    }
    /**
     * Grants permission to delete the provisioned concurrency configuration for an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteProvisionedConcurrencyConfig.html
     */
    toDeleteProvisionedConcurrencyConfig() {
        this.to('lambda:DeleteProvisionedConcurrencyConfig');
        return this;
    }
    /**
     * Grants permission to disable replication for a Lambda@Edge function
     *
     * Access Level: Permissions management
     */
    toDisableReplication() {
        this.to('lambda:DisableReplication');
        return this;
    }
    /**
     * Grants permission to enable replication for a Lambda@Edge function
     *
     * Access Level: Permissions management
     */
    toEnableReplication() {
        this.to('lambda:EnableReplication');
        return this;
    }
    /**
     * Grants permission to view details about an account's limits and usage in an AWS Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetAccountSettings.html
     */
    toGetAccountSettings() {
        this.to('lambda:GetAccountSettings');
        return this;
    }
    /**
     * Grants permission to view details about an AWS Lambda function alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetAlias.html
     */
    toGetAlias() {
        this.to('lambda:GetAlias');
        return this;
    }
    /**
     * Grants permission to view details about an AWS Lambda event source mapping
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifFunctionArn()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetEventSourceMapping.html
     */
    toGetEventSourceMapping() {
        this.to('lambda:GetEventSourceMapping');
        return this;
    }
    /**
     * Grants permission to view details about an AWS Lambda function
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunction.html
     */
    toGetFunction() {
        this.to('lambda:GetFunction');
        return this;
    }
    /**
     * Grants permission to view details about the reserved concurrency configuration for a function
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionConcurrency.html
     */
    toGetFunctionConcurrency() {
        this.to('lambda:GetFunctionConcurrency');
        return this;
    }
    /**
     * Grants permission to view details about the version-specific settings of an AWS Lambda function or version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionConfiguration.html
     */
    toGetFunctionConfiguration() {
        this.to('lambda:GetFunctionConfiguration');
        return this;
    }
    /**
     * Grants permission to view the configuration for asynchronous invocation for a function, version, or alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionEventInvokeConfig.html
     */
    toGetFunctionEventInvokeConfig() {
        this.to('lambda:GetFunctionEventInvokeConfig');
        return this;
    }
    /**
     * Grants permission to view details about a version of an AWS Lambda layer. Note this action also supports GetLayerVersionByArn API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetLayerVersion.html
     */
    toGetLayerVersion() {
        this.to('lambda:GetLayerVersion');
        return this;
    }
    /**
     * Grants permission to view the resource-based policy for a version of an AWS Lambda layer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetLayerVersionPolicy.html
     */
    toGetLayerVersionPolicy() {
        this.to('lambda:GetLayerVersionPolicy');
        return this;
    }
    /**
     * Grants permission to view the resource-based policy for an AWS Lambda function, version, or alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetPolicy.html
     */
    toGetPolicy() {
        this.to('lambda:GetPolicy');
        return this;
    }
    /**
     * Grants permission to view the provisioned concurrency configuration for an AWS Lambda function's alias or version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetProvisionedConcurrencyConfig.html
     */
    toGetProvisionedConcurrencyConfig() {
        this.to('lambda:GetProvisionedConcurrencyConfig');
        return this;
    }
    /**
     * (Deprecated) Grants permission to invoke a function asynchronously
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_InvokeAsync.html
     */
    toInvokeAsync() {
        this.to('lambda:InvokeAsync');
        return this;
    }
    /**
     * Grants permission to invoke an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html
     */
    toInvokeFunction() {
        this.to('lambda:InvokeFunction');
        return this;
    }
    /**
     * Grants permission to retrieve a list of aliases for an AWS Lambda function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListAliases.html
     */
    toListAliases() {
        this.to('lambda:ListAliases');
        return this;
    }
    /**
     * Grants permission to retrieve a list of AWS Lambda event source mappings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListEventSourceMappings.html
     */
    toListEventSourceMappings() {
        this.to('lambda:ListEventSourceMappings');
        return this;
    }
    /**
     * Grants permission to retrieve a list of configurations for asynchronous invocation for a function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListFunctionEventInvokeConfigs.html
     */
    toListFunctionEventInvokeConfigs() {
        this.to('lambda:ListFunctionEventInvokeConfigs');
        return this;
    }
    /**
     * Grants permission to retrieve a list of AWS Lambda functions, with the version-specific configuration of each function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListFunctions.html
     */
    toListFunctions() {
        this.to('lambda:ListFunctions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of versions of an AWS Lambda layer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayerVersions.html
     */
    toListLayerVersions() {
        this.to('lambda:ListLayerVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of AWS Lambda layers, with details about the latest version of each layer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayers.html
     */
    toListLayers() {
        this.to('lambda:ListLayers');
        return this;
    }
    /**
     * Grants permission to retrieve a list of provisioned concurrency configurations for an AWS Lambda function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListProvisionedConcurrencyConfigs.html
     */
    toListProvisionedConcurrencyConfigs() {
        this.to('lambda:ListProvisionedConcurrencyConfigs');
        return this;
    }
    /**
     * Grants permission to retrieve a list of tags for an AWS Lambda function
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListTags.html
     */
    toListTags() {
        this.to('lambda:ListTags');
        return this;
    }
    /**
     * Grants permission to retrieve a list of versions for an AWS Lambda function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListVersionsByFunction.html
     */
    toListVersionsByFunction() {
        this.to('lambda:ListVersionsByFunction');
        return this;
    }
    /**
     * Grants permission to create an AWS Lambda layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PublishLayerVersion.html
     */
    toPublishLayerVersion() {
        this.to('lambda:PublishLayerVersion');
        return this;
    }
    /**
     * Grants permission to create an AWS Lambda function version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PublishVersion.html
     */
    toPublishVersion() {
        this.to('lambda:PublishVersion');
        return this;
    }
    /**
     * Grants permission to configure reserved concurrency for an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PutFunctionConcurrency.html
     */
    toPutFunctionConcurrency() {
        this.to('lambda:PutFunctionConcurrency');
        return this;
    }
    /**
     * Grants permission to configures options for asynchronous invocation on an AWS Lambda function, version, or alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PutFunctionEventInvokeConfig.html
     */
    toPutFunctionEventInvokeConfig() {
        this.to('lambda:PutFunctionEventInvokeConfig');
        return this;
    }
    /**
     * Grants permission to configure provisioned concurrency for an AWS Lambda function's alias or version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PutProvisionedConcurrencyConfig.html
     */
    toPutProvisionedConcurrencyConfig() {
        this.to('lambda:PutProvisionedConcurrencyConfig');
        return this;
    }
    /**
     * Grants permission to remove a statement from the permissions policy for a version of an AWS Lambda layer
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_RemoveLayerVersionPermission.html
     */
    toRemoveLayerVersionPermission() {
        this.to('lambda:RemoveLayerVersionPermission');
        return this;
    }
    /**
     * Grants permission to revoke function-use permission from an AWS service or another account
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPrincipal()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_RemovePermission.html
     */
    toRemovePermission() {
        this.to('lambda:RemovePermission');
        return this;
    }
    /**
     * Grants permission to add tags to an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_TagResources.html
     */
    toTagResource() {
        this.to('lambda:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        this.to('lambda:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the configuration of an AWS Lambda function's alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateAlias.html
     */
    toUpdateAlias() {
        this.to('lambda:UpdateAlias');
        return this;
    }
    /**
     * Grants permission to update the configuration of an AWS Lambda event source mapping
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFunctionArn()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateEventSourceMapping.html
     */
    toUpdateEventSourceMapping() {
        this.to('lambda:UpdateEventSourceMapping');
        return this;
    }
    /**
     * Grants permission to update the code of an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionCode.html
     */
    toUpdateFunctionCode() {
        this.to('lambda:UpdateFunctionCode');
        return this;
    }
    /**
     * Grants permission to modify the version-specific settings of an AWS Lambda function
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifLayer()
     * - .ifVpcIds()
     * - .ifSubnetIds()
     * - .ifSecurityGroupIds()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionConfiguration.html
     */
    toUpdateFunctionConfiguration() {
        this.to('lambda:UpdateFunctionConfiguration');
        return this;
    }
    /**
     * Grants permission to modify the configuration for asynchronous invocation for an AWS Lambda function, version, or alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionEventInvokeConfig.html
     */
    toUpdateFunctionEventInvokeConfig() {
        this.to('lambda:UpdateFunctionEventInvokeConfig');
        return this;
    }
    /**
     * Adds a resource of type function to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param functionName - Identifier for the functionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFunction(functionName, account, region, partition) {
        var arn = 'arn:${Partition}:lambda:${Region}:${Account}:function:${FunctionName}';
        arn = arn.replace('${FunctionName}', functionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type function version to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param functionName - Identifier for the functionName.
     * @param version - Identifier for the version.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFunctionVersion(functionName, version, account, region, partition) {
        var arn = 'arn:${Partition}:lambda:${Region}:${Account}:function:${FunctionName}:${Version}';
        arn = arn.replace('${FunctionName}', functionName);
        arn = arn.replace('${Version}', version);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type function alias to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param functionName - Identifier for the functionName.
     * @param alias - Identifier for the alias.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFunctionAlias(functionName, alias, account, region, partition) {
        var arn = 'arn:${Partition}:lambda:${Region}:${Account}:function:${FunctionName}:${Alias}';
        arn = arn.replace('${FunctionName}', functionName);
        arn = arn.replace('${Alias}', alias);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type layer to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param layerName - Identifier for the layerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLayer(layerName, account, region, partition) {
        var arn = 'arn:${Partition}:lambda:${Region}:${Account}:layer:${LayerName}';
        arn = arn.replace('${LayerName}', layerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type layerVersion to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param layerName - Identifier for the layerName.
     * @param layerVersion - Identifier for the layerVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLayerVersion(layerName, layerVersion, account, region, partition) {
        var arn = 'arn:${Partition}:lambda:${Region}:${Account}:layer:${LayerName}:${LayerVersion}';
        arn = arn.replace('${LayerName}', layerName);
        arn = arn.replace('${LayerVersion}', layerVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type eventSourceMapping to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEventSourceMapping(uUID, account, region, partition) {
        var arn = 'arn:${Partition}:lambda:${Region}:${Account}:event-source-mapping:${UUID}';
        arn = arn.replace('${UUID}', uUID);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the ARN of an AWS Lambda function
     *
     * Applies to actions:
     * - .toCreateEventSourceMapping()
     * - .toDeleteEventSourceMapping()
     * - .toGetEventSourceMapping()
     * - .toUpdateEventSourceMapping()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFunctionArn(value, operator) {
        return this.if(`lambda:FunctionArn`, value, operator || 'ArnLike');
    }
    /**
     * Filters access by the ARN of an AWS Lambda layer
     *
     * Applies to actions:
     * - .toCreateFunction()
     * - .toUpdateFunctionConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLayer(value, operator) {
        return this.if(`lambda:Layer`, value, operator || 'StringLike');
    }
    /**
     * Filters access by restricting the AWS service or account that can invoke a function
     *
     * Applies to actions:
     * - .toAddPermission()
     * - .toRemovePermission()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrincipal(value, operator) {
        return this.if(`lambda:Principal`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ID of security groups configured for the AWS Lambda function
     *
     * Applies to actions:
     * - .toCreateFunction()
     * - .toUpdateFunctionConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecurityGroupIds(value, operator) {
        return this.if(`lambda:SecurityGroupIds`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ID of subnets configured for the AWS Lambda function
     *
     * Applies to actions:
     * - .toCreateFunction()
     * - .toUpdateFunctionConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubnetIds(value, operator) {
        return this.if(`lambda:SubnetIds`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ID of the VPC configured for the AWS Lambda function
     *
     * Applies to actions:
     * - .toCreateFunction()
     * - .toUpdateFunctionConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcIds(value, operator) {
        return this.if(`lambda:VpcIds`, value, operator || 'StringLike');
    }
}
exports.Lambda = Lambda;
//# sourceMappingURL=data:application/json;base64,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