"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lex = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [lex](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonlex.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Lex extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [lex](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonlex.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'lex';
        this.accessLevelList = {
            "Write": [
                "CreateBotVersion",
                "CreateIntentVersion",
                "CreateSlotTypeVersion",
                "DeleteBot",
                "DeleteBotAlias",
                "DeleteBotChannelAssociation",
                "DeleteBotVersion",
                "DeleteIntent",
                "DeleteIntentVersion",
                "DeleteSession",
                "DeleteSlotType",
                "DeleteSlotTypeVersion",
                "DeleteUtterances",
                "PostContent",
                "PostText",
                "PutBot",
                "PutBotAlias",
                "PutIntent",
                "PutSession",
                "PutSlotType",
                "StartImport"
            ],
            "Read": [
                "GetBot",
                "GetBotAlias",
                "GetBotChannelAssociation",
                "GetBuiltinIntent",
                "GetBuiltinIntents",
                "GetBuiltinSlotTypes",
                "GetExport",
                "GetImport",
                "GetIntent",
                "GetSession",
                "GetSlotType",
                "ListTagsForResource"
            ],
            "List": [
                "GetBotAliases",
                "GetBotChannelAssociations",
                "GetBotVersions",
                "GetBots",
                "GetIntentVersions",
                "GetIntents",
                "GetSlotTypeVersions",
                "GetSlotTypes",
                "GetUtterancesView"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Creates a new version based on the $LATEST version of the specified bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_CreateBotVersion.html
     */
    toCreateBotVersion() {
        this.to('lex:CreateBotVersion');
        return this;
    }
    /**
     * Creates a new version based on the $LATEST version of the specified intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_CreateIntentVersion.html
     */
    toCreateIntentVersion() {
        this.to('lex:CreateIntentVersion');
        return this;
    }
    /**
     * Creates a new version based on the $LATEST version of the specified slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_CreateSlotTypeVersion.html
     */
    toCreateSlotTypeVersion() {
        this.to('lex:CreateSlotTypeVersion');
        return this;
    }
    /**
     * Deletes all versions of a bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBot.html
     */
    toDeleteBot() {
        this.to('lex:DeleteBot');
        return this;
    }
    /**
     * Deletes an alias for a specific bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotAlias.html
     */
    toDeleteBotAlias() {
        this.to('lex:DeleteBotAlias');
        return this;
    }
    /**
     * Deletes the association between a Amazon Lex bot alias and a messaging platform.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotChannelAssociation.html
     */
    toDeleteBotChannelAssociation() {
        this.to('lex:DeleteBotChannelAssociation');
        return this;
    }
    /**
     * Deletes a specific version of a bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotVersion.html
     */
    toDeleteBotVersion() {
        this.to('lex:DeleteBotVersion');
        return this;
    }
    /**
     * Deletes all versions of an intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteIntent.html
     */
    toDeleteIntent() {
        this.to('lex:DeleteIntent');
        return this;
    }
    /**
     * Deletes a specific version of an intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteIntentVersion.html
     */
    toDeleteIntentVersion() {
        this.to('lex:DeleteIntentVersion');
        return this;
    }
    /**
     * Removes session information for a specified bot, alias, and user ID.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_DeleteSession.html
     */
    toDeleteSession() {
        this.to('lex:DeleteSession');
        return this;
    }
    /**
     * Deletes all versions of a slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteSlotType.html
     */
    toDeleteSlotType() {
        this.to('lex:DeleteSlotType');
        return this;
    }
    /**
     * Deletes a specific version of a slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteSlotTypeVersion.html
     */
    toDeleteSlotTypeVersion() {
        this.to('lex:DeleteSlotTypeVersion');
        return this;
    }
    /**
     * Deletes the information Amazon Lex maintains for utterances on a specific bot and userId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteUtterances.html
     */
    toDeleteUtterances() {
        this.to('lex:DeleteUtterances');
        return this;
    }
    /**
     * Returns information for a specific bot. In addition to the bot name, the bot version or alias is required.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBot.html
     */
    toGetBot() {
        this.to('lex:GetBot');
        return this;
    }
    /**
     * Returns information about a Amazon Lex bot alias.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotAlias.html
     */
    toGetBotAlias() {
        this.to('lex:GetBotAlias');
        return this;
    }
    /**
     * Returns a list of aliases for a given Amazon Lex bot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotAliases.html
     */
    toGetBotAliases() {
        this.to('lex:GetBotAliases');
        return this;
    }
    /**
     * Returns information about the association between a Amazon Lex bot and a messaging platform.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotChannelAssociation.html
     */
    toGetBotChannelAssociation() {
        this.to('lex:GetBotChannelAssociation');
        return this;
    }
    /**
     * Returns a list of all of the channels associated with a single bot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotChannelAssociations.html
     */
    toGetBotChannelAssociations() {
        this.to('lex:GetBotChannelAssociations');
        return this;
    }
    /**
     * Returns information for all versions of a specific bot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotVersions.html
     */
    toGetBotVersions() {
        this.to('lex:GetBotVersions');
        return this;
    }
    /**
     * Returns information for the $LATEST version of all bots, subject to filters provided by the client.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBots.html
     */
    toGetBots() {
        this.to('lex:GetBots');
        return this;
    }
    /**
     * Returns information about a built-in intent.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinIntent.html
     */
    toGetBuiltinIntent() {
        this.to('lex:GetBuiltinIntent');
        return this;
    }
    /**
     * Gets a list of built-in intents that meet the specified criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinIntents.html
     */
    toGetBuiltinIntents() {
        this.to('lex:GetBuiltinIntents');
        return this;
    }
    /**
     * Gets a list of built-in slot types that meet the specified criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinSlotTypes.html
     */
    toGetBuiltinSlotTypes() {
        this.to('lex:GetBuiltinSlotTypes');
        return this;
    }
    /**
     * Exports Amazon Lex Resource in a requested format.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetExport.html
     */
    toGetExport() {
        this.to('lex:GetExport');
        return this;
    }
    /**
     * Gets information about an import job started with StartImport.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetImport.html
     */
    toGetImport() {
        this.to('lex:GetImport');
        return this;
    }
    /**
     * Returns information for a specific intent. In addition to the intent name, you must also specify the intent version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetIntent.html
     */
    toGetIntent() {
        this.to('lex:GetIntent');
        return this;
    }
    /**
     * Returns information for all versions of a specific intent.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetIntentVersions.html
     */
    toGetIntentVersions() {
        this.to('lex:GetIntentVersions');
        return this;
    }
    /**
     * Returns information for the $LATEST version of all intents, subject to filters provided by the client.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetIntents.html
     */
    toGetIntents() {
        this.to('lex:GetIntents');
        return this;
    }
    /**
     * Returns session information for a specified bot, alias, and user ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_GetSession.html
     */
    toGetSession() {
        this.to('lex:GetSession');
        return this;
    }
    /**
     * Returns information about a specific version of a slot type. In addition to specifying the slot type name, you must also specify the slot type version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotType.html
     */
    toGetSlotType() {
        this.to('lex:GetSlotType');
        return this;
    }
    /**
     * Returns information for all versions of a specific slot type.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotTypeVersions.html
     */
    toGetSlotTypeVersions() {
        this.to('lex:GetSlotTypeVersions');
        return this;
    }
    /**
     * Returns information for the $LATEST version of all slot types, subject to filters provided by the client.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotTypes.html
     */
    toGetSlotTypes() {
        this.to('lex:GetSlotTypes');
        return this;
    }
    /**
     * Returns a view of aggregate utterance data for versions of a bot for a recent time period.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetUtterancesView.html
     */
    toGetUtterancesView() {
        this.to('lex:GetUtterancesView');
        return this;
    }
    /**
     * Lists tags for a Lex resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('lex:ListTagsForResource');
        return this;
    }
    /**
     * Sends user input (text or speech) to Amazon Lex.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostContent.html
     */
    toPostContent() {
        this.to('lex:PostContent');
        return this;
    }
    /**
     * Sends user input (text-only) to Amazon Lex.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostText.html
     */
    toPostText() {
        this.to('lex:PostText');
        return this;
    }
    /**
     * Creates or updates the $LATEST version of a Amazon Lex conversational bot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutBot.html
     */
    toPutBot() {
        this.to('lex:PutBot');
        return this;
    }
    /**
     * Creates or updates an alias for the specific bot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutBotAlias.html
     */
    toPutBotAlias() {
        this.to('lex:PutBotAlias');
        return this;
    }
    /**
     * Creates or updates the $LATEST version of an intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutIntent.html
     */
    toPutIntent() {
        this.to('lex:PutIntent');
        return this;
    }
    /**
     * Creates a new session or modifies an existing session with an Amazon Lex bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PutSession.html
     */
    toPutSession() {
        this.to('lex:PutSession');
        return this;
    }
    /**
     * Creates or updates the $LATEST version of a slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutSlotType.html
     */
    toPutSlotType() {
        this.to('lex:PutSlotType');
        return this;
    }
    /**
     * Starts a job to import a resource to Amazon Lex.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_StartImport.html
     */
    toStartImport() {
        this.to('lex:StartImport');
        return this;
    }
    /**
     * Adds or overwrites tags to a Lex resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_TagResource.html
     */
    toTagResource() {
        this.to('lex:TagResource');
        return this;
    }
    /**
     * Removes tags from a Lex resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        this.to('lex:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type bot to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotMetadata.html
     *
     * @param botName - Identifier for the botName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBot(botName, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bot version to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotMetadata.html
     *
     * @param botName - Identifier for the botName.
     * @param botVersion - Identifier for the botVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBotVersion(botName, botVersion, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}:${BotVersion}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${BotVersion}', botVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bot alias to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotAliasMetadata.html
     *
     * @param botName - Identifier for the botName.
     * @param botAlias - Identifier for the botAlias.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBotAlias(botName, botAlias, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}:${BotAlias}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${BotAlias}', botAlias);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type channel to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotChannelAssociation.html
     *
     * @param botName - Identifier for the botName.
     * @param botAlias - Identifier for the botAlias.
     * @param channelName - Identifier for the channelName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannel(botName, botAlias, channelName, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot-channel:${BotName}:${BotAlias}:${ChannelName}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${BotAlias}', botAlias);
        arn = arn.replace('${ChannelName}', channelName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type intent version to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_Intent.html
     *
     * @param intentName - Identifier for the intentName.
     * @param intentVersion - Identifier for the intentVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIntentVersion(intentName, intentVersion, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:intent:${IntentName}:${IntentVersion}';
        arn = arn.replace('${IntentName}', intentName);
        arn = arn.replace('${IntentVersion}', intentVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type slottype version to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_SlotTypeMetadata.html
     *
     * @param slotName - Identifier for the slotName.
     * @param slotVersion - Identifier for the slotVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSlottypeVersion(slotName, slotVersion, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:slottype:${SlotName}:${SlotVersion}';
        arn = arn.replace('${SlotName}', slotName);
        arn = arn.replace('${SlotVersion}', slotVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Enables you to control access based on the intents included in the request.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAssociatedIntents(value, operator) {
        return this.if(`lex:associatedIntents`, value, operator || 'StringLike');
    }
    /**
     * Enables you to control access based on the slot types included in the request.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAssociatedSlotTypes(value, operator) {
        return this.if(`lex:associatedSlotTypes`, value, operator || 'StringLike');
    }
    /**
     * Enables you to control access based on the channel type included in the request.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChannelType(value, operator) {
        return this.if(`lex:channelType`, value, operator || 'StringLike');
    }
}
exports.Lex = Lex;
//# sourceMappingURL=data:application/json;base64,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