"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kafka = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [kafka](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedstreamingforapachekafka.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kafka extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [kafka](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedstreamingforapachekafka.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'kafka';
        this.accessLevelList = {
            "Write": [
                "BatchAssociateScramSecret",
                "BatchDisassociateScramSecret",
                "CreateCluster",
                "CreateConfiguration",
                "DeleteCluster",
                "DeleteConfiguration",
                "UpdateBrokerCount",
                "UpdateBrokerStorage",
                "UpdateClusterConfiguration",
                "UpdateClusterKafkaVersion",
                "UpdateConfiguration",
                "UpdateMonitoring"
            ],
            "Read": [
                "DescribeCluster",
                "DescribeClusterOperation",
                "DescribeConfiguration",
                "DescribeConfigurationRevision",
                "GetBootstrapBrokers"
            ],
            "List": [
                "GetCompatibleKafkaVersions",
                "ListClusterOperations",
                "ListClusters",
                "ListConfigurations",
                "ListNodes",
                "ListScramSecrets",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to associate one or more Scram Secrets with an Amazon MSK cluster.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:CreateGrant
     * - kms:RetireGrant
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-scram-secrets.html#BatchAssociateScramSecret
     */
    toBatchAssociateScramSecret() {
        this.to('kafka:BatchAssociateScramSecret');
        return this;
    }
    /**
     * Grants permission to disassociate one or more Scram Secrets from an Amazon MSK cluster.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:RetireGrant
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-scram-secrets.html#BatchDisassociateScramSecret
     */
    toBatchDisassociateScramSecret() {
        this.to('kafka:BatchDisassociateScramSecret');
        return this;
    }
    /**
     * Grants permission to create a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - kms:CreateGrant
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#CreateCluster
     */
    toCreateCluster() {
        this.to('kafka:CreateCluster');
        return this;
    }
    /**
     * Grants permission to create a configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#CreateConfiguration
     */
    toCreateConfiguration() {
        this.to('kafka:CreateConfiguration');
        return this;
    }
    /**
     * Grants permission to delete a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DeleteCluster
     */
    toDeleteCluster() {
        this.to('kafka:DeleteCluster');
        return this;
    }
    /**
     * Grants permission to delete the specified MSK configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-arn.html#DeleteConfiguration
     */
    toDeleteConfiguration() {
        this.to('kafka:DeleteConfiguration');
        return this;
    }
    /**
     * Grants permission to describe a cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DescribeCluster
     */
    toDescribeCluster() {
        this.to('kafka:DescribeCluster');
        return this;
    }
    /**
     * Returns a description of the cluster operation specified by the ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/operations-clusteroperationarn.html#DescribeClusterOperation
     */
    toDescribeClusterOperation() {
        this.to('kafka:DescribeClusterOperation');
        return this;
    }
    /**
     * Grants permission to describe a configuration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-configurationarn.html#DescribeConfiguration
     */
    toDescribeConfiguration() {
        this.to('kafka:DescribeConfiguration');
        return this;
    }
    /**
     * Grants permission to describe a configuration revision.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-configurationarn-revision.html#DescribeConfigurationRevision
     */
    toDescribeConfigurationRevision() {
        this.to('kafka:DescribeConfigurationRevision');
        return this;
    }
    /**
     * Grants permission to get connection details for the broker nodes in a cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-bootstrap-brokers.html#GetBootstrapBrokers
     */
    toGetBootstrapBrokers() {
        this.to('kafka:GetBootstrapBrokers');
        return this;
    }
    /**
     * Returns a list of the Apache Kafka versions to which you can update this cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/compatible-kafka-versions.html#GetCompatibleKafkaVersions
     */
    toGetCompatibleKafkaVersions() {
        this.to('kafka:GetCompatibleKafkaVersions');
        return this;
    }
    /**
     * Returns a list of all the operations that have been performed on the specified MSK cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-operations.html#ListClusterOperations
     */
    toListClusterOperations() {
        this.to('kafka:ListClusterOperations');
        return this;
    }
    /**
     * Grants permission to return a list of all clusters in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#ListClusters
     */
    toListClusters() {
        this.to('kafka:ListClusters');
        return this;
    }
    /**
     * Grants permission to return a list of all configurations in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#CreateConfiguration
     */
    toListConfigurations() {
        this.to('kafka:ListConfigurations');
        return this;
    }
    /**
     * Grants permission to return a list of nodes in a cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes.html#ListNodes
     */
    toListNodes() {
        this.to('kafka:ListNodes');
        return this;
    }
    /**
     * Grants permission to return a list of the Scram Secrets associated with an Amazon MSK cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-scram-secrets.html#ListScramSecrets
     */
    toListScramSecrets() {
        this.to('kafka:ListScramSecrets');
        return this;
    }
    /**
     * Grants permission to list tags of a MSK resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#ListTagsForResource
     */
    toListTagsForResource() {
        this.to('kafka:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to tag a MSK resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#TagResource
     */
    toTagResource() {
        this.to('kafka:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a MSK resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#UntagResource
     */
    toUntagResource() {
        this.to('kafka:UntagResource');
        return this;
    }
    /**
     * Updates the number of broker nodes of the cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-count.html#UpdateBrokerCount
     */
    toUpdateBrokerCount() {
        this.to('kafka:UpdateBrokerCount');
        return this;
    }
    /**
     * Updates the storage size of the broker nodes of the cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-storage.html#UpdateBrokerStorage
     */
    toUpdateBrokerStorage() {
        this.to('kafka:UpdateBrokerStorage');
        return this;
    }
    /**
     * Update Kafka configuration running on a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-configuration.html#UpdateClusterConfiguration
     */
    toUpdateClusterConfiguration() {
        this.to('kafka:UpdateClusterConfiguration');
        return this;
    }
    /**
     * Updates the cluster to the specified Apache Kafka version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-version.html#UpdateClusterKafkaVersion
     */
    toUpdateClusterKafkaVersion() {
        this.to('kafka:UpdateClusterKafkaVersion');
        return this;
    }
    /**
     * Grants permission to create a new revision of the configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-arn.html#updateconfiguration
     */
    toUpdateConfiguration() {
        this.to('kafka:UpdateConfiguration');
        return this;
    }
    /**
     * Updates the monitoring settings for the cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-monitoring.html#UpdateMonitoring
     */
    toUpdateMonitoring() {
        this.to('kafka:UpdateMonitoring');
        return this;
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * @param clusterName - Identifier for the clusterName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterName, uUID, account, region, partition) {
        var arn = 'arn:${Partition}:kafka:${Region}:${Account}:cluster/${ClusterName}/${UUID}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${UUID}', uUID);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Kafka = Kafka;
//# sourceMappingURL=data:application/json;base64,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