"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mgh = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mgh](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmigrationhub.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mgh extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mgh](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmigrationhub.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mgh';
        this.accessLevelList = {
            "Write": [
                "AssociateCreatedArtifact",
                "AssociateDiscoveredResource",
                "CreateHomeRegionControl",
                "CreateProgressUpdateStream",
                "DeleteProgressUpdateStream",
                "DisassociateCreatedArtifact",
                "DisassociateDiscoveredResource",
                "ImportMigrationTask",
                "NotifyApplicationState",
                "NotifyMigrationTaskState",
                "PutResourceAttributes"
            ],
            "Read": [
                "DescribeApplicationState",
                "DescribeMigrationTask",
                "GetHomeRegion"
            ],
            "List": [
                "DescribeHomeRegionControls",
                "ListCreatedArtifacts",
                "ListDiscoveredResources",
                "ListMigrationTasks",
                "ListProgressUpdateStreams"
            ]
        };
    }
    /**
     * Associate a given AWS artifact to a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_AssociateCreatedArtifact.html
     */
    toAssociateCreatedArtifact() {
        this.to('mgh:AssociateCreatedArtifact');
        return this;
    }
    /**
     * Associate a given ADS resource to a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_AssociateDiscoveredResource.html
     */
    toAssociateDiscoveredResource() {
        this.to('mgh:AssociateDiscoveredResource');
        return this;
    }
    /**
     * Create a Migration Hub Home Region Control
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_CreateHomeRegionControl.html
     */
    toCreateHomeRegionControl() {
        this.to('mgh:CreateHomeRegionControl');
        return this;
    }
    /**
     * Create a ProgressUpdateStream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_CreateProgressUpdateStream.html
     */
    toCreateProgressUpdateStream() {
        this.to('mgh:CreateProgressUpdateStream');
        return this;
    }
    /**
     * Delete a ProgressUpdateStream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DeleteProgressUpdateStream.html
     */
    toDeleteProgressUpdateStream() {
        this.to('mgh:DeleteProgressUpdateStream');
        return this;
    }
    /**
     * Get an Application Discovery Service Application's state
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeApplicationState.html
     */
    toDescribeApplicationState() {
        this.to('mgh:DescribeApplicationState');
        return this;
    }
    /**
     * List Home Region Controls
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeHomeRegionControls.html
     */
    toDescribeHomeRegionControls() {
        this.to('mgh:DescribeHomeRegionControls');
        return this;
    }
    /**
     * Describe a MigrationTask
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeMigrationTask.html
     */
    toDescribeMigrationTask() {
        this.to('mgh:DescribeMigrationTask');
        return this;
    }
    /**
     * Disassociate a given AWS artifact from a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DisassociateCreatedArtifact.html
     */
    toDisassociateCreatedArtifact() {
        this.to('mgh:DisassociateCreatedArtifact');
        return this;
    }
    /**
     * Disassociate a given ADS resource from a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DisassociateDiscoveredResource.html
     */
    toDisassociateDiscoveredResource() {
        this.to('mgh:DisassociateDiscoveredResource');
        return this;
    }
    /**
     * Get the Migration Hub Home Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_GetHomeRegion.html
     */
    toGetHomeRegion() {
        this.to('mgh:GetHomeRegion');
        return this;
    }
    /**
     * Import a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ImportMigrationTask.html
     */
    toImportMigrationTask() {
        this.to('mgh:ImportMigrationTask');
        return this;
    }
    /**
     * List associated created artifacts for a MigrationTask
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListCreatedArtifacts.html
     */
    toListCreatedArtifacts() {
        this.to('mgh:ListCreatedArtifacts');
        return this;
    }
    /**
     * List associated ADS resources from MigrationTask
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListDiscoveredResources.html
     */
    toListDiscoveredResources() {
        this.to('mgh:ListDiscoveredResources');
        return this;
    }
    /**
     * List MigrationTasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListMigrationTasks.html
     */
    toListMigrationTasks() {
        this.to('mgh:ListMigrationTasks');
        return this;
    }
    /**
     * List ProgressUpdateStreams
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListProgressUpdateStreams.html
     */
    toListProgressUpdateStreams() {
        this.to('mgh:ListProgressUpdateStreams');
        return this;
    }
    /**
     * Update an Application Discovery Service Application's state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_NotifyApplicationState.html
     */
    toNotifyApplicationState() {
        this.to('mgh:NotifyApplicationState');
        return this;
    }
    /**
     * Notify latest MigrationTask state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_NotifyMigrationTaskState.html
     */
    toNotifyMigrationTaskState() {
        this.to('mgh:NotifyMigrationTaskState');
        return this;
    }
    /**
     * Put ResourceAttributes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_PutResourceAttributes.html
     */
    toPutResourceAttributes() {
        this.to('mgh:PutResourceAttributes');
        return this;
    }
    /**
     * Adds a resource of type progressUpdateStream to the statement
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ProgressUpdateStreamSummary.html
     *
     * @param stream - Identifier for the stream.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProgressUpdateStream(stream, account, region, partition) {
        var arn = 'arn:${Partition}:mgh:${Region}:${Account}:progressUpdateStream/${Stream}';
        arn = arn.replace('${Stream}', stream);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type migrationTask to the statement
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_MigrationTask.html
     *
     * @param stream - Identifier for the stream.
     * @param task - Identifier for the task.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onMigrationTask(stream, task, account, region, partition) {
        var arn = 'arn:${Partition}:mgh:${Region}:${Account}:progressUpdateStream/${Stream}/migrationTask/${Task}';
        arn = arn.replace('${Stream}', stream);
        arn = arn.replace('${Task}', task);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mgh = Mgh;
//# sourceMappingURL=data:application/json;base64,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