"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Opsworks = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [opsworks](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsopsworks.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Opsworks extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [opsworks](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsopsworks.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'opsworks';
        this.accessLevelList = {
            "Write": [
                "AssignInstance",
                "AssignVolume",
                "AssociateElasticIp",
                "AttachElasticLoadBalancer",
                "CloneStack",
                "CreateApp",
                "CreateDeployment",
                "CreateInstance",
                "CreateLayer",
                "CreateStack",
                "CreateUserProfile",
                "DeleteApp",
                "DeleteInstance",
                "DeleteLayer",
                "DeleteStack",
                "DeleteUserProfile",
                "DeregisterEcsCluster",
                "DeregisterElasticIp",
                "DeregisterInstance",
                "DeregisterRdsDbInstance",
                "DeregisterVolume",
                "DetachElasticLoadBalancer",
                "DisassociateElasticIp",
                "GrantAccess",
                "RebootInstance",
                "RegisterEcsCluster",
                "RegisterElasticIp",
                "RegisterInstance",
                "RegisterRdsDbInstance",
                "RegisterVolume",
                "SetLoadBasedAutoScaling",
                "SetTimeBasedAutoScaling",
                "StartInstance",
                "StartStack",
                "StopInstance",
                "StopStack",
                "TagResource",
                "UnassignInstance",
                "UnassignVolume",
                "UntagResource",
                "UpdateApp",
                "UpdateElasticIp",
                "UpdateInstance",
                "UpdateLayer",
                "UpdateMyUserProfile",
                "UpdateRdsDbInstance",
                "UpdateStack",
                "UpdateVolume"
            ],
            "List": [
                "DescribeAgentVersions",
                "DescribeApps",
                "DescribeCommands",
                "DescribeDeployments",
                "DescribeEcsClusters",
                "DescribeElasticIps",
                "DescribeElasticLoadBalancers",
                "DescribeInstances",
                "DescribeLayers",
                "DescribeLoadBasedAutoScaling",
                "DescribeMyUserProfile",
                "DescribePermissions",
                "DescribeRaidArrays",
                "DescribeRdsDbInstances",
                "DescribeServiceErrors",
                "DescribeStackProvisioningParameters",
                "DescribeStackSummary",
                "DescribeStacks",
                "DescribeTimeBasedAutoScaling",
                "DescribeUserProfiles",
                "DescribeVolumes",
                "ListTags"
            ],
            "Read": [
                "GetHostnameSuggestion"
            ],
            "Permissions management": [
                "SetPermission",
                "UpdateUserProfile"
            ]
        };
    }
    /**
     * Assign a registered instance to a layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AssignInstance.html
     */
    toAssignInstance() {
        this.to('opsworks:AssignInstance');
        return this;
    }
    /**
     * Assigns one of the stack's registered Amazon EBS volumes to a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AssignVolume.html
     */
    toAssignVolume() {
        this.to('opsworks:AssignVolume');
        return this;
    }
    /**
     * Associates one of the stack's registered Elastic IP addresses with a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AssociateElasticIp.html
     */
    toAssociateElasticIp() {
        this.to('opsworks:AssociateElasticIp');
        return this;
    }
    /**
     * Attaches an Elastic Load Balancing load balancer to a specified layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AttachElasticLoadBalancer.html
     */
    toAttachElasticLoadBalancer() {
        this.to('opsworks:AttachElasticLoadBalancer');
        return this;
    }
    /**
     * Creates a clone of a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CloneStack.html
     */
    toCloneStack() {
        this.to('opsworks:CloneStack');
        return this;
    }
    /**
     * Creates an app for a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateApp.html
     */
    toCreateApp() {
        this.to('opsworks:CreateApp');
        return this;
    }
    /**
     * Runs deployment or stack commands
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateDeployment.html
     */
    toCreateDeployment() {
        this.to('opsworks:CreateDeployment');
        return this;
    }
    /**
     * Creates an instance in a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateInstance.html
     */
    toCreateInstance() {
        this.to('opsworks:CreateInstance');
        return this;
    }
    /**
     * Creates a layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateLayer.html
     */
    toCreateLayer() {
        this.to('opsworks:CreateLayer');
        return this;
    }
    /**
     * Creates a new stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateStack.html
     */
    toCreateStack() {
        this.to('opsworks:CreateStack');
        return this;
    }
    /**
     * Creates a new user profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateUserProfile.html
     */
    toCreateUserProfile() {
        this.to('opsworks:CreateUserProfile');
        return this;
    }
    /**
     * Deletes a specified app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteApp.html
     */
    toDeleteApp() {
        this.to('opsworks:DeleteApp');
        return this;
    }
    /**
     * Deletes a specified instance, which terminates the associated Amazon EC2 instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteInstance.html
     */
    toDeleteInstance() {
        this.to('opsworks:DeleteInstance');
        return this;
    }
    /**
     * Deletes a specified layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteLayer.html
     */
    toDeleteLayer() {
        this.to('opsworks:DeleteLayer');
        return this;
    }
    /**
     * Deletes a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteStack.html
     */
    toDeleteStack() {
        this.to('opsworks:DeleteStack');
        return this;
    }
    /**
     * Deletes a user profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteUserProfile.html
     */
    toDeleteUserProfile() {
        this.to('opsworks:DeleteUserProfile');
        return this;
    }
    /**
     * Deletes a user profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterEcsCluster.html
     */
    toDeregisterEcsCluster() {
        this.to('opsworks:DeregisterEcsCluster');
        return this;
    }
    /**
     * Deregisters a specified Elastic IP address
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterElasticIp.html
     */
    toDeregisterElasticIp() {
        this.to('opsworks:DeregisterElasticIp');
        return this;
    }
    /**
     * Deregister a registered Amazon EC2 or on-premises instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterInstance.html
     */
    toDeregisterInstance() {
        this.to('opsworks:DeregisterInstance');
        return this;
    }
    /**
     * Deregisters an Amazon RDS instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterRdsDbInstance.html
     */
    toDeregisterRdsDbInstance() {
        this.to('opsworks:DeregisterRdsDbInstance');
        return this;
    }
    /**
     * Deregisters an Amazon EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterVolume.html
     */
    toDeregisterVolume() {
        this.to('opsworks:DeregisterVolume');
        return this;
    }
    /**
     * Describes the available AWS OpsWorks agent versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeAgentVersions.html
     */
    toDescribeAgentVersions() {
        this.to('opsworks:DescribeAgentVersions');
        return this;
    }
    /**
     * Requests a description of a specified set of apps
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeApps.html
     */
    toDescribeApps() {
        this.to('opsworks:DescribeApps');
        return this;
    }
    /**
     * Describes the results of specified commands
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeCommands.html
     */
    toDescribeCommands() {
        this.to('opsworks:DescribeCommands');
        return this;
    }
    /**
     * Requests a description of a specified set of deployments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeDeployments.html
     */
    toDescribeDeployments() {
        this.to('opsworks:DescribeDeployments');
        return this;
    }
    /**
     * Describes Amazon ECS clusters that are registered with a stack
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeEcsClusters.html
     */
    toDescribeEcsClusters() {
        this.to('opsworks:DescribeEcsClusters');
        return this;
    }
    /**
     * Describes Elastic IP addresses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeElasticIps.html
     */
    toDescribeElasticIps() {
        this.to('opsworks:DescribeElasticIps');
        return this;
    }
    /**
     * Describes a stack's Elastic Load Balancing instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeElasticLoadBalancers.html
     */
    toDescribeElasticLoadBalancers() {
        this.to('opsworks:DescribeElasticLoadBalancers');
        return this;
    }
    /**
     * Requests a description of a set of instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeInstances.html
     */
    toDescribeInstances() {
        this.to('opsworks:DescribeInstances');
        return this;
    }
    /**
     * Requests a description of one or more layers in a specified stack
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeLayers.html
     */
    toDescribeLayers() {
        this.to('opsworks:DescribeLayers');
        return this;
    }
    /**
     * Describes load-based auto scaling configurations for specified layers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeLoadBasedAutoScaling.html
     */
    toDescribeLoadBasedAutoScaling() {
        this.to('opsworks:DescribeLoadBasedAutoScaling');
        return this;
    }
    /**
     * Describes a user's SSH information
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeMyUserProfile.html
     */
    toDescribeMyUserProfile() {
        this.to('opsworks:DescribeMyUserProfile');
        return this;
    }
    /**
     * Describes the permissions for a specified stack
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribePermissions.html
     */
    toDescribePermissions() {
        this.to('opsworks:DescribePermissions');
        return this;
    }
    /**
     * Describe an instance's RAID arrays
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeRaidArrays.html
     */
    toDescribeRaidArrays() {
        this.to('opsworks:DescribeRaidArrays');
        return this;
    }
    /**
     * Describes Amazon RDS instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeRdsDbInstances.html
     */
    toDescribeRdsDbInstances() {
        this.to('opsworks:DescribeRdsDbInstances');
        return this;
    }
    /**
     * Describes AWS OpsWorks service errors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeServiceErrors.html
     */
    toDescribeServiceErrors() {
        this.to('opsworks:DescribeServiceErrors');
        return this;
    }
    /**
     * Requests a description of a stack's provisioning parameters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeStackProvisioningParameters.html
     */
    toDescribeStackProvisioningParameters() {
        this.to('opsworks:DescribeStackProvisioningParameters');
        return this;
    }
    /**
     * Describes the number of layers and apps in a specified stack, and the number of instances in each state, such as running_setup or online
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeStackSummary.html
     */
    toDescribeStackSummary() {
        this.to('opsworks:DescribeStackSummary');
        return this;
    }
    /**
     * Requests a description of one or more stacks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeStacks.html
     */
    toDescribeStacks() {
        this.to('opsworks:DescribeStacks');
        return this;
    }
    /**
     * Describes time-based auto scaling configurations for specified instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeTimeBasedAutoScaling.html
     */
    toDescribeTimeBasedAutoScaling() {
        this.to('opsworks:DescribeTimeBasedAutoScaling');
        return this;
    }
    /**
     * Describe specified users
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeUserProfiles.html
     */
    toDescribeUserProfiles() {
        this.to('opsworks:DescribeUserProfiles');
        return this;
    }
    /**
     * Describes an instance's Amazon EBS volumes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeVolumes.html
     */
    toDescribeVolumes() {
        this.to('opsworks:DescribeVolumes');
        return this;
    }
    /**
     * Detaches a specified Elastic Load Balancing instance from its layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DetachElasticLoadBalancer.html
     */
    toDetachElasticLoadBalancer() {
        this.to('opsworks:DetachElasticLoadBalancer');
        return this;
    }
    /**
     * Disassociates an Elastic IP address from its instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DisassociateElasticIp.html
     */
    toDisassociateElasticIp() {
        this.to('opsworks:DisassociateElasticIp');
        return this;
    }
    /**
     * Gets a generated host name for the specified layer, based on the current host name theme
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_GetHostnameSuggestion.html
     */
    toGetHostnameSuggestion() {
        this.to('opsworks:GetHostnameSuggestion');
        return this;
    }
    /**
     * Grants RDP access to a Windows instance for a specified time period
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RebootInstance.html
     */
    toGrantAccess() {
        this.to('opsworks:GrantAccess');
        return this;
    }
    /**
     * Returns a list of tags that are applied to the specified stack or layer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_ListTags.html
     */
    toListTags() {
        this.to('opsworks:ListTags');
        return this;
    }
    /**
     * Reboots a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RebootInstance.html
     */
    toRebootInstance() {
        this.to('opsworks:RebootInstance');
        return this;
    }
    /**
     * Registers a specified Amazon ECS cluster with a stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterEcsCluster.html
     */
    toRegisterEcsCluster() {
        this.to('opsworks:RegisterEcsCluster');
        return this;
    }
    /**
     * Registers an Elastic IP address with a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterElasticIp.html
     */
    toRegisterElasticIp() {
        this.to('opsworks:RegisterElasticIp');
        return this;
    }
    /**
     * Registers instances with a specified stack that were created outside of AWS OpsWorks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterInstance.html
     */
    toRegisterInstance() {
        this.to('opsworks:RegisterInstance');
        return this;
    }
    /**
     * Registers an Amazon RDS instance with a stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterRdsDbInstance.html
     */
    toRegisterRdsDbInstance() {
        this.to('opsworks:RegisterRdsDbInstance');
        return this;
    }
    /**
     * Registers an Amazon EBS volume with a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterVolume.html
     */
    toRegisterVolume() {
        this.to('opsworks:RegisterVolume');
        return this;
    }
    /**
     * Specify the load-based auto scaling configuration for a specified layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_SetLoadBasedAutoScaling.html
     */
    toSetLoadBasedAutoScaling() {
        this.to('opsworks:SetLoadBasedAutoScaling');
        return this;
    }
    /**
     * Specifies a user's permissions
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_SetPermission.html
     */
    toSetPermission() {
        this.to('opsworks:SetPermission');
        return this;
    }
    /**
     * Specify the time-based auto scaling configuration for a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_SetTimeBasedAutoScaling.html
     */
    toSetTimeBasedAutoScaling() {
        this.to('opsworks:SetTimeBasedAutoScaling');
        return this;
    }
    /**
     * Starts a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StartInstance.html
     */
    toStartInstance() {
        this.to('opsworks:StartInstance');
        return this;
    }
    /**
     * Starts a stack's instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StartStack.html
     */
    toStartStack() {
        this.to('opsworks:StartStack');
        return this;
    }
    /**
     * Stops a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StopInstance.html
     */
    toStopInstance() {
        this.to('opsworks:StopInstance');
        return this;
    }
    /**
     * Stops a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StopStack.html
     */
    toStopStack() {
        this.to('opsworks:StopStack');
        return this;
    }
    /**
     * Apply tags to a specified stack or layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('opsworks:TagResource');
        return this;
    }
    /**
     * Unassigns a registered instance from all of it's layers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UnassignInstance.html
     */
    toUnassignInstance() {
        this.to('opsworks:UnassignInstance');
        return this;
    }
    /**
     * Unassigns an assigned Amazon EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UnassignVolume.html
     */
    toUnassignVolume() {
        this.to('opsworks:UnassignVolume');
        return this;
    }
    /**
     * Removes tags from a specified stack or layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('opsworks:UntagResource');
        return this;
    }
    /**
     * Updates a specified app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateApp.html
     */
    toUpdateApp() {
        this.to('opsworks:UpdateApp');
        return this;
    }
    /**
     * Updates a registered Elastic IP address's name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateElasticIp.html
     */
    toUpdateElasticIp() {
        this.to('opsworks:UpdateElasticIp');
        return this;
    }
    /**
     * Updates a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateInstance.html
     */
    toUpdateInstance() {
        this.to('opsworks:UpdateInstance');
        return this;
    }
    /**
     * Updates a specified layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateLayer.html
     */
    toUpdateLayer() {
        this.to('opsworks:UpdateLayer');
        return this;
    }
    /**
     * Updates a user's SSH public key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateMyUserProfile.html
     */
    toUpdateMyUserProfile() {
        this.to('opsworks:UpdateMyUserProfile');
        return this;
    }
    /**
     * Updates an Amazon RDS instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateRdsDbInstance.html
     */
    toUpdateRdsDbInstance() {
        this.to('opsworks:UpdateRdsDbInstance');
        return this;
    }
    /**
     * Updates a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateStack.html
     */
    toUpdateStack() {
        this.to('opsworks:UpdateStack');
        return this;
    }
    /**
     * Updates a specified user profile
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateUserProfile.html
     */
    toUpdateUserProfile() {
        this.to('opsworks:UpdateUserProfile');
        return this;
    }
    /**
     * Updates an Amazon EBS volume's name or mount point
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateVolume.html
     */
    toUpdateVolume() {
        this.to('opsworks:UpdateVolume');
        return this;
    }
    /**
     * Adds a resource of type stack to the statement
     *
     * https://docs.aws.amazon.com/opsworks/latest/userguide/workingstacks.html
     *
     * @param stackId - Identifier for the stackId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onStack(stackId, account, region, partition) {
        var arn = 'arn:${Partition}:opsworks:${Region}:${Account}:stack/${StackId}/';
        arn = arn.replace('${StackId}', stackId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Opsworks = Opsworks;
//# sourceMappingURL=data:application/json;base64,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