"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mobiletargeting = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mobiletargeting](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonpinpoint.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mobiletargeting extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mobiletargeting](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonpinpoint.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mobiletargeting';
        this.accessLevelList = {
            "Write": [
                "CreateApp",
                "CreateCampaign",
                "CreateEmailTemplate",
                "CreateExportJob",
                "CreateImportJob",
                "CreateJourney",
                "CreatePushTemplate",
                "CreateRecommenderConfiguration",
                "CreateSegment",
                "CreateSmsTemplate",
                "CreateVoiceTemplate",
                "DeleteAdmChannel",
                "DeleteApnsChannel",
                "DeleteApnsSandboxChannel",
                "DeleteApnsVoipChannel",
                "DeleteApnsVoipSandboxChannel",
                "DeleteApp",
                "DeleteBaiduChannel",
                "DeleteCampaign",
                "DeleteEmailChannel",
                "DeleteEmailTemplate",
                "DeleteEndpoint",
                "DeleteEventStream",
                "DeleteGcmChannel",
                "DeleteJourney",
                "DeletePushTemplate",
                "DeleteRecommenderConfiguration",
                "DeleteSegment",
                "DeleteSmsChannel",
                "DeleteSmsTemplate",
                "DeleteUserEndpoints",
                "DeleteVoiceChannel",
                "DeleteVoiceTemplate",
                "PutEventStream",
                "PutEvents",
                "RemoveAttributes",
                "SendMessages",
                "SendUsersMessages",
                "UpdateAdmChannel",
                "UpdateApnsChannel",
                "UpdateApnsSandboxChannel",
                "UpdateApnsVoipChannel",
                "UpdateApnsVoipSandboxChannel",
                "UpdateApplicationSettings",
                "UpdateBaiduChannel",
                "UpdateCampaign",
                "UpdateEmailChannel",
                "UpdateEmailTemplate",
                "UpdateEndpoint",
                "UpdateEndpointsBatch",
                "UpdateGcmChannel",
                "UpdateJourney",
                "UpdateJourneyState",
                "UpdatePushTemplate",
                "UpdateRecommenderConfiguration",
                "UpdateSegment",
                "UpdateSmsChannel",
                "UpdateSmsTemplate",
                "UpdateTemplateActiveVersion",
                "UpdateVoiceChannel",
                "UpdateVoiceTemplate"
            ],
            "Read": [
                "GetAdmChannel",
                "GetApnsChannel",
                "GetApnsSandboxChannel",
                "GetApnsVoipChannel",
                "GetApnsVoipSandboxChannel",
                "GetApp",
                "GetBaiduChannel",
                "GetCampaign",
                "GetCampaignVersion",
                "GetEmailChannel",
                "GetEmailTemplate",
                "GetEndpoint",
                "GetEventStream",
                "GetExportJob",
                "GetGcmChannel",
                "GetImportJob",
                "GetJourney",
                "GetPushTemplate",
                "GetRecommenderConfiguration",
                "GetSegment",
                "GetSegmentVersion",
                "GetSmsChannel",
                "GetSmsTemplate",
                "GetUserEndpoints",
                "GetVoiceChannel",
                "GetVoiceTemplate",
                "PhoneNumberValidate"
            ],
            "List": [
                "GetApplicationSettings",
                "GetApps",
                "GetCampaignActivities",
                "GetCampaignVersions",
                "GetCampaigns",
                "GetChannels",
                "GetExportJobs",
                "GetImportJobs",
                "GetRecommenderConfigurations",
                "GetSegmentExportJobs",
                "GetSegmentImportJobs",
                "GetSegmentVersions",
                "GetSegments",
                "ListJourneys",
                "ListTagsForResource",
                "ListTemplateVersions",
                "ListTemplates"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Create an app.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods
     */
    toCreateApp() {
        this.to('mobiletargeting:CreateApp');
        return this;
    }
    /**
     * Create a campaign for an app.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaigns.html#rest-api-campaigns-methods
     */
    toCreateCampaign() {
        this.to('mobiletargeting:CreateCampaign');
        return this;
    }
    /**
     * Create an email template.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods
     */
    toCreateEmailTemplate() {
        this.to('mobiletargeting:CreateEmailTemplate');
        return this;
    }
    /**
     * Create an export job that exports endpoint definitions to Amazon S3.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods
     */
    toCreateExportJob() {
        this.to('mobiletargeting:CreateExportJob');
        return this;
    }
    /**
     * Import endpoint definitions from to create a segment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-jobs.html#rest-api-import-jobs-methods
     */
    toCreateImportJob() {
        this.to('mobiletargeting:CreateImportJob');
        return this;
    }
    /**
     * Create a Journey for an app.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods
     */
    toCreateJourney() {
        this.to('mobiletargeting:CreateJourney');
        return this;
    }
    /**
     * Create a push notification template.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods
     */
    toCreatePushTemplate() {
        this.to('mobiletargeting:CreatePushTemplate');
        return this;
    }
    /**
     * Create an Amazon Pinpoint configuration for a recommender model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html#CreateRecommenderConfiguration
     */
    toCreateRecommenderConfiguration() {
        this.to('mobiletargeting:CreateRecommenderConfiguration');
        return this;
    }
    /**
     * Create a segment that is based on endpoint data reported to Pinpoint by your app. To allow a user to create a segment by importing endpoint data from outside of Pinpoint, allow the mobiletargeting:CreateImportJob action.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segments.html#rest-api-segments-methods
     */
    toCreateSegment() {
        this.to('mobiletargeting:CreateSegment');
        return this;
    }
    /**
     * Create an sms message template.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods
     */
    toCreateSmsTemplate() {
        this.to('mobiletargeting:CreateSmsTemplate');
        return this;
    }
    /**
     * Create a voice message template.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods
     */
    toCreateVoiceTemplate() {
        this.to('mobiletargeting:CreateVoiceTemplate');
        return this;
    }
    /**
     * Delete the ADM channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-adm-channel.html#rest-api-adm-channel-methods
     */
    toDeleteAdmChannel() {
        this.to('mobiletargeting:DeleteAdmChannel');
        return this;
    }
    /**
     * Delete the APNs channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-channel.html#rest-api-apns-channel-methods
     */
    toDeleteApnsChannel() {
        this.to('mobiletargeting:DeleteApnsChannel');
        return this;
    }
    /**
     * Delete the APNs sandbox channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-sandbox-channel.html#rest-api-apns-sandbox-channel-methods
     */
    toDeleteApnsSandboxChannel() {
        this.to('mobiletargeting:DeleteApnsSandboxChannel');
        return this;
    }
    /**
     * Delete the APNs VoIP channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-channel.html#rest-api-apns-voip-channel-methods
     */
    toDeleteApnsVoipChannel() {
        this.to('mobiletargeting:DeleteApnsVoipChannel');
        return this;
    }
    /**
     * Delete the APNs VoIP sandbox channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-sandbox-channel.html#rest-api-apns-voip-sandbox-channel-methods
     */
    toDeleteApnsVoipSandboxChannel() {
        this.to('mobiletargeting:DeleteApnsVoipSandboxChannel');
        return this;
    }
    /**
     * Delete a specific campaign.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods
     */
    toDeleteApp() {
        this.to('mobiletargeting:DeleteApp');
        return this;
    }
    /**
     * Delete the Baidu channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-baidu-channel.html#rest-api-baidu-channel-methods
     */
    toDeleteBaiduChannel() {
        this.to('mobiletargeting:DeleteBaiduChannel');
        return this;
    }
    /**
     * Delete a specific campaign.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign.html#rest-api-campaign-methods
     */
    toDeleteCampaign() {
        this.to('mobiletargeting:DeleteCampaign');
        return this;
    }
    /**
     * Delete the email channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-email-channel.html#rest-api-email-channel-methods
     */
    toDeleteEmailChannel() {
        this.to('mobiletargeting:DeleteEmailChannel');
        return this;
    }
    /**
     * Delete an email template or an email template version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods
     */
    toDeleteEmailTemplate() {
        this.to('mobiletargeting:DeleteEmailTemplate');
        return this;
    }
    /**
     * Delete an endpoint.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoint.html#rest-api-endpoint-methods
     */
    toDeleteEndpoint() {
        this.to('mobiletargeting:DeleteEndpoint');
        return this;
    }
    /**
     * Delete the event stream for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-event-stream.html#rest-api-event-stream-methods
     */
    toDeleteEventStream() {
        this.to('mobiletargeting:DeleteEventStream');
        return this;
    }
    /**
     * Delete the GCM channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-gcm-channel.html#rest-api-gcm-channel-methods
     */
    toDeleteGcmChannel() {
        this.to('mobiletargeting:DeleteGcmChannel');
        return this;
    }
    /**
     * Delete a specific journey.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods
     */
    toDeleteJourney() {
        this.to('mobiletargeting:DeleteJourney');
        return this;
    }
    /**
     * Delete a push notification template or a push notification template version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods
     */
    toDeletePushTemplate() {
        this.to('mobiletargeting:DeletePushTemplate');
        return this;
    }
    /**
     * Delete an Amazon Pinpoint configuration for a recommender model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#DeleteRecommenderConfiguration
     */
    toDeleteRecommenderConfiguration() {
        this.to('mobiletargeting:DeleteRecommenderConfiguration');
        return this;
    }
    /**
     * Delete a specific segment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment.html#rest-api-segment-methods
     */
    toDeleteSegment() {
        this.to('mobiletargeting:DeleteSegment');
        return this;
    }
    /**
     * Delete the SMS channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-sms-channel.html#rest-api-sms-channel-methods
     */
    toDeleteSmsChannel() {
        this.to('mobiletargeting:DeleteSmsChannel');
        return this;
    }
    /**
     * Delete an sms message template or an sms message template version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods
     */
    toDeleteSmsTemplate() {
        this.to('mobiletargeting:DeleteSmsTemplate');
        return this;
    }
    /**
     * Delete all of the endpoints that are associated with a user ID.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-user.html#rest-api-user-methods
     */
    toDeleteUserEndpoints() {
        this.to('mobiletargeting:DeleteUserEndpoints');
        return this;
    }
    /**
     * Delete the Voice channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-voice-channel.html#rest-api-voice-channel-methods
     */
    toDeleteVoiceChannel() {
        this.to('mobiletargeting:DeleteVoiceChannel');
        return this;
    }
    /**
     * Delete a voice message template or a voice message template version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods
     */
    toDeleteVoiceTemplate() {
        this.to('mobiletargeting:DeleteVoiceTemplate');
        return this;
    }
    /**
     * Retrieve information about the Amazon Device Messaging (ADM) channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-adm-channel.html#rest-api-adm-channel-methods
     */
    toGetAdmChannel() {
        this.to('mobiletargeting:GetAdmChannel');
        return this;
    }
    /**
     * Retrieve information about the APNs channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-channel.html#rest-api-apns-channel-methods
     */
    toGetApnsChannel() {
        this.to('mobiletargeting:GetApnsChannel');
        return this;
    }
    /**
     * Retrieve information about the APNs sandbox channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-sandbox-channel.html#rest-api-apns-sandbox-channel-methods
     */
    toGetApnsSandboxChannel() {
        this.to('mobiletargeting:GetApnsSandboxChannel');
        return this;
    }
    /**
     * Retrieve information about the APNs VoIP channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-channel.html#rest-api-apns-voip-channel-methods
     */
    toGetApnsVoipChannel() {
        this.to('mobiletargeting:GetApnsVoipChannel');
        return this;
    }
    /**
     * Retrieve information about the APNs VoIP sandbox channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-sandbox-channel.html#rest-api-apns-voip-sandbox-channel-methods
     */
    toGetApnsVoipSandboxChannel() {
        this.to('mobiletargeting:GetApnsVoipSandboxChannel');
        return this;
    }
    /**
     * Retrieve information about a specific app in your Amazon Pinpoint account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods
     */
    toGetApp() {
        this.to('mobiletargeting:GetApp');
        return this;
    }
    /**
     * Retrieve the default settings for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-settings.html#rest-api-settings-methods
     */
    toGetApplicationSettings() {
        this.to('mobiletargeting:GetApplicationSettings');
        return this;
    }
    /**
     * Retrieve a list of apps in your Amazon Pinpoint account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apps.html#rest-api-apps-methods
     */
    toGetApps() {
        this.to('mobiletargeting:GetApps');
        return this;
    }
    /**
     * Retrieve information about the Baidu channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-baidu-channel.html#rest-api-baidu-channel-methods
     */
    toGetBaiduChannel() {
        this.to('mobiletargeting:GetBaiduChannel');
        return this;
    }
    /**
     * Retrieve information about a specific campaign.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign.html#rest-api-campaign-methods
     */
    toGetCampaign() {
        this.to('mobiletargeting:GetCampaign');
        return this;
    }
    /**
     * Retrieve information about the activities performed by a campaign.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign-activities.html#rest-api-campaign-activities-methods
     */
    toGetCampaignActivities() {
        this.to('mobiletargeting:GetCampaignActivities');
        return this;
    }
    /**
     * Retrieve information about a specific campaign version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign-version.html#rest-api-campaign-version-methods
     */
    toGetCampaignVersion() {
        this.to('mobiletargeting:GetCampaignVersion');
        return this;
    }
    /**
     * Retrieve information about the current and prior versions of a campaign.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign-versions.html#rest-api-campaign-versions-methods
     */
    toGetCampaignVersions() {
        this.to('mobiletargeting:GetCampaignVersions');
        return this;
    }
    /**
     * Retrieve information about all campaigns for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaigns.html#rest-api-campaigns-methods
     */
    toGetCampaigns() {
        this.to('mobiletargeting:GetCampaigns');
        return this;
    }
    /**
     * Get all channels information for your app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-channels.html#rest-api-channels-methods
     */
    toGetChannels() {
        this.to('mobiletargeting:GetChannels');
        return this;
    }
    /**
     * Obtain information about the email channel in an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-email-channel.html#rest-api-email-channel-methods
     */
    toGetEmailChannel() {
        this.to('mobiletargeting:GetEmailChannel');
        return this;
    }
    /**
     * Retrieve information about a specific or the active version of an email template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods
     */
    toGetEmailTemplate() {
        this.to('mobiletargeting:GetEmailTemplate');
        return this;
    }
    /**
     * Retrieve information about a specific endpoint.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoint.html#rest-api-endpoint-methods
     */
    toGetEndpoint() {
        this.to('mobiletargeting:GetEndpoint');
        return this;
    }
    /**
     * Retrieve information about the event stream for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-event-stream.html#rest-api-event-stream-methods
     */
    toGetEventStream() {
        this.to('mobiletargeting:GetEventStream');
        return this;
    }
    /**
     * Obtain information about a specific export job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods
     */
    toGetExportJob() {
        this.to('mobiletargeting:GetExportJob');
        return this;
    }
    /**
     * Retrieve a list of all of the export jobs for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods
     */
    toGetExportJobs() {
        this.to('mobiletargeting:GetExportJobs');
        return this;
    }
    /**
     * Retrieve information about the GCM channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-gcm-channel.html#rest-api-gcm-channel-methods
     */
    toGetGcmChannel() {
        this.to('mobiletargeting:GetGcmChannel');
        return this;
    }
    /**
     * Retrieve information about a specific import job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-job.html#rest-api-import-job-methods
     */
    toGetImportJob() {
        this.to('mobiletargeting:GetImportJob');
        return this;
    }
    /**
     * Retrieve information about all import jobs for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-jobs.html#rest-api-import-jobs-methods
     */
    toGetImportJobs() {
        this.to('mobiletargeting:GetImportJobs');
        return this;
    }
    /**
     * Retrieve information about a specific journey.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods
     */
    toGetJourney() {
        this.to('mobiletargeting:GetJourney');
        return this;
    }
    /**
     * Retrieve information about a specific or the active version of an push notification template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods
     */
    toGetPushTemplate() {
        this.to('mobiletargeting:GetPushTemplate');
        return this;
    }
    /**
     * Retrieve information about an Amazon Pinpoint configuration for a recommender model.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#GetRecommenderConfiguration
     */
    toGetRecommenderConfiguration() {
        this.to('mobiletargeting:GetRecommenderConfiguration');
        return this;
    }
    /**
     * Retrieve information about all the recommender model configurations that are associated with an Amazon Pinpoint account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html#GetRecommenderConfigurations
     */
    toGetRecommenderConfigurations() {
        this.to('mobiletargeting:GetRecommenderConfigurations');
        return this;
    }
    /**
     * Retrieve information about a specific segment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment.html#rest-api-segment-methods
     */
    toGetSegment() {
        this.to('mobiletargeting:GetSegment');
        return this;
    }
    /**
     * Retrieve information about jobs that export endpoint definitions from segments to Amazon S3.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods
     */
    toGetSegmentExportJobs() {
        this.to('mobiletargeting:GetSegmentExportJobs');
        return this;
    }
    /**
     * Retrieve information about jobs that create segments by importing endpoint definitions from .
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-jobs.html#rest-api-import-jobs-methods
     */
    toGetSegmentImportJobs() {
        this.to('mobiletargeting:GetSegmentImportJobs');
        return this;
    }
    /**
     * Retrieve information about a specific segment version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment-version.html#rest-api-segment-version-methods
     */
    toGetSegmentVersion() {
        this.to('mobiletargeting:GetSegmentVersion');
        return this;
    }
    /**
     * Retrieve information about the current and prior versions of a segment.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment-versions.html#rest-api-segment-versions-methods
     */
    toGetSegmentVersions() {
        this.to('mobiletargeting:GetSegmentVersions');
        return this;
    }
    /**
     * Retrieve information about the segments for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segments.html#rest-api-segments-methods
     */
    toGetSegments() {
        this.to('mobiletargeting:GetSegments');
        return this;
    }
    /**
     * Obtain information about the SMS channel in an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-sms-channel.html#rest-api-sms-channel-methods
     */
    toGetSmsChannel() {
        this.to('mobiletargeting:GetSmsChannel');
        return this;
    }
    /**
     * Retrieve information about a specific or the active version of an sms message template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods
     */
    toGetSmsTemplate() {
        this.to('mobiletargeting:GetSmsTemplate');
        return this;
    }
    /**
     * Retrieve information about the endpoints that are associated with a user ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-user.html#rest-api-user-methods
     */
    toGetUserEndpoints() {
        this.to('mobiletargeting:GetUserEndpoints');
        return this;
    }
    /**
     * Obtain information about the Voice channel in an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-voice-channel.html#rest-api-voice-channel-methods
     */
    toGetVoiceChannel() {
        this.to('mobiletargeting:GetVoiceChannel');
        return this;
    }
    /**
     * Retrieve information about a specific or the active version of a voice message template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods
     */
    toGetVoiceTemplate() {
        this.to('mobiletargeting:GetVoiceTemplate');
        return this;
    }
    /**
     * Retrieve information about all journeys for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys.html#apps-application-id-journeys-http-methods
     */
    toListJourneys() {
        this.to('mobiletargeting:ListJourneys');
        return this;
    }
    /**
     * List tags for a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-tags.html#rest-api-tags-methods-get
     */
    toListTagsForResource() {
        this.to('mobiletargeting:ListTagsForResource');
        return this;
    }
    /**
     * Retrieve all versions about a specific template.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-template-type-versions.html#templates-template-name-template-type-versions-http-methods
     */
    toListTemplateVersions() {
        this.to('mobiletargeting:ListTemplateVersions');
        return this;
    }
    /**
     * Retrieve metadata about the queried templates.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates.html#templates-http-methods
     */
    toListTemplates() {
        this.to('mobiletargeting:ListTemplates');
        return this;
    }
    /**
     * Obtain metadata for a phone number, such as the number type (mobile, landline, or VoIP), location, and provider.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-phone-number-validate.html#rest-api-phone-number-validate-methods
     */
    toPhoneNumberValidate() {
        this.to('mobiletargeting:PhoneNumberValidate');
        return this;
    }
    /**
     * Create or update an event stream for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-event-stream.html#rest-api-event-stream-methods
     */
    toPutEventStream() {
        this.to('mobiletargeting:PutEventStream');
        return this;
    }
    /**
     * Create or update events for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-events.html#rest-api-events-methods
     */
    toPutEvents() {
        this.to('mobiletargeting:PutEvents');
        return this;
    }
    /**
     * Used to remove the attributes for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods
     */
    toRemoveAttributes() {
        this.to('mobiletargeting:RemoveAttributes');
        return this;
    }
    /**
     * Send an SMS message or push notification to specific endpoints.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-messages.html#rest-api-messages-methods
     */
    toSendMessages() {
        this.to('mobiletargeting:SendMessages');
        return this;
    }
    /**
     * Send an SMS message or push notification to all endpoints that are associated with a specific user ID.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-users-messages.html#rest-api-users-messages-methods
     */
    toSendUsersMessages() {
        this.to('mobiletargeting:SendUsersMessages');
        return this;
    }
    /**
     * Adds tags to a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-tags.html#rest-api-tags-methods-post
     */
    toTagResource() {
        this.to('mobiletargeting:TagResource');
        return this;
    }
    /**
     * Removes tags from a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-tags.html#rest-api-tags-methods-delete
     */
    toUntagResource() {
        this.to('mobiletargeting:UntagResource');
        return this;
    }
    /**
     * Update the Amazon Device Messaging (ADM) channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-adm-channel.html#rest-api-adm-channel-methods
     */
    toUpdateAdmChannel() {
        this.to('mobiletargeting:UpdateAdmChannel');
        return this;
    }
    /**
     * Update the Apple Push Notification service (APNs) channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-channel.html#rest-api-apns-channel-methods
     */
    toUpdateApnsChannel() {
        this.to('mobiletargeting:UpdateApnsChannel');
        return this;
    }
    /**
     * Update the Apple Push Notification service (APNs) sandbox channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-sandbox-channel.html#rest-api-apns-sandbox-channel-methods
     */
    toUpdateApnsSandboxChannel() {
        this.to('mobiletargeting:UpdateApnsSandboxChannel');
        return this;
    }
    /**
     * Update the Apple Push Notification service (APNs) VoIP channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-channel.html#rest-api-apns-voip-channel-methods
     */
    toUpdateApnsVoipChannel() {
        this.to('mobiletargeting:UpdateApnsVoipChannel');
        return this;
    }
    /**
     * Update the Apple Push Notification service (APNs) VoIP sandbox channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-sandbox-channel.html#rest-api-apns-voip-sandbox-channel-methods
     */
    toUpdateApnsVoipSandboxChannel() {
        this.to('mobiletargeting:UpdateApnsVoipSandboxChannel');
        return this;
    }
    /**
     * Update the default settings for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-settings.html#rest-api-settings-methods
     */
    toUpdateApplicationSettings() {
        this.to('mobiletargeting:UpdateApplicationSettings');
        return this;
    }
    /**
     * Update the Baidu channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-baidu-channel.html#rest-api-baidu-channel-methods
     */
    toUpdateBaiduChannel() {
        this.to('mobiletargeting:UpdateBaiduChannel');
        return this;
    }
    /**
     * Update a specific campaign.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign.html#rest-api-campaign-methods
     */
    toUpdateCampaign() {
        this.to('mobiletargeting:UpdateCampaign');
        return this;
    }
    /**
     * Update the email channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-email-channel.html#rest-api-email-channel-methods
     */
    toUpdateEmailChannel() {
        this.to('mobiletargeting:UpdateEmailChannel');
        return this;
    }
    /**
     * Update a specific email template under the same version or generate a new version.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods
     */
    toUpdateEmailTemplate() {
        this.to('mobiletargeting:UpdateEmailTemplate');
        return this;
    }
    /**
     * Create an endpoint or update the information for an endpoint.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoint.html#rest-api-endpoint-methods
     */
    toUpdateEndpoint() {
        this.to('mobiletargeting:UpdateEndpoint');
        return this;
    }
    /**
     * Create or update endpoints as a batch operation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoints.html#rest-api-endpoints-methods
     */
    toUpdateEndpointsBatch() {
        this.to('mobiletargeting:UpdateEndpointsBatch');
        return this;
    }
    /**
     * Update the Firebase Cloud Messaging (FCM) or Google Cloud Messaging (GCM) API key that allows to send push notifications to your Android app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-gcm-channel.html#rest-api-gcm-channel-methods
     */
    toUpdateGcmChannel() {
        this.to('mobiletargeting:UpdateGcmChannel');
        return this;
    }
    /**
     * Update a specific journey.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods
     */
    toUpdateJourney() {
        this.to('mobiletargeting:UpdateJourney');
        return this;
    }
    /**
     * Update a specific journey state.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id-state.html#apps-application-id-journeys-journey-id-state-http-methods
     */
    toUpdateJourneyState() {
        this.to('mobiletargeting:UpdateJourneyState');
        return this;
    }
    /**
     * Update a specific push notification template under the same version or generate a new version.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods
     */
    toUpdatePushTemplate() {
        this.to('mobiletargeting:UpdatePushTemplate');
        return this;
    }
    /**
     * Update an Amazon Pinpoint configuration for a recommender model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#UpdateRecommenderConfiguration
     */
    toUpdateRecommenderConfiguration() {
        this.to('mobiletargeting:UpdateRecommenderConfiguration');
        return this;
    }
    /**
     * Update a specific segment.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment.html#rest-api-segment-methods
     */
    toUpdateSegment() {
        this.to('mobiletargeting:UpdateSegment');
        return this;
    }
    /**
     * Update the SMS channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-sms-channel.html#rest-api-sms-channel-methods
     */
    toUpdateSmsChannel() {
        this.to('mobiletargeting:UpdateSmsChannel');
        return this;
    }
    /**
     * Update a specific sms message template under the same version or generate a new version.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods
     */
    toUpdateSmsTemplate() {
        this.to('mobiletargeting:UpdateSmsTemplate');
        return this;
    }
    /**
     * Upate the active version parameter of a specific template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-template-type-versions.html#templates-template-name-template-type-versions-http-methods
     */
    toUpdateTemplateActiveVersion() {
        this.to('mobiletargeting:UpdateTemplateActiveVersion');
        return this;
    }
    /**
     * Update the Voice channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-voice-channel.html#rest-api-voice-channel-methods
     */
    toUpdateVoiceChannel() {
        this.to('mobiletargeting:UpdateVoiceChannel');
        return this;
    }
    /**
     * Update a specific voice message template under the same version or generate a new version.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods
     */
    toUpdateVoiceTemplate() {
        this.to('mobiletargeting:UpdateVoiceTemplate');
        return this;
    }
    /**
     * Adds a resource of type apps to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/developerguide/gettingstarted.html#gettingstarted-addapp
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApps(appId, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type campaigns to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaigns.html
     *
     * @param appId - Identifier for the appId.
     * @param campaignId - Identifier for the campaignId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCampaigns(appId, campaignId, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}/campaigns/${CampaignId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${CampaignId}', campaignId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type journeys to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys.html
     *
     * @param appId - Identifier for the appId.
     * @param journeyId - Identifier for the journeyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJourneys(appId, journeyId, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}/journeys/${JourneyId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${JourneyId}', journeyId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type segments to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segments.html
     *
     * @param appId - Identifier for the appId.
     * @param segmentId - Identifier for the segmentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSegments(appId, segmentId, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}/segments/${SegmentId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${SegmentId}', segmentId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type templates to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates.html
     *
     * @param templateName - Identifier for the templateName.
     * @param channelType - Identifier for the channelType.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTemplates(templateName, channelType, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:templates/${TemplateName}/${ChannelType}';
        arn = arn.replace('${TemplateName}', templateName);
        arn = arn.replace('${ChannelType}', channelType);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type recommenders to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html
     *
     * @param recommenderId - Identifier for the recommenderId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRecommenders(recommenderId, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:recommenders/${RecommenderId}';
        arn = arn.replace('${RecommenderId}', recommenderId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mobiletargeting = Mobiletargeting;
//# sourceMappingURL=data:application/json;base64,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